/** @file main.cpp
* hlavni zdrojovy soubor programu
* @author Radek Hranicky
*/

/*
LIS Noise Cleaner - Tento nastroj slouzi k filtraci sumu vytvoreneho programem LDP
Copyright (C) 2012 Radek Hranicky

Tento program je svobodny software: muzete jej sirit a upravovat podle ustanoveni Obecne verejne licence GNU (GNU General Public License),
vydavane Free Software Foundation a to bud podle 3. verze teto Licence, nebo (podle vaseho uvazeni) kterekoli pozdejsi verze.

Tento program je rozsirovan v nadeji, ze bude uzitecny, avas BEZ JAKEKOLIV ZARUKY. Neposkytuji se ani odvozene zaruky PRODEJNOSTI
anebo VHODNOSTI PRO URCITY UCEL. Dalsi podrobnosti hledejte v Obecne verejne licenci GNU.

Kopii Obecne verejne licence GNU jste meli obdrzet spolu s timto programem. Pokud se tak nestalo, najdete ji zde: <http://www.gnu.org/licenses/>.
*/

#include <iostream>
#include <pcap.h>

#include "main.h"
#include "errors.h"
#include "config.h"
#include "packet.h"

using namespace std;
TConfig Config;

/**
 * Hlavni funkce programu
 * @param argc pocet zadanych parametru
 * @param argv vycet zadanych parametru
 * @return navratovy kod programu
 */
int main (int argc, char *argv[]) {
  initConfig(Config);
  struct pcap_pkthdr pcap_header;
  const u_char *packet;
  int errNum = OK;

  // zpracovani vstupnich parametru programu
  if (argc == 2 && string(argv[1]) == "-h") {
    printHelp();
    return EXIT_SUCCESS;
  }
  if (argc != 3) {
    printError(EPARAMS);
    printHelp();
    return EXIT_FAILURE;
  }
  Config.input_file = argv[1];  // ulozime nazev vstupniho souboru
  Config.output_file = argv[2]; // ulozime nazev vystupniho souboru

  // ===================================== FAZE 1 - cteni a analyza paketu ==================================
  Config.pcapHandler = pcap_open_offline(Config.input_file.c_str(), Config.pcapErrbuf); // otevreni vstupu
  if (Config.pcapHandler == NULL) {
    printError(EOPENIN);
    return EXIT_FAILURE;
  }
  while ((packet = pcap_next(Config.pcapHandler, &pcap_header)) != NULL) {              // nacteni paketu
    errNum = processPacket_phase1((u_char *)packet, Config);              // zpracovani paketu
    if (errNum != OK) {
      break;
    }
  }
  pcap_close(Config.pcapHandler);                                                       // zavreni vstupu

  // =========================== FAZE 2 - cteni a zapis paketu do vystupniho souboru =========================
  Config.pcapHandler = pcap_open_offline(Config.input_file.c_str(), Config.pcapErrbuf); // otevreni vstupu
  if (Config.pcapHandler == NULL) {
    printError(EOPENIN);
    return EXIT_FAILURE;
  }
  Config.pcapDump = pcap_dump_open(Config.pcapHandler, Config.output_file.c_str());     // otevreni vystupu
  if (Config.pcapDump == NULL) {
    printError(EOPENOUT);
    return EXIT_FAILURE;
  }
  while ((packet = pcap_next(Config.pcapHandler, &pcap_header)) != NULL) {              // nacteni paketu
    errNum = processPacket_phase2(&pcap_header, (u_char *)packet, Config);              // zpracovani paketu
    if (errNum != OK) {
      break;
    }
  }
  pcap_dump_close(Config.pcapDump);                                                     // zavreni vystupu

  pcap_close(Config.pcapHandler);                                                       // zavreni vstupu

  // =======================================================================================================

  cout << "====================================" << endl;
  cout << " Filtrovani bylo uspesne dokonceno! " << endl;
  cout << "====================================" << endl;
  cout << "- zpracovano paketu: " << Config.pktCount << endl;
  cout << "- z toho TCP segmentu: " << Config.tcpCount << endl;
  cout << "- z toho jednobytovych: " << Config.byteTcpCount << endl;
  cout << "------------------------------------" << endl;
  cout << "- odfiltrovano paketu se sumem: " << Config.pktCount - Config.pktWrittenCount << endl;
  cout << "- pocet paketu na vystupu: " << Config.pktWrittenCount << endl;
  cout << "====================================" << endl;

  if (errNum == OK) {
    return EXIT_SUCCESS; // vse OK
  } else {
    printError(errNum);
    return EXIT_FAILURE; // nastala chyba
  }
}
