/** @file translator.cpp
* zdrojovy soubor s funkcemi pro porovnavani a preklad prefixu
* @author Radek Hranicky
*/

/*
LIS Deception Proxy - Tento nastroj slouzi pro demonstraci utoku na system pro zakonne odposlechy
Copyright (C) 2012 Radek Hranicky

Tento program je svobodny software: muzete jej sirit a upravovat podle ustanoveni Obecne verejne licence GNU (GNU General Public License),
vydavane Free Software Foundation a to bud podle 3. verze teto Licence, nebo (podle vaseho uvazeni) kterekoli pozdejsi verze.

Tento program je rozsirovan v nadeji, ze bude uzitecny, avas BEZ JAKEKOLIV ZARUKY. Neposkytuji se ani odvozene zaruky PRODEJNOSTI
anebo VHODNOSTI PRO URCITY UCEL. Dalsi podrobnosti hledejte v Obecne verejne licenci GNU.

Kopii Obecne verejne licence GNU jste meli obdrzet spolu s timto programem. Pokud se tak nestalo, najdete ji zde: <http://www.gnu.org/licenses/>.
*/

#include <iostream>
#include <arpa/inet.h>
#include <math.h>

#include "translator.h"
#include "errors.h"

using namespace std;

/**
 * Funkce porovna shodu prefixu s danou IPv6 adresou
 * @param ipv6Addr IPv6 adresa
 * @param prefix IPv6 prefix
 * @param prefixLen delka prefixu (pocet bitu, ktere rozlisujeme)
 * @return shoda/neshoda
 */
bool comparePrefix(in6_addr ipv6Addr, in6_addr prefix, int prefixLen) {
  if (prefixLen > MAX_PREFIX_LEN) {
    prefixLen = MAX_PREFIX_LEN;
  } else if (prefixLen <= 0) {
    return true;
  }

  /*
    Protoze uvazujeme libovolnou delku prefixu, budeme
    porovnavat adresu s prefixem bit po bitu.
  */
  bool match = true;
  bool endSearch = false;
  int bitNum = 1; // cislo bitu (v ramci cele adresy)
  for (int octetNum = 1; octetNum <= 16; octetNum++) { // postupne prochazime oktety adresy
    for (int t=128; t>0; t=t/2) {                      // a bity v jednotlivych oktetech
      bool bit_a;
      bool bit_b;
      if (ipv6Addr.s6_addr[octetNum-1] & t) {
        bit_a = true;  // dany bit v adrese je 1
      } else {
        bit_a = false; // dany bit v adrese je 0
      }
      if (prefix.s6_addr[octetNum-1] & t) {
        bit_b = true;  // dany bit v prefixu je 1
      } else {
        bit_b = false; // dany bit v prefixu je 0
      }
      if (bit_a != bit_b) { // bity nesouhlasi
        match = false;      // => adresa se neshoduje s timto prefixem
        break;
      }
      bitNum++;
      if (bitNum > prefixLen) { // dosli jsme na konec delky prefixu
        endSearch = true;
        break;
      }
    }
    if (!match || endSearch) { // pokud jsme dosli na konec prefixu nebo nastala neshoda
      break;                   // => muzeme cyklus ukoncit
    }
  }
  return match;
}

/**
 * Funkce detekuje, zda program bezi na architekture
 * pouzivajici little-endian nebo big-endian usporadani.
 */
int detectEndianness () {
  int num = 1;
  if(*(char *)&num == 1)
  {
    return END_LITTLE;
  }
  else
  {
    return END_BIG;
  }
}


/**
 * Spojeni dvou oktetu do jednoho hextetu
 * v zavislosti na pouzite architekture.
 * (napr. z 0xfa a 0x12 vytvori 0xfa12)
 * @param octet_a 1. oktet
 * @param octet_a 2. oktet
 * @return hextetu
 */
uint16_t joinOctets(uint8_t octet_a, uint8_t octet_b) {
  int endi = detectEndianness();

  int16_t hextet;
  union {
      uint8_t bytes[2];
      uint16_t hext;
  } join;
  if (endi == END_LITTLE) { // little-endian
    join.bytes[0] = octet_b;
    join.bytes[1] = octet_a;
  } else {                 // big-endian
    join.bytes[0] = octet_a;
    join.bytes[1] = octet_b;
  }

  hextet = join.hext;
  return hextet;
}


/**
 * Rozdeleni hextetu na dva oktety
 * v zavislosti na pouzite architekture.
 * (napr. z 0xfa12 na 0xfa a 0x12)
 * @param hextet
 * @param 1. oktet (vrati funkce)
 * @param 2. oktet (vrati funkce)
 */
void splitOctets(uint16_t hextet, uint8_t& octet_a, uint8_t& octet_b) {
  int endi = detectEndianness();

  union {
      uint8_t bytes[2];
      uint16_t hext;
  } join;
  join.hext = hextet;
  if (endi == END_LITTLE) { // little-endian
    octet_b = join.bytes[0];
    octet_a = join.bytes[1];
  } else {                  // big-endian
    octet_a = join.bytes[0];
    octet_b = join.bytes[1];
  }

  return;
}


/**
 * Prepise nepotrebne bity v adrese nulami
 * (Pro jistotu, kdyby v konfiguracnim souboru delka prefixu
 *  "za lomitkem" neodpovidala uvedenemu prefixu)
 */
void zerosFill(in6_addr& ipv6Addr, int effectiveLength) {
  int bitNum = 1;
  for (int octetNum = 1; octetNum <= 16; octetNum++) {    // postupne prochazime oktety adresy
    int bitShift = 7;
    for (int t=128; t>0; t=t/2) {                         // a bity v jednotlivych oktetech
      if (bitNum > effectiveLength) {
        ipv6Addr.s6_addr[octetNum-1] &= ~(1 << bitShift); // vynulovani bitu
      }
      bitShift--;
      bitNum++;
    }
  }
  return;
}


/**
 * Prevod IPv6 adresy z oktetove reprezentaci na hextetovou
 * @param ipv6Addr adresa v oktetove reprezentaci
 * @return adresa v hextetove reprezentaci
 */
in6_hex_addr in6ToHexAddr(in6_addr& ipv6Addr) {
  in6_hex_addr hextet_form;
  int octetNum = 0;
  for (int hextetNum=0; hextetNum<=7; hextetNum++) {
    hextet_form.addr_hextet[hextetNum] = joinOctets(ipv6Addr.s6_addr[octetNum], ipv6Addr.s6_addr[octetNum+1]);
    octetNum += 2;
  }
  return hextet_form;
}


/**
 * Prevod IPv6 adresy z hextetove zpet na oktetovou (in6_addr) reprezentaci
 * @param ipv6Addr adresa v oktetove reprezentaci
 * @return adresa v hextetove reprezentaci
 */
in6_addr hexToIn6Addr(in6_hex_addr& hextet_form) {
  in6_addr octet_form;
  int octetNum = 0;
  for (int hextetNum=0; hextetNum<=7; hextetNum++) {
    splitOctets(hextet_form.addr_hextet[hextetNum], octet_form.s6_addr[octetNum], octet_form.s6_addr[octetNum+1]);
    octetNum += 2;
  }
  return octet_form;
}


/**
 * Vypocet korekce
 * @param prefixFrom zdrojovy prefix
 * @return prefixTo cilovy prefix
 * @return korekce
 */
uint16_t calculateAdjustment(in6_hex_addr& prefixFrom, in6_hex_addr& prefixTo) {
  /*
   "one's complement" aritmetika je resena potoci union{}
    Vysledek operace (scitani/odcitani) je ulozen do celych 32bitu
    a nasledne se vrchnich 16 a 16bitu secte a vypocita jednickovy
    doplnek souctu.
  */
  union {
    uint16_t sum16[2];
    uint32_t sum32;
  } inSum;
  union {
    uint16_t sum16[2];
    uint32_t sum32;
  } exSum;

  inSum.sum32 = 0;
  exSum.sum32 = 0;
  for (int hextetNum=0; hextetNum<=7; hextetNum++) {
    inSum.sum32 += prefixFrom.addr_hextet[hextetNum];
    exSum.sum32 += prefixTo.addr_hextet[hextetNum];
  }

  uint16_t inSum_final = inSum.sum16[0] + inSum.sum16[1];
  uint16_t exSum_final = exSum.sum16[0] + exSum.sum16[1];
  inSum_final = ~inSum_final;
  exSum_final = ~exSum_final;

  // Mame jednickove doplnky kontrolnich souctu obou prefixu.
  // Jejich rozdilem (v "one's complement" aritmetice) ziskame
  // pozadovanou korekci.
  union {
    uint16_t adj16[2];
    uint32_t adj32;
  } adj;
  adj.adj32 = 0;
  adj.adj32 = exSum_final - inSum_final;
  uint16_t adj_final = adj.adj16[0] + adj.adj16[1];

  return adj_final;
}

/**
 * Funkce overi, zda IID adresy neobsahuje same nuly
 * @param ipv6Addr IPv6 adresa
 * @return validni ano/ne
 */
bool validateInterfaceID(in6_addr ipv6Addr) {
  bool zero_iid = true;
  for (int octet = 8; octet <= 15; octet++) {
    if (ipv6Addr.s6_addr[octet] != 0x00) {
        zero_iid = false;
    }
  }
  if (zero_iid) {
    return false; // IID adresy neni v poradku
  } else {
    return true;  // IID adresy je v poradku
  }
}


/**
 * Funkce preklada prefix dane adresy na pozadovany prefix
 * a aplikuje korekci pro zachovani kontrolniho souctu.
 * @param ipv6Addr IPv6 adresa k prelozeni
 * @param ipv6AddrTranslated prelozena IPv6 adresa (funkce ji vytvori)
 * @param prefixFrom zdrojovy prefix
 * @param prefixFromLen delka zdrojoveho prefixu
 * @param prefixTo cilovy prefix
 * @param prefixToLen delka ciloveho prefixu
 * @return chybovy kod
 */
int translateAddress(in6_addr ipv6Addr, in6_addr& ipv6AddrTranslated,
                      in6_addr prefixFrom, int prefixFromLen,
                      in6_addr prefixTo,   int prefixToLen
                    ) {
  /*
    Funkce predpoklada, ze adresa k prelozeni odpovida zdrojovemu prefixu.
    V ramci redukce mnozstvi vypoctu funkce toto sama jiz neoveruje
    a predpoklada, ze k overeni doslo pred jejim volanim.
    (Pokud funkce neodpovida prefixu, muze vest preklad k nespravne korekci.)
  */

  int bitsToChange;  // pocet bitu, ktere bude preklad (bez korekce) menit.
  int effectiveBits; // 1) pocet bitu, ktere budou nahrazeny bity ciloveho prefixu
  int zeros;         // 2) pocet nul, ktere budou doplneny do delky delsiho z prefixu

  if (prefixFromLen > prefixToLen) {
    bitsToChange = prefixFromLen;
    effectiveBits = prefixToLen;
    zeros = bitsToChange - effectiveBits;
  } else {
    bitsToChange = prefixToLen;
    effectiveBits = prefixToLen;
    zeros = 0;
  }

  // "paranoidni" vyplneni zbytku prefixu nulami pro pripad, ze by delka prefixu
  // "za lomitkem" v konfiguracnim souboru neodpovidala uvedenemu prefixu.
  zerosFill(prefixFrom, prefixFromLen);
  zerosFill(prefixTo, prefixToLen);

  ipv6AddrTranslated = ipv6Addr;

  bool endOfTranslation = false;
  int bitNum = 1; // cislo bitu (v ramci cele adresy)
  for (int octetNum = 1; octetNum <= 16; octetNum++) { // postupne prochazime oktety adresy
    int bitShift = 7;
    for (int t=128; t>0; t=t/2) {                      // a bity v jednotlivych oktetech
      if (bitNum <= effectiveBits) {
        if (prefixTo.s6_addr[octetNum-1] & t) {
          // dany bit v prefixu je 1
          ipv6AddrTranslated.s6_addr[octetNum-1] |= (1 << bitShift);
        } else {
          // dany bit v prefixu je 0
          ipv6AddrTranslated.s6_addr[octetNum-1] &= ~(1 << bitShift);
        }
      } else {
        // bit nastavime na 0
        ipv6AddrTranslated.s6_addr[octetNum-1] &= ~(1 << bitShift);
      }

      bitShift--;
      bitNum++;
      if (bitNum > bitsToChange) {
        endOfTranslation = true; // dale jiz neni treba nic menit
        break;
      }
    }
    if (endOfTranslation) { // pokud jsme dosli na konec prefixu nebo nastala neshoda
      break;                   // => muzeme cyklus ukoncit
    }
  }

  /*
    Samotny prefix byl nahrazen. Nyni je treba vypocitat korekci
    a aplikovat ji na vhodne zvolene bity adresy.
  */

  // Pro ucely vypoctu korekce prevedeme adresu i prefixy do hextetove reprezentace
  in6_hex_addr hex_translatedAddr = in6ToHexAddr(ipv6AddrTranslated);
  in6_hex_addr hex_prefixFrom = in6ToHexAddr(prefixFrom);
  in6_hex_addr hex_prefixTo = in6ToHexAddr(prefixTo);

  // Vypocitame korekci
  uint16_t adjustment = calculateAdjustment(hex_prefixFrom, hex_prefixTo);

  // Urcime na kterem hextetu bude aplikovana korekce
  int hextetNum;
  int firstHextet;
  bool hextet_found = false;
  if (bitsToChange <= 48) { // pokud je prefix delky 48 nebo kratsi
    hextetNum = 3;  // bity 48..63
    hextet_found = true;
  } else { // delsi prefix
    // Podle delky prefixu urcime, pocinaje kterym hextetem zacneme
    // hledat hextet k aplikovani korekce.
    if (bitsToChange > 48 && bitsToChange <= 64) {
      firstHextet = 4;
    } else if (bitsToChange > 64 && bitsToChange <= 80) {
      firstHextet = 5;
    } else if (bitsToChange > 80 && bitsToChange <= 96) {
      firstHextet = 6;
    } else if (bitsToChange > 96 && bitsToChange <= 112) {
      firstHextet = 7;
    } else if (bitsToChange > 112) {
      firstHextet = -1; // neni mozna aplikovat korekci
    }
    if (firstHextet >= 4) {
      for (hextetNum=firstHextet; hextetNum<=7; hextetNum++) {
        if (hex_translatedAddr.addr_hextet[hextetNum] != 0xFFFF) {
          hextet_found = true;
          break;
        }
      }
    }
  }

  /*
    Muzeme na aplikovat korekci na prislusny hextet
    (v pripade nenalezeni jde o anycast adresu a nelze provest korekci)
  */
  if (hextet_found) {
    union {
      uint16_t bits16[2];
      uint32_t bits32;
    } adjustedBits;

    adjustedBits.bits32 = 0;
    adjustedBits.bits32 = hex_translatedAddr.addr_hextet[hextetNum] + adjustment;

    uint16_t adjustedBits_final = adjustedBits.bits16[0] + adjustedBits.bits16[1];
    hex_translatedAddr.addr_hextet[hextetNum] = adjustedBits_final;

    if (hex_translatedAddr.addr_hextet[hextetNum] == 0xFFFF) {
      hex_translatedAddr.addr_hextet[hextetNum] = 0x0000;
    }
  }
  ipv6AddrTranslated = hexToIn6Addr(hex_translatedAddr);

  return OK;
}
