/** @file packet.cpp
* zdrojovy soubor obsahujici implementaci tridy Packet
* @author Radek Hranicky
*/

/*
LIS Deception Proxy - Tento nastroj slouzi pro demonstraci utoku na system pro zakonne odposlechy
Copyright (C) 2012 Radek Hranicky

Tento program je svobodny software: muzete jej sirit a upravovat podle ustanoveni Obecne verejne licence GNU (GNU General Public License),
vydavane Free Software Foundation a to bud podle 3. verze teto Licence, nebo (podle vaseho uvazeni) kterekoli pozdejsi verze.

Tento program je rozsirovan v nadeji, ze bude uzitecny, avas BEZ JAKEKOLIV ZARUKY. Neposkytuji se ani odvozene zaruky PRODEJNOSTI
anebo VHODNOSTI PRO URCITY UCEL. Dalsi podrobnosti hledejte v Obecne verejne licenci GNU.

Kopii Obecne verejne licence GNU jste meli obdrzet spolu s timto programem. Pokud se tak nestalo, najdete ji zde: <http://www.gnu.org/licenses/>.
*/

#include <iostream>
#include <csignal>
#include <fstream>
#include <sstream>
#include <list>
#include <regex.h>
#include <libnet.h>
#include <pcap.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/if_ether.h>
#include <netinet/ip6.h>
#include <netinet/tcp.h>

#include "packet.h"
#include "tcp.h"

/**
 * Konstruktor tridy Packet
 * @param ukazatel na zacatek paketu
 * @param delka paketu
 * @param smer paketu
 */
Packet::Packet (uint8_t* packetPtr, unsigned int packetLen, TPacketDirection pktDirection, unsigned long int num) {
  ipv6_datagram = NULL;
  ipv6_header = NULL;

  icmpv6_header = NULL;

  tcp_header = NULL;
  tcp_data = NULL;

  ipv6_datagramLen = packetLen;
  direction = pktDirection;
  targetInt = "";
  capturedInt = "";
  pNum = num;

  try {
    ipv6_datagram = new uint8_t[ipv6_datagramLen];
    if (ipv6_datagram == NULL) {
      ipv6_datagramLen = 0;
      return;
    }
  } catch (std::bad_alloc &b) {
    ipv6_datagram = NULL;
    ipv6_datagramLen = 0;
    return;
  }

  // kontrola zda velikost zachyceneho datagramu skutecne odpovida
  if (IPV6_FIXED_HEADER_LEN + (unsigned int)(ntohs(((struct ip6_hdr *)packetPtr)->ip6_ctlun.ip6_un1.ip6_un1_plen)) != packetLen) {
    ipv6_datagram = NULL;
    ipv6_datagramLen = 0;
    return;
  }

  memcpy(ipv6_datagram, packetPtr, ipv6_datagramLen);

  ipv6_header = (struct ip6_hdr *)ipv6_datagram;
  ipv6_srcIP_untranslated = ipv6_header->ip6_src;
  ipv6_dstIP_untranslated = ipv6_header->ip6_dst;

  if (ipv6_header->ip6_ctlun.ip6_un1.ip6_un1_nxt == NEXT_HEADER_TCP) {
    tcp_header = (struct tcphdr *)(ipv6_datagram + IPV6_FIXED_HEADER_LEN);
    if (this->tcp_dataLen() > 0) {
      tcp_data = (uint8_t *)tcp_header + this->tcp_header_len();
    }
  } else if (ipv6_header->ip6_ctlun.ip6_un1.ip6_un1_nxt == ICMPV6_NEXT_HEADER) {
    icmpv6_header = (t_icmpv6 *)(ipv6_datagram + IPV6_FIXED_HEADER_LEN);
  }
}

/**
 * Oriznuti paketu na urcitou velikost
 */
void Packet::truncPacket(unsigned int newLen) {
  if (newLen < ipv6_datagramLen) {
    ipv6_datagramLen = newLen;
  }
}

/**
 * Ziskani cisla paketu
 */
unsigned long int Packet::get_pktNum() {
  return pNum;
}

/**
 * Zjisteni smeru paketu
 * @return smer paketu
 */
TPacketDirection Packet::packetDirection() {
  return direction;
}

/**
 * Nastaveni rozhrani, na kterm byl paket zachycen
 * @param nazev rozhrani
 */
void Packet::setCapturedInterface(string cptIf) {
  capturedInt = cptIf;
}

/**
 * Zjisteni rozhrani na kterem byl paket zachycen
 * @param nazev rozhrani
 */
string Packet::capturedInterface() {
  return capturedInt;
}

/**
 * Nastaveni ciloveho rozhrani, kam ma byt paket odeslan
 * @param nazev rozhrani
 */
void Packet::setTargetInterface(string targetIf) {
  targetInt = targetIf;
}

/**
 * Zjisteni ciloveho rozhrani, kam ma byt paket odeslan
 * @return nazev rozhrani
 */
string Packet::targetInterface() {
  return targetInt;
}

/**
 * Zjisteni delky IPv6 datagramu
 * @return delka IPv6 datagramu
 */
unsigned int Packet::ipv6_len() {
  return ipv6_datagramLen;
}

/**
 * Zjisteni zdrojove IPv6 adresy
 * @return zdrojova IPv6 adresa
 */
in6_addr Packet::ipv6_srcIP() {
  return ipv6_header->ip6_src;
}

/**
 * Zjisteni IPv6 adresy odesilatele
 * @return IPv6 adresa odesilatele
 */
in6_addr Packet::ipv6_sndIP() {
  if (direction == TO_RECEIVER) {
    return ipv6_header->ip6_src;
  } else {
    return ipv6_dstIP_untranslated;
  }
}

/**
 * Zjisteni IPv6 adresy prijemce
 * @return IPv6 adresa prijemce
 */
in6_addr Packet::ipv6_rcvIP() {
  if (direction == TO_RECEIVER) {
    return ipv6_header->ip6_dst;
  } else {
    return ipv6_header->ip6_src;
  }
}

/**
 * Nastaveni prelozene zdrojove IPv6 adresy
 * @param prelozena zdrojova IPv6 adresa
 */
void Packet::set_translated_ipv6_srcIP(in6_addr addr) {
  ipv6_srcIP_untranslated = ipv6_header->ip6_src;
  ipv6_header->ip6_src = addr;
}

/**
 * Nastaveni prelozene cilove IPv6 adresy
 * @param prelozena cilova IPv6 adresa
 */
void Packet::set_translated_ipv6_dstIP(in6_addr addr) {
  ipv6_dstIP_untranslated = ipv6_header->ip6_dst;
  ipv6_header->ip6_dst = addr;
}

/**
 * zjisteni cilove IPv6 adresy
 * @return cilova IPv6 adresa
 */
in6_addr Packet::ipv6_dstIP() {
  return ipv6_header->ip6_dst;
}

/**
 * Zjisteni hodnoty Hop Limit z IPv6 datagramu
 * @return hodnota Hop Limit z IPv6 datagramu
 */
uint8_t Packet::ipv6_hlim() {
  if (ipv6_header == NULL) {
    return 0;
  } else {
    return ipv6_header->ip6_ctlun.ip6_un1.ip6_un1_hlim;
  }
}

/**
 * Nastaveni hodnoty Hop Limit IPv6 datagramu
 * @param hodnota Hop Limit IPv6 datagramu
 */
void Packet::set_ipv6_hlim(uint8_t newHlim) {
  if (ipv6_header != NULL) {
    ipv6_header->ip6_ctlun.ip6_un1.ip6_un1_hlim = newHlim;\
  }
}

/**
 * Zjisteni hodnoty Payload Length z IPv6 datagramu
 * @return hodnota Payload Length z IPv6 datagramu
 */
uint16_t Packet::ipv6_plen() {
  if (ipv6_header == NULL) {
    return 0;
  }
  return ntohs(ipv6_header->ip6_ctlun.ip6_un1.ip6_un1_plen);
}

/**
 * Zjisteni hodnoty Payload Length z IPv6 datagramu
 * @return hodnota Payload Length z IPv6 datagramu
 */
void Packet::set_ipv6_plen(uint16_t newPlen) {
  if (ipv6_header != NULL) {
    ipv6_header->ip6_ctlun.ip6_un1.ip6_un1_plen = htons(newPlen);
  }
}

/**
 * Jde o TCP segment?
 * @return ano, ci ne
 */
bool Packet::is_tcp() {
  if (tcp_header == NULL) {
    return false;
  } else {
    return true;
  }
}

/**
 * Zjisteni delky TCP hlavicky
 * @return delky TCP hlavicky
 */
unsigned int Packet::tcp_header_len() {
  if (tcp_header == NULL) {
    return 0;
  } else {
    return ((unsigned int)(tcp_header->doff * 4)); // delka je ve 4bytovych slovech
  }
}


/**
 * Vypocet TCP checksum
 */
void Packet::tcp_calculate_checksum() {
  if (ipv6_header != NULL && tcp_header != NULL) {
    tcpChecksum_for_ipv6(ipv6_header, tcp_header, ipv6_plen());
  }
}


/**
 * Zjisteni portu odesilatele TCP segmentu
 * @return port odesilatele TCP segmentu
 */
uint16_t Packet::tcp_sndPort() {
  if (tcp_header == NULL) {
    return 0;
  }
  if (direction == TO_RECEIVER) {
    return ntohs(tcp_header->source);
  } else {
    return ntohs(tcp_header->dest);
  }
}

/**
 * Zjisteni portu prijemce TCP segmentu
 * @return port prijemce TCP segmentu
 */
uint16_t Packet::tcp_rcvPort() {
  if (tcp_header == NULL) {
    return 0;
  }
  if (direction == TO_RECEIVER) {
    return ntohs(tcp_header->dest);
  } else {
    return ntohs(tcp_header->source);
  }
}

/**
 * Je v TCP hlavicce nastaven priznak SYN?
 * @return ano, ci ne
 */
bool Packet::tcp_syn() {
  if (tcp_header == NULL) {
    return false;
  }
  if (tcp_header->syn == 1) {
    return true;
  } else {
    return false;
  }
}

/**
 * Je v TCP hlavicce nastaven priznak ACK?
 * @return ano, ci ne
 */
bool Packet::tcp_ack() {
  if (tcp_header == NULL) {
    return false;
  }
  if (tcp_header->ack == 1) {
    return true;
  } else {
    return false;
  }
}

/**
 * Je v TCP hlavicce nastaven priznak FIN?
 * @return ano, ci ne
 */
bool Packet::tcp_fin() {
  if (tcp_header == NULL) {
    return false;
  }
  if (tcp_header->fin == 1) {
    return true;
  } else {
    return false;
  }
}

/**
 * Je v TCP hlavicce nastaven priznak RST?
 * @return ano, ci ne
 */
bool Packet::tcp_rst() {
  if (tcp_header == NULL) {
    return false;
  }
  if (tcp_header->rst == 1) {
    return true;
  } else {
    return false;
  }
}

/**
 * Zjisteni delky dat TCP segmentu
 * @return delka dat TCP segmentu
 */
unsigned int Packet::tcp_dataLen() {
  if (ipv6_header == NULL || tcp_header == NULL) {
    return 0;
  } else {
    unsigned char *tcp_data;
    unsigned int tcp_dataLen;
    unsigned short int data_offset = tcp_header->doff;
    tcp_data = ((unsigned char *)tcp_header) + data_offset * 4;  // offset je ve 4B blocich
    tcp_dataLen = (ipv6_datagram + ipv6_datagramLen) - tcp_data;
    return (unsigned int)tcp_dataLen;
  }
}

/**
 * Zjisteni Acknowledgement Number z TCP hlavicky
 * @return Acknowledgement Number z TCP hlavicky
 */
uint32_t Packet::tcp_ackSeq() {
  if (tcp_header == NULL) {
    return 0;
  } else {
    return ntohl(tcp_header->ack_seq);
  }
}

/**
 * Zjisteni Sequence Number z TCP hlavicky
 * @return Sequence Number z TCP hlavicky
 */
uint32_t Packet::tcp_seq() {
  if (tcp_header == NULL) {
    return 0;
  } else {
    return ntohl(tcp_header->seq);
  }
}

/**
 * Nastaveni Sequence Number v TCP hlavicce
 * @param Sequence Number v TCP hlavicce
 */
void Packet::set_tcp_seq(uint32_t newSeq) {
  if (tcp_header != NULL) {
    tcp_header->seq = htonl(newSeq);
  }
}

/**
 * Jde o ICMPv6 zpravu?
 * @return ano, ci ne
 */
bool Packet::is_icmpv6() {
  if (icmpv6_header == NULL) {
    return false;
  } else {
    return true;
  }
}

/**
 * Jde o odpoved na ICMPv6 echo?
 * @return ano, ci ne
 */
bool Packet::is_icmpv6_echoReply() {
  if (icmpv6_header == NULL) {
    return false;
  }
  if (icmpv6_header->code == ICMPV6_ECHOREPLY_CODE
    && icmpv6_header->type == ICMPV6_ECHOREPLY_TYPE)
  {
    return true;
  } else {
    return false;
  }
}

/**
 * Zjisteni ID odpovedi na ICMPv6 echo
 * @return ID odpovedi na ICMPv6 echo
 */
uint16_t Packet::icmpv6_echoReply_id() {
  if (!is_icmpv6_echoReply()) {
    return 0;
  }
  t_Ip6Icmp6 *message;
  message = (t_Ip6Icmp6 *)ipv6_datagram;
  return ntohs(message->icmpv6_body.s_echo.identifier);
}

/**
 * Zjisteni hodnoty Sequence odpovedi na ICMPv6 echo
 * @return hodnota Sequence odpovedi na ICMPv6 echo
 */
uint16_t Packet::icmpv6_echoReply_seq() {
  if (!is_icmpv6_echoReply()) {
    return 0;
  }
  t_Ip6Icmp6 *message;
  message = (t_Ip6Icmp6 *)ipv6_datagram;
  return ntohs(message->icmpv6_body.s_echo.sequence);
}


/**
 * Ziskani ukazatele na IPv6 datagram
 */
uint8_t * Packet::ipv6Ptr() {
  return ipv6_datagram;
}

/**
 * Ziskani ukazatele na IPv6 hlavicku
 */
ip6_hdr * Packet::ipv6_headerPtr() {
  return ipv6_header;
}

/**
 * Ziskani ukazatele na TCP hlavicku
 */
tcphdr * Packet::tcp_headerPtr() {
  return tcp_header;
}

/**
 * Ziskani ukazatele na TCP data
 */
uint8_t * Packet::tcpDataPtr() {
  return tcp_data;
}


/**
 * Destruktor tridy Packet
 */
Packet::~Packet() {
  if (ipv6_datagram != NULL) {
    try {
      delete [] ipv6_datagram;
    } catch (...) {
      return;
    }
  }
}
