#include <stdlib.h>
#include <stdio.h>

#include <cv.h>
#include <libxml/parser.h>
#include <libxml/tree.h>

#include "BUT_Caretaker_Trajectory.h"

#include "common.h"
#include "lukas.h"
#include "viterbi_pivo.h"


void createModel(xmlNode* node);

#define SCENARIO_CONFIG "scenarioConfig"
#define SCENARIO BAD_CAST( "scenario" )
#define DEFAULT_SCENARIO BAD_CAST( "default_scenario" )
#define MODEL "model"
#define MODEL_FILENAME BAD_CAST("file")
#define MODEL_THRESHOLD BAD_CAST("threshold")
#define MODEL_NAME BAD_CAST("name")

#define ALLOC_STEP 128

/** 
  * this function parses attributes of model from xml
  * @param node "scenarioConfig" node
  * @param model 
  */

void createModelFromNode(xmlNode* node, struct trajectoryModel* model) {

  char *filename = (char *) xmlGetProp( node, MODEL_FILENAME );
  model->threshold = atof((char *) xmlGetProp( node,  MODEL_THRESHOLD ));
  model->name = (char *) xmlGetProp( node,  MODEL_NAME );

  int state, i;
  FILE *fp = fopen (filename, "r");
  
  if (!fp) {
    fprintf(stderr, "model \"%s\" does not exists", filename);
    exit(1);
  }
  fscanf(fp, "%d", &model->N); 
  model->N += 2; 
  fscanf(fp, "%d", &model->P); 
  
  /* alokace po model */
  
  model->means = Alloc_2_Float (model->N,model->P);
  model->vars  = Alloc_2_Float (model->N,model->P);
  model->a     = Alloc_2_Float (model->N,model->N);

  /* means a vars pro 1 a posledni stav budou nuly, do zbytku nacist  */
  for (i = 0; i < model->P; i++) {
    model->means[0][i] = model->means[model->N-1][i] = model->vars[0][i] = model->vars[model->N-1][i] = 0.0; 
  }
	  
  //printf("zeros ok"); fflush(stdout);
	    
  for (state = 1; state <= model->N-2; state++) {
    for (i = 0; i < model->P; i++) {
      fscanf(fp, "%f", &model->means[state][i]);
    }
  }
  
			      
  //printf("means ok"); fflush(stdout);
				
  for (state = 1; state <= model->N-2; state++) {
    for (i=0; i<model->P; i++) {
      fscanf(fp, "%f", &model->vars[state][i]);
    }
  }

  //printf("vars ok"); fflush(stdout);
  /* transition probas */
  for (state = 0; state < model->N; state++) {
    for (i = 0; i < model->N; i++) {
      fscanf(fp, "%f", &model->a[state][i]);
    }
  }

  fclose(fp);
  
}


/**
  * setups models from xml file for selected scenario
  * @param root node with configuration
  * @param scenario name
  */

struct TrajectoryClassifier* trajectoryClassifierCreate (xmlNode *root, char* scenario) {

  struct TrajectoryClassifier *c = malloc (sizeof (struct TrajectoryClassifier ));
  c->trajectory = NULL;
  c->trajectoryLen = 0;
  c->trajectoryAlloc = 0;
  c->P = 1;

  if (!scenario) {
    scenario = (char *) BAD_CAST(xmlGetProp( root, DEFAULT_SCENARIO) );
  }

  int modelCount = 0;
  int i = 0;

  for (xmlNode* curNode = root->children; curNode; curNode = curNode->next) {
     if ((strcmp(SCENARIO_CONFIG, (char *) BAD_CAST( curNode->name )) == 0)    // in Node scenarioConfig
      && (strcmp(scenario, (char *) xmlGetProp( curNode, SCENARIO )) == 0)) {  // for selected scenario

       // count how many models we have
       for (xmlNode* modelNode = curNode->children; modelNode; modelNode = modelNode->next) {
	 if ( strcmp((char*)BAD_CAST( modelNode->name), MODEL) == 0) {
	   modelCount++;
	 }
       }
       
       // memmory allocation
       c->models = (struct trajectoryModel*) malloc ( modelCount * sizeof (struct trajectoryModel));
       for (xmlNode* modelNode = curNode->children; modelNode; modelNode = modelNode->next) {
	 if ( strcmp((char*)BAD_CAST( modelNode->name), MODEL) == 0) {
	   createModelFromNode(modelNode, &c->models[i++]);
           c->P = c->models[i-1].P;
	 }
       }

     }
  }
  
  c->modelCount = modelCount;

  c->likelyhoods = (float*) malloc ( sizeof (float) * c->modelCount);

#if 0
  for (int i = 0; i < c->modelCount; i++) {
    printf("model: %d -> %f\n", i, c->models[i].threshold);
    for (int j = 0; j < c->models[i].N; j++) {
      for (int k = 0; k < c->models[i].N; k++) {
        printf("%f ",c->models[i].a[j][k]);
      }
      printf("\n");
    }
  }
#endif
  
  
  return c;
}

/**
  * creates a copy of struct TrajectoryClassifier
  */ 

struct TrajectoryClassifier* trajectoryClassifierClone (struct TrajectoryClassifier* c1) { 
  struct TrajectoryClassifier* c2 = malloc(sizeof(struct TrajectoryClassifier));
  c2->P               = c1->P;
  c2->modelCount      = c1->modelCount;
  for (int i = 0; i < c2->modelCount; i++) {
    c2->models[i].N = c1->models[i].N;
    c2->models[i].P = c1->models[i].P;

    c2->models[i].means = Alloc_2_Float (c2->models[i].N,c2->models[i].P);
    c2->models[i].vars  = Alloc_2_Float (c2->models[i].N,c2->models[i].P);
    c2->models[i].a     = Alloc_2_Float (c2->models[i].N,c2->models[i].N); 
    
    for (int j = 0; j < c2->models[i].N; j++) {
      for (int k = 0; k < c2->models[i].P; k++) {
        c2->models[i].means[j][k] = c1->models[i].means[j][k];
        c2->models[i].vars[j][k]  = c1->models[i].vars[j][k];
      }
      for (int k = 0; k < c2->models[i].N; k++) {
        c2->models[i].a[j][k]  = c1->models[i].a[j][k];
      }
    }
	    
    c2->models[i].threshold = c1->models[i].threshold;
    strcpy(c2->models[i].name, c1->models[i].name);
  }
  
  c2->trajectoryLen   = c1->trajectoryLen;
  c2->trajectoryAlloc = c1->trajectoryAlloc;

  c2->trajectory = Alloc_2_Float(c1->trajectoryAlloc, c1->P); 
  for (int i = 0; i < c1->trajectoryLen; i++) {
    for (int j = 0; j < c1->P; j++) {
      c2->trajectory[i][j] = c1->trajectory[i][j];
    }
  }

  c2->likelyhoods = (float*) malloc ( sizeof (float) * c2->modelCount);
  for (int i = 0; i < c2->modelCount; i++) {
    c1->likelyhoods[i] = c2->likelyhoods[i];
  }
  return c2; 
}

/**
  *
  */

void trajectoryClassifierRelease(struct TrajectoryClassifier** c) {

  struct trajectoryModel* m;

  for (int i = 0; i < (*c)->modelCount; i++) {
    m = &((*c)->models[i]);
    Free_2_Float(m->means, m->N);
    Free_2_Float(m->vars, m->N);
    Free_2_Float(m->a, m->N);
    
  }
  free((*c)->models);
  Free_2_Float((*c)->trajectory, (*c)->trajectoryAlloc	);
  free(*c);
}

/**
  * computes logaritmical viterbi likelyhood over all models and trajectory
  * @param c TrajctoryClassifier object (aka model definition and whole trajectory)
  * @param trajectoryClass placeholder for winner class id (return -1 when not pass thru threshold)
  * @param likelyhoods placeholder for all likelyhoods 
  */

void trajectoryEvaluate( const struct TrajectoryClassifier *c, int* trajectoryClass, float* likelyhood) {

  for (int i = 0; i < c->modelCount; i++) {
    c->likelyhoods[i] =  viterbi (c->models[i].N,c->models[i].P,c->models[i].means,c->models[i].vars,c->models[i].a, c->trajectory, c->trajectoryLen);
  }
  // which model fits best with trajectory
  float max = -1e20f;
  for (int i = 0; i < c->modelCount; i++) {
    if (c->likelyhoods[i] > max)  {
      max = c->likelyhoods[i];
      *trajectoryClass = i;
      *likelyhood = c->likelyhoods[i];
      // threshold check
      if (c->likelyhoods[i] < c->models[i].threshold) {
        *trajectoryClass = -1;
      }
    }
    
  }
}



/**
  * computes logaritmical viterbi likelyhood over whole trajectory
  * @param c TrajctoryClassifier object (aka model definition and whole trajectory)
  * @param trajectoryClass placeholder for winner class id
  * @param likelyhoods placeholder for all likelyhoods (return -1 when not pass thru threshold)
  */

void trajectoryProcess( struct TrajectoryClassifier* c, const float* trajectory, const int trajectoryLen, int* trajectoryClass, float* likelyhood) {
  c->trajectoryLen = 0;

  float* coords;
  for (int i = 0 ; i < trajectoryLen; i++) {
    coords = (float*)trajectory + i * c->P ;
    trajectoryAddCoords(c, coords);
  }
  trajectoryEvaluate(c, trajectoryClass, likelyhood);

}

/**
  * adding new coordinates
  * (with dynamic memory reallocation)
  */

void trajectoryAddCoords( struct TrajectoryClassifier *c, const float* coords ) {

  // reallocation of memory
  while (c->trajectoryLen >= c->trajectoryAlloc) { 
    int freeSize = c->trajectoryAlloc;
    c->trajectoryAlloc += ALLOC_STEP;
    float **t = Alloc_2_Float(c->trajectoryAlloc, c->P); 
    
    for (int i = 0; i < c->trajectoryLen; i++) {
      for (int j = 0; j < c->P; j++) {
        t[i][j] = c->trajectory[i][j];
      }
    }
    
    if (c->trajectory != NULL) {
      Free_2_Float(t, freeSize);
    }
    c->trajectory = t;
    
  }

  // adding of new coord
  for (int i = 0; i < c->P; i++) {
    c->trajectory[c->trajectoryLen][i] = coords[i];
  }
  c->trajectoryLen++;

}

/**
  * removes trajectory from memory
  * (without deallocation)
  */

void trajectoryClear( struct TrajectoryClassifier* c ) {
  c->trajectoryLen = 0;
}

void printTrajectoryClassifierInfo(const struct TrajectoryClassifier* c) {
  printf("P = %d\n", c->P);
  printf("modelCount = %d\n", c->modelCount);

  for (int i = 0; i < c->modelCount; i++) {
    printf("model %d:\nN = %d\nP = %d\nthreshold = %f\nname = %s\ntrajectoryLen = %d\ntrajectoryAlloc = %d\n", i, c->models[i].N, c->models[i].P, c->models[i].threshold, c->models[i].name, c->trajectoryLen, c->trajectoryAlloc);

    printf("means\n");
    
    for (int j = 0; j < c->models[i].N; j++) {
      for (int k = 0; k < c->models[i].P; k++) {
        printf("%f ",c->models[i].means[j][k]);
      }
      printf("\n");
    }

    printf("vars\n");
    for (int j = 0; j < c->models[i].N; j++) {
      for (int k = 0; k < c->models[i].P; k++) {
        printf("%f ",c->models[i].vars[j][k]);
      }
      printf("\n");
    }
    printf("a\n");
    for (int j = 0; j < c->models[i].N; j++) {
      for (int k = 0; k < c->models[i].N; k++) {
        printf("%f ", c->models[i].a[j][k]);
      }
      printf("\n");      
    }	    
    
  }

  printf("trajectory:\n");
  for (int i = 0; i < c->trajectoryLen; i++) {
    for (int j = 0; j < c->P; j++) {
      printf("%f ",c->trajectory[i][j]);
    }
    printf("\n");
  }

  printf("likelyhoods:\n");
  for (int i = 0; i < c->modelCount; i++) {
    printf("%f\n",c->likelyhoods[i]);
  }
  
}

/**
  * @param in array of [x, y]
  * @param out array of [x, y, dx, dy]
  * @param inSize count of rows
  */

void convert2Dto4DWithAlloc (float * in, float **out, const int rowCount) {
  float *oa = (float*) malloc ( rowCount * 4 * sizeof(float) );
  
  if (rowCount < 1) {
    return;
  }
  
  float lastX, lastY, dx, dy;
  lastX = in[0];
  lastY = in[1];
  
  for (int i = 0; i < rowCount; i++) {
    dx = lastX - in[2*i+0];
    dy = lastY - in[2*i+1];
    oa[4*i+0] = lastX = in[2*i+0];
    oa[4*i+1] = lastY = in[2*i+1];
    oa[4*i+2] = dx;
    oa[4*i+3] = dy;
    
  }
  *out = oa;

}
