#!/usr/bin/env python3

"""Classifier.py: Abstract class for external data resolver"""
__author__      = "Radek Hranicky, Jan Polisensky"

from abc import abstractmethod

class Resolver():

    def __init__(self):
        """
        ! Constructor of the Classifier abstract class
        """
        self.name = ""
        self.timeout = 10
        self.logging = False

    def getName(self):
        """
        ! Returns the name of the classifier
        @return classifier name
        """
        return self.name

    @abstractmethod
    def resolve(self, domain_name, ip_list, timeout):
        """
        ! Resolves the query for external data
        @param domain_name Name of the domain
        @param ip_list List of related IP addreses
        @param maximum number of seconds to wait for the result
        @return Returns the resolved data

        Example output:
        {
            "resolver_name": "ssl"        # Name of the resolver
            "success": True,              # True or False
            "error_description": '',      # e.g. 'unable to contact remote server'
            "created": datetime.datetime  # Timestamp of the data sample creation
            "data": {
            # ============== RESOLVER-SPECIFIC STRUCTURE ============== #
               "issuer":"O=Google Trust Services LLC",
               "end_date":datetime.datetime(2022, 10, 31, 8, 17, 42),
               "start_date":datetime.datetime(2022, 8, 8, 8, 17, 43)
            # ========================================================= #
            }
        }
        """
        pass
    
    def log(self, message):
        """
        ! Logs the message to the console
        @param message Message to be logged
        """
        if self.logging:
            print("Resolver: " + self.name + ": " + message)
        else:
            print("Logging is disabled")
