#!/usr/bin/env python3

"""Config.py: Configuration object for DomainRadar app"""
__author__      = "Radek Hranicky"

import logging
import configparser
import sys

class Config:
    def __init__(self, filename):
        """
        ! Constructor of the app Config class
        @param filename Name of the config INI file to load
        """
        self.api_url = "";
        self.api_version = "0.0"
        self.sec_token = "";
        self.force_ipv4 = False

        self.parser = configparser.ConfigParser()
        self.loadConfig(filename)

    def loadConfig(self, filename):
        """
        ! Loads the configuration from the given INI file
        @param filename Name of the config INI file to load
        """
        try:
            self.parser.read(filename)
        except IOError:
            logging.critical('Unable to read config file!')
            sys.exit(1)

        try:
            general = self.parser['GENERAL']
        except KeyError:
            logging.critical('GENERAL section is missing in config!')
            sys.exit(1)

        try:
            self.api_url = general['API_URL']
            if self.api_url[-1] == '/':
                self.api_url = self.api_url[:len(self.api_url) - 1]

        except KeyError:
            logging.critical('URL of the QRadar API (API_URL) is not specified!')
            sys.exit(1)

        try:
            self.api_version = str(general['API_VERSION'])
        except KeyError:
            logging.critical('Version of the QRadar API (API_VERSION) is not specified!')
            sys.exit(1)

        try:
            self.sec_token = general['SEC_TOKEN']
        except KeyError:
            logging.critical('QRadar App token (SEC_TOKEN) is not specified!')
            sys.exit(1)

        try:
            self.force_ipv4 = bool(general['FORCE_IPV4'])
        except KeyError:
            pass


    def get_api_url(self):
        """
        ! Returns the URL of the QRadar REST API
        @return URL (string)
        """
        return self.api_url

    def get_api_version(self):
        """
        ! Returns the used version QRadar REST API
        @return Version (string)
        """
        return self.api_version

    def get_sec_token(self):
        """
        ! Return the SEC authtentication token for the QRadar API
        @return SEC token (string)
        """
        return self.sec_token

    def get_force_ipv4(self):
        """
        ! Check whether the app should force IPv4 or not
        @return True if IPv4 should be forced, False otherwise
        """
        return self.force_ipv4
