//
//  SupportProcesses.cpp
//  ModelAQUAS-gen2
//
//  Created by Martin Hruby on 11/09/2019.
//  Copyright © 2019 Martin Hruby. All rights reserved.
//

#include "SupportProcesses.h"

// ----------------------------------------------------------------------
//
Sampler::Sampler(const SamplerCFG &cfg) : AQSimElement(SimElementPrio::logging)
{
    //
    _CFG = cfg;
    
    //
    if (_CFG.logCSVFilename.empty() == false) {
        //
        _logCSVFile = fopen(_CFG.logCSVFilename.c_str(), "wt");
    }
}


// ----------------------------------------------------------------------
//
Sampler::~Sampler()
{
    //
    if (_logCSVFile != nullptr) {
        //
        fclose(_logCSVFile);
        
        //
        _logCSVFile = nullptr;
    }
}

// ----------------------------------------------------------------------
//
SIMEL_ptr   Sampler::ActivateInSim(const SIM_ptr &simulator,
                                   const SamplerCFG &cfg)
{
    //
    auto _model = std::make_shared<Sampler>(cfg);
    
    //
    ActivateModel(simulator, _model);
    
    //
    return _model;
}

// ----------------------------------------------------------------------
//
SimResponse Sampler::behavior(const SIMEL_ptr &selfPTR)
{
    //
    doSample(*partOf->simContext());
    
    //
    activationTime = activationTime.plusSeconds(_CFG.timeStep);
    
    //
    return SimResponse::reactivate;
}


// ----------------------------------------------------------------------
// CSV output, format is as follows:
// - model time MM
// - model time SS
// MAP part:
// - SNP infusion [working unit]
// - current MAP
// - current MAP with noise
// NMT part:
// - Rocuronium infusion
// - ROC in plasma concentration
// - absolute amount of ROC in patient's body
// - ROC model, 3 compartments
// - 4 TOF readings, TOF1, TOF2, ....
// BP/NTG part:
// - NTG concentration in blood plasma
void    Sampler::doSample(const AQSimContext &state)
{
    //
    std::stringstream _str;

    // ------------------------------------------------------------------
    //
    _str << (int) (state._modelTime() / 60) << ",";
    _str << (int) (state._modelTime() % 60) << ",";
    _str << (int) (state._modelTime()) << ",";
    
    // ------------------------------------------------------------------
    // Physiological outputs
    // - MAP pure
    // - MAP plus noise
    // - TOF0
    _str << state.MAP(false) << ",";
    _str << state.MAP(true) << ",";
    _str << state._TOF[0] << ",";
    
    // ------------------------------------------------------------------
    // Infusion & bolus
    _str << state.infusion(Drugs::SNP) << ",";
    _str << state.infusion(Drugs::NTG) << ",";
    _str << state.infusion(Drugs::Rocuronium) << ",";
    _str << state.bolus(Drugs::Rocuronium) << ",";
    
    // ------------------------------------------------------------------
    // additional
    _str << state._rocuroniumCp_mg_ml << ",";
    
    // ------------------------------------------------------------------
    //
    if (_logCSVFile != nullptr) {
        //
        fprintf(_logCSVFile, "%s\n", _str.str().c_str());
    }
}
