//
//  SimEngine.hpp
//  ModelAQUAS-gen2
//
//  Created by Martin Hruby on 26/08/2019.
//  Copyright © 2019 Martin Hruby. All rights reserved.
//

#ifndef SimEngine_hpp
#define SimEngine_hpp

#include <random>

// ----------------------------------------------------------------------
//
#include "BasicDefs.h"
#include "Infusion.h"
#include "PatientInfo.h"
#include "GeneralInterface.h"

// ----------------------------------------------------------------------
//
__FORWARD(Patient, PAT_ptr, PAT_wptr)
__FORWARD(AQSimulator, SIM_ptr, SIM_wptr)
__FORWARD(AQSimElement, SIMEL_ptr, SIMEL_wptr)
__FORWARD(AQSimContext, SIMC_ptr, SIMC_wptr)

// ----------------------------------------------------------------------
// Temporaty context of the simulation.
// ----------------------------------------------------------------------
// Contains:
// 1) static part describing the patient (weight, sensitivity, ...)
// 2) the model time - [s]
// 3) currently infused amounts of drugs (in their defined units)
// 4) current levels of drugs in the bloodstream
// ---
// 4a) Rocuronium - as sum of all compartments
// 4b) Sodium Nitropruside
// 4c) Nitroglycerin
struct AQSimContext {
    // ------------------------------------------------------------------
    // STATIC:
    // referring the patient (static part of the context)
    PAT_ptr     patient;
    
    // ------------------------------------------------------------------
    // DYNAMIC:
    // model time - values valid at this particular moment
    AQTime      _modelTime;
    
    // ------------------------------------------------------------------
    // Infusion/Bolus counters
    // Infusion is supposed to be in [ml/h]
    // Bolus is supposed to be in [ml]
    // Particular models MUST convert this information to its internal
    // working units.
    std::array<double, (int) Drugs::__last> _infusion;
    std::array<double, (int) Drugs::__last> _bolus;
    
    // ------------------------------------------------------------------
    // General state variables
    // ------------------------------------------------------------------
    // BP/MAP [mmHg]
    double      _deltaMAP = 0;
    double      _MAPNoise = 0;
    
    // ------------------------------------------------------------------
    // NMT/TOF
    TOFRecord   _TOF;
    
    // ------------------------------------------------------------------
    // internal state values (for partial models)
    // ------------------------------------------------------------------
    // 1) NMT::Rocuronium
    double              _rocuroniumTotal = 0;
    double              _rocuroniumAmountCentralCompartment = 0;
    double              _rocuroniumCp_mg_ml = 0;
    CompartmentState    __rocComps;
    
    // ------------------------------------------------------------------
    // NTG
    double      _NTG_Concentration = 0;
    
    // ------------------------------------------------------------------
    //
    AQSimContext() {
        //
        _infusion.fill(0);
        _bolus.fill(0);
        __rocComps.fill(0);
    }
    
    // ------------------------------------------------------------------
    //
    static  SIMC_ptr    EMPTY() {
        //
        return std::make_shared<AQSimContext>();
    }
    
    // ------------------------------------------------------------------
    //
    static  SIMC_ptr    CLONE(const SIMC_ptr &from) {
        //
        return std::make_shared<AQSimContext>(*from);
    }
    
    // ------------------------------------------------------------------
    // Infusiion setter/getter. Primitive operations assuming
    // correct inputs.
    void        _addInfusion(Drugs drg, double valueInWUnits);
    void        _addBolus(Drugs drg, double valueInWUnits);
    
    // ------------------------------------------------------------------
    // get amounts in units of infusion [ml/h] or bolus [ml]
    double      infusion(Drugs drg) const { return _infusion[(int) drg]; }
    double      bolus(Drugs drg) const { return _bolus[(int) drg]; }
    
    // ------------------------------------------------------------------
    //
    double      MAP(bool addingNoise = false) const;
    TOFRecord   TOF() const { return _TOF; }
};


// ----------------------------------------------------------------------
//
typedef std::vector<SIMC_ptr> SIMC_pvec;

// ----------------------------------------------------------------------
//
enum class SimResponse {
    ok, error,
    reactivate
};

// ----------------------------------------------------------------------
// Simulation Element - abstract class for dynamic elements of the
// simulation (events, processes)
class AQSimElement {
protected:
    //
    AQTime          activationTime;
    SimElementPrio  prio = SimElementPrio::noPrio;
    
    //
    SIM_wptr        partOf = nullptr;
    
    //
public:
    //
    AQSimElement(SimElementPrio ewPrio) {
        //
        prio = ewPrio;
    }
    
    // ------------------------------------------------------------------
    //
    static  SIMEL_ptr   ActivateModel(const SIM_ptr &simulator,
                                      const SIMEL_ptr &model);
    
    // ------------------------------------------------------------------
    //
    const AQTime    at() const { return activationTime; }
    const SIMC_ptr  &SC() const;
    
    // ------------------------------------------------------------------
    //
    virtual SimResponse behavior(const SIMEL_ptr &selfPTR) {
        return SimResponse::ok;
    }
    
    // ------------------------------------------------------------------
    //
    friend class    AQSimulator;
};


// ----------------------------------------------------------------------
//
typedef std::vector<SIMEL_ptr> SIMEL_pvec;


// ----------------------------------------------------------------------
//
class AQSimulator {
    // ------------------------------------------------------------------
    //
    SIMC_ptr        _simContext;
    SIMC_pvec       _simcHistory;
    PatModelConfig  _CFG;
    
    // ------------------------------------------------------------------
    //
    void            __moveTo(const AQTime &tnow);
    
    //
    std::random_device __rndDevice;  //Will be used to obtain a seed for the random number engine
    std::mt19937       __rndGen; //Standard mersenne_twister_engine seeded with rd()
    
    
    // ------------------------------------------------------------------
    //
    SIMEL_pvec      _elements;
public:
    // ------------------------------------------------------------------
    //
    AQSimulator(const SIMC_ptr &startingSimc,
                const PatModelConfig &cfg) : __rndGen(__rndDevice())
    {
        //
        _simContext = startingSimc;
        _CFG = cfg;
    }
    
    // ------------------------------------------------------------------
    //
    static SIM_ptr      CREATE(const SIMC_ptr &startingSimc,
                               const PatModelConfig &cfg = PatModelConfig())
    {
        //
        return std::make_shared<AQSimulator>(startingSimc, cfg);
    }
    
    // ------------------------------------------------------------------
    //
    const SIMC_ptr      &simContext() const { return _simContext; }
    int                 modelTimeS() const { return simContext()->_modelTime(); }
    SIMC_ptr            simcHistory(const AQTime &tmAt) const;
    
    // ------------------------------------------------------------------
    //
    bool                addElement(const SIMEL_ptr &anElement);
    
    // ------------------------------------------------------------------
    // will iterate until reaching given time
    bool                stepsTo(const AQTime &tm);
    
    // ------------------------------------------------------------------
    //
    void                printUserOutputHeader() const;
    void                printUserOutput() const;
    
    // ------------------------------------------------------------------
    //
    std::mt19937       &rndDevice() { return __rndGen; }
};

#endif /* SimEngine_hpp */
