//
//  PatientInfo.hpp
//  ModelAQUAS-gen2
//
//  Created by Martin Hruby on 26/08/2019.
//  Copyright © 2019 Martin Hruby. All rights reserved.
//

#ifndef PatientInfo_hpp
#define PatientInfo_hpp

//
#include "BasicDefs.h"

// ----------------------------------------------------------------------
//
__FORWARD(Patient, PAT_ptr, PAT_wptr)

//
class Infusion;

// ----------------------------------------------------------------------
// Classes of patient's sensitivity to SNP
enum class SNPSensitivity {
    //
    low, high, normal, userdef
};

// ----------------------------------------------------------------------
// ... if relevant
enum class PatientSex {
    //
    male, female, notRelevant
};

// ----------------------------------------------------------------------
// Coefficients to the BP/SNP Patient Model
struct BPSNPBasicModelCoeffs {
    // time constants:
    // ------------------------------------------------------------------
    // body constant/metabolization
    // Slate: 20..60s, causing total clearance in 120->300s
    int     Tau = 50;
    
    // ------------------------------------------------------------------
    // initial transport delay,
    // Slate: 35-70s
    int     Ti = 35;
    
    // ------------------------------------------------------------------
    // recirculation delay
    // Slate: 30-75s
    int     Tc = 45;
    
    //
    int     T1() const { return Ti; }
    int     T2() const { return Ti + Tc;}
    
    // ------------------------------------------------------------------
    // recirculation constant
    // Slate & other literature: 0.2 -> 0.5, generally <0,1>
    double  alpha = 0.2;
    // patients SNP sensitivity
    // Slate:
    // - low sensitivity - 0.25
    // - normal - 1
    // - high - 8 (upto 9)
    // ----------------------------
    // the unit is mmHg/(ml/h), meaning:
    // also expressed in negative numbers,
    // K==-1 means decrease of 1 mmHg per unit 1 ml/h infusion
    // example: K=-1, infusion=20 ml/h
    // ==> potential decrease is -20mmHg
    // however, together with the recirculation constant "alpha"
    // the steady state SNP effect is K*(1+alpha) per 1 mL/h of infusion
    double  K = 1;
    
    //
    BPSNPBasicModelCoeffs() {
        //
    }
    
    //
    BPSNPBasicModelCoeffs(double eK, int eTau, int eTi, int eTc,
                          double eAlpha)
    {
        //
        K = eK; alpha = eAlpha;
        //
        Ti = eTi; Tc = eTc; Tau = eTau;
    }
};


// ----------------------------------------------------------------------
// Coefficients for NTG model
// https://sepia.unil.ch/pharmacology/index.php?id=86
struct BPNTGModelCoeffs {
    // ------------------------------------------------------------------
    // Volume of distribution - Vd.
    // Definition : "Fluid volume that would be required to contain
    // the amount of drug present in the body at the same concentration
    // as in the plasma."
    // ------------------------------------------------------------------
    // Vd is specific/individual for every patient.
    // Example: Vd=10L
    // Concentration of NTG in blood plasma C = A / Vd
    // We need 20 ug of NTG in order to achieve C=2ug/L concentration
    // for this particular patient.
    double          Vd = 10; // [L]
    
    // ------------------------------------------------------------------
    // Clearance - CL. [ amount/time ]
    // Clearance 1 L/min means 1L (of Vd) plasma cleared from the drug.
    // i.e. the body can remove the drug from 1L of Vd per minute.
    double          CL = 1; // [L/min]
    
    // ------------------------------------------------------------------
    // Elimination Rate Constant.
    // CL = Kel * Vd [min^{-1}]
    double          Kel() const {
        //
        return CL / Vd;
    }
    
    // ------------------------------------------------------------------
    // Elimination Rate Constant.
    // CL = Kel * Vd [s^{-1}]
    double          Kel_s() const {
        //
        return (CL / Vd) / 60.0;
    }
    
    // ------------------------------------------------------------------
    // t_{1/2} - time to achive half concentration
    // Definition : "Time it takes for the plasma concentration or the amount
    // of drug in the body to be reduced by 50%."
    // t12 = ln(2)/Kel [min]
    double          t12_min() const {
        //
        return 0.693 / Kel();
    }
    
    // ------------------------------------------------------------------
    // Patient's sensitivity to NTG expressed in [ug/L*%], i.e.
    // percents of MAP0 decrese per 1 ug/L of NTG concentration
    // Example:
    // MAP0 = 150 mmHg
    // C = 3 ug/L
    // sensitivity = 1.5
    // -> X = 3 * 1.5 [%]
    // -> deltaMAP = MAP/100 * X
    // -> MAP = MAP0 - deltaMAP
    double          sensitivity = 1.5;
};


// ----------------------------------------------------------------------
// Hill function for computing effect of drugs
struct HillFunction {
    // maximum effect (e.g. 100%)
    double          Emax = 100.0;
    
    // !!! user specific units
    double          EC50 = 0;
    
    // exponent
    double          gamma = 0;
    
    //
    HillFunction(double _emax, double _ec50, double _gamma) {
        //
        Emax = _emax;
        EC50 = _ec50;
        gamma = _gamma;
    }
    
    //
    double          operator()(double input) const {
        //
        auto _inputs = pow(input, gamma);
        auto _e50 = pow(EC50, gamma);
        
        //
        auto _semi = Emax * (_inputs / (_e50 + _inputs));
        
        //
        return std::min(Emax, _semi);
    }
};

// ----------------------------------------------------------------------
// Setting NMT/Rocuronium parameters
struct NTGRocuroniumCoeffs {
    // Volume of central compartment [ml/kg]
    double          Vc = 38;
    
    //
    double          VcPatient_ml(double weight) const {
        //
        return Vc * weight;
    }
    
    //
    HillFunction    effect;
    
    //
    NTGRocuroniumCoeffs() : effect(100.0, 0.823/1000.0, 4.79) {
        //
    }
};

// ----------------------------------------------------------------------
// This information (should) remains constant in duration of the whole
// simulation.
struct Patient {
    // ------------------------------------------------------------------
    // Global Information
    // ------------------------------------------------------------------
    PatientSex      sex = PatientSex::notRelevant;
    
    // ------------------------------------------------------------------
    // Objective input parameter
    // [kg]
    double          weight = 100.0;
    
    // ------------------------------------------------------------------
    // Amount of blood [l], either set manually or computed using the weight
    double          volumeOfBlood = 6.0;
    double          estimateBloodVolume() const;
    
    // ------------------------------------------------------------------
    // Initial (measured) patients MAP [mmHg]
    double          MAP0 = 120;
    
    // ------------------------------------------------------------------
    // SNP/MAP
    SNPSensitivity          SNPsensitivity = SNPSensitivity::normal;
    BPSNPBasicModelCoeffs   SNPuserDefined;
    BPNTGModelCoeffs        NTGuserDefined;
    NTGRocuroniumCoeffs     NMTRocuronium;
    
    // ------------------------------------------------------------------
    //
    Patient() { }
    
    // ------------------------------------------------------------------
    //
    static PAT_ptr  CREATE() {
        //
        return std::make_shared<Patient>();
    }
    
    // ------------------------------------------------------------------
    // Infusions
    Infusion        initialNMTBolus(Drugs nmtDrug) const;
};


// ----------------------------------------------------------------------
// Train of Four (TOF)
struct TOFRecord {
    //
    typedef std::array<int, 4> _TOFPrimitive;
    
    //
    _TOFPrimitive   values;
    
    //
    TOFRecord() {
        //
        values.fill(0);
    }
    
    //
    const int & operator [](int idx) const { return values[idx]; }
    
    //
    void    set(int idx, int val) { values[idx] = val; }
    
    //
    TOFRecord(int t1, int t2, int t3, int t4) {
        //
        values[0] = t1; values[1] = t2;
        values[2] = t2; values[3] = t3;
    }
};

#endif /* PatientInfo_hpp */
