//
//  InfusionProcess.cpp
//  ModelAQUAS-gen2
//
//  Created by Martin Hruby on 26/08/2019.
//  Copyright © 2019 Martin Hruby. All rights reserved.
//

//
#include "InfusionProcess.h"


// ----------------------------------------------------------------------
// Bolus construction
InfusionProcess::InfusionProcess(const AQTime &at,
                                 const Infusion &infu) : _infusion(infu),  AQSimElement(SimElementPrio::infusionBolus)
{
    // ...
    AQException::ASSERT(infu.isBolus(),
                        AQExceptionCodes::wrongUnits,
                        "Infusion process; bolus injection expected");
    
    // first activation and process priority
    activationTime = at;
    prio = SimElementPrio::infusionBolus;
}

// ----------------------------------------------------------------------
// Pump configuration
InfusionProcess::InfusionProcess(const Infusion &infu,
                                 const AQTimeRange &tr) : _infusion(infu),  AQSimElement(SimElementPrio::infusionFlow)
{
    // ...
    AQException::ASSERT(infu.isFlow(),
                        AQExceptionCodes::wrongUnits,
                        "Infusion process; flow/infusion injection expected");
    
    // starting at...
    _rangeOfTime = tr;
    activationTime = _rangeOfTime.from;
    prio = SimElementPrio::infusionFlow;
}

// ----------------------------------------------------------------------
// user API, entering an infusion process to sim, FLOW
SIMEL_ptr   InfusionProcess::ActivateInSim(const SIM_ptr &simulator,
                                           const Infusion &infu,
                                           const AQTimeRange &tr)
{
    //
    auto _model = std::make_shared<InfusionProcess>(infu, tr);
    
    //
    ActivateModel(simulator, _model);
    
    //
    return _model;
}

// ----------------------------------------------------------------------
// user API, entering an infusion process to sim, BOLUS
SIMEL_ptr   InfusionProcess::ActivateInSim(const SIM_ptr &simulator, 
                                           const AQTime &at,
                                           const Infusion &infu)
{
    //
    //auto _model = std::make_shared<InfusionProcess>(at, infu);
    
    //
    auto _bolInfu = Infusion::FLOW(infu.primitiveAmount()*2, infu.drug());
    auto _range = AQTimeRange(at, at.plusSeconds(30));
    auto _model = std::make_shared<InfusionProcess>(_bolInfu, _range);
    
    //
    ActivateModel(simulator, _model);
    
    //
    return _model;
}

// ----------------------------------------------------------------------
// Some drugs are applied in two waves:
// 1) first initial bolus
// 2) then a continuous infusion (flow)
// ----------------------------------------------------------------------
// This procedure construct pre-defined starting boluses
SIMEL_ptr   InfusionProcess::INITBolus(const SIM_ptr &simulator, Drugs drg)
{
    // ------------------------------------------------------------------
    //
    Infusion _anInf = Infusion::EMPTY();
    
    // ------------------------------------------------------------------
    // ... for predefined drugs
    switch (drg) {
        case Drugs::Rocuronium:
            // knowing the particular patient...
            _anInf = simulator->simContext()->patient->initialNMTBolus(drg);
            break;
            
        default:
            break;
    }
    
    // ------------------------------------------------------------------
    //
    if (_anInf.primitiveAmount() <= 0) {
        //
        return nullptr;
    }
    
    // ------------------------------------------------------------------
    //
    auto _model = std::make_shared<InfusionProcess>(AQTime::T0(), _anInf);
    
    //
    if (simulator->addElement(_model) == false) {
        //
        return nullptr;
    }
    
    //
    return _model;
}


// ----------------------------------------------------------------------
// Infusion: PUMP or BOLUS
// ----------------------------------------------------------------------
SimResponse InfusionProcess::behavior(const SIMEL_ptr &selfPTR)
{
    // ------------------------------------------------------------------
    // write access to the current sim state
    auto &__simc = partOf->simContext();
    
    // ------------------------------------------------------------------
    // Pump regime
    if (_infusion.isFlow() == true) {
        // adding the amount to the bloodstream
        __simc->_addInfusion(_infusion.drug(), _infusion.primitiveAmount());
        
        // next reactivation at...
        activationTime = __simc->_modelTime.plusSeconds(1);
        
        //
        if (activationTime <= _rangeOfTime.to) {
            //
            return SimResponse::reactivate;
        }
    } else {
        // --------------------------------------------------------------
        // Bolus type of infusion. The designated amount is
        // directly added to the infusion counter
        // --------------------------------------------------------------
        __simc->_addBolus(_infusion.drug(), _infusion.primitiveAmount());
    }
    
    //
    return SimResponse::ok;
}
