//
//  Infusion.hpp
//  ModelAQUAS-gen2
//
//  Created by Martin Hruby on 26/08/2019.
//  Copyright © 2019 Martin Hruby. All rights reserved.
//

#ifndef Infusion_hpp
#define Infusion_hpp

//
#include "BasicDefs.h"
#include "PatientInfo.h"

// ----------------------------------------------------------------------
// units of amount
enum class AMU {
    //
    mg, ug, ng
};

// ----------------------------------------------------------------------
// ->
double  percentLim(double percDecr);
double  percentOfLim(double baseValue, double percDecr);


// ----------------------------------------------------------------------
// Dilution: mg/ml
// ----------------------------------------------------------------------
struct Dilution {
    // miligrams of drug essence in 1ml of solution
    double  mgsInMl = 1;
    
    // ------------------------------------------------------------------
    // [mg]
    Dilution(double amount, double inVolume) {
        //
        mgsInMl = amount / inVolume;
    }
    
    // ------------------------------------------------------------------
    //
    static  Dilution    UNIT() {
        //
        return Dilution(1, 1);
    }
    
    // ------------------------------------------------------------------
    //
    static  Dilution    DEF(Drugs drg);
    
    // ------------------------------------------------------------------
    // output amount of drug [mg] in case of:
    // 1) bolus [ml] -> [mg]
    // 2) infusion [ml/h] -> [mg]
    double              drugAmountFromBolus(double mlsOfBolus,
                                            AMU outputUnit = AMU::mg) const;
    double              drugAmountFromInfusion(double mlsPerHour,
                                               int duringSeconds,
                                               AMU outputUnit = AMU::mg) const;
    
    // ------------------------------------------------------------------
    //
    double              solutionAmountToAchive(double absAmountOfDrugMG) const;

    // ------------------------------------------------------------------
    // ...
    double  miligrams(double miliLiters = 1) const {
        //
        return mgsInMl * miliLiters;
    }
};


// ----------------------------------------------------------------------
// Expressing infusion is essential for correct working of the simulation.
// One should differentiate:
// ----------------------------------------------------------------------
// All infusion is in [ml]. Infusion process MUST convert the infusion
// unit to working unit of a particular drug ([ml], [ml/hour])
// ----------------------------------------------------------------------
class Infusion {
protected:
    // ------------------------------------------------------------------
    // ...
    enum class  Type { bolus, flow };
    
    // ------------------------------------------------------------------
    // AMount specifi
    double      _value = 0;
    Type        _type = Type::bolus;
    Drugs       _drug;
    
    // ------------------------------------------------------------------
    //
    static AQTime   TIMESTEP() { return AQTime(1); }
    
    // ------------------------------------------------------------------
    //
    Infusion(Type tp, double val, const Drugs drg) {
        //
        _value = val; _type = tp;
        _drug = drg;
    }
    
public:
    // ------------------------------------------------------------------
    //
    static  Infusion    EMPTY() { return BOLUS(0, Drugs::SNP); }
    
    // ------------------------------------------------------------------
    //
    bool                isBolus() const { return _type == Type::bolus; }
    bool                isFlow() const { return _type == Type::flow; }
    double              primitiveAmount() const { return _value; }
    Drugs               drug() const { return _drug; }
    
    // ------------------------------------------------------------------
    // val [ml] of diluted drug
    static  Infusion    BOLUS(double val, Drugs drg) {
        //
        return Infusion(Type::bolus, val, drg);
    }
    
    // ------------------------------------------------------------------
    // val [ml/h] of infusion
    static  Infusion    FLOW(double val, Drugs drg) {
        //
        return Infusion(Type::flow, val, drg);
    }
};

#endif /* Infusion_hpp */
