//
//  BasicDefs.hpp
//  ModelAQUAS-gen2
//
//  Created by Martin Hruby on 26/08/2019.
//  Copyright © 2019 Martin Hruby. All rights reserved.
//

#ifndef BasicDefs_hpp
#define BasicDefs_hpp

#include <stdio.h>

#include <vector>
#include <memory>
#include <algorithm>
#include <array>
#include <string>
#include <sstream>

#include <math.h>

// ----------------------------------------------------------------------
//
#define __FORWARD(CLASSN,PTRNAME,WPTRNAME) class CLASSN; typedef std::shared_ptr<CLASSN> PTRNAME; typedef CLASSN *WPTRNAME;


//
typedef std::string rstring;

//
typedef std::array<double, 4> CompartmentState;

// ----------------------------------------------------------------------
// Exceptions handling in the simulator
enum class AQExceptionCodes {
    undefTimeMoment,
    allWentOK,
    eventNotRegisteredInSimulator,
    emptyModel,
    unitsNotDefined,
    wrongUnits,
    bolusInputNotSupported
};


// ----------------------------------------------------------------------
//
struct AQException {
    // ------------------------------------------------------------------
    //
    AQExceptionCodes code = AQExceptionCodes::allWentOK;
    rstring          comment;
    
    // ------------------------------------------------------------------
    //
    AQException(AQExceptionCodes cd, const rstring &txt) {
        //
        code = cd; comment = txt;
    }
    
    // ------------------------------------------------------------------
    //
    static  void    ASSERT(bool value,
                           AQExceptionCodes cd,
                           const rstring &txt = "");
    
    // ------------------------------------------------------------------
    //
    void            print() const;
};

// ----------------------------------------------------------------------
// structure for model time
class AQTime {
public:
    // ------------------------------------------------------------------
    //
    typedef int Type;
    
protected:
    // model time in [s]
    Type    modelTime = 0;
    
public:
    // ------------------------------------------------------------------
    //
    AQTime(Type st0 = 0);
    AQTime(const AQTime &from, Type st0, bool mustBeValid);
    
    // ------------------------------------------------------------------
    // initial real time T0, for imports/exports
    static  AQTime  T0() { return AQTime(0); }
    static  AQTime  TINF() { return AQTime(1000000000); }
    
    // ------------------------------------------------------------------
    //
    AQTime  plusSeconds(Type secs) const {
        //
        return AQTime(*this, modelTime + secs, true);
    }
    
    // ------------------------------------------------------------------
    //
    AQTime  minusSeconds(Type secs) const {
        //
        return AQTime(*this, modelTime - secs, false);;
    }
    
    // ------------------------------------------------------------------
    //
    int     operator () () const { return modelTime; }
    bool    valid() const { return modelTime >= 0; }
    
    // ------------------------------------------------------------------
    //
    bool    operator < (const AQTime &other) const {
        //
        return modelTime < other.modelTime;
    }
    
    // ------------------------------------------------------------------
    //
    bool    operator <= (const AQTime &other) const {
        //
        return modelTime <= other.modelTime;
    }
    
    // ------------------------------------------------------------------
    //
    bool    operator == (const AQTime &other) const {
        //
        return modelTime == other.modelTime;
    }
    
    // ------------------------------------------------------------------
    //
    bool    operator != (const AQTime &other) const {
        //
        return (modelTime == other.modelTime) == false;
    }
    
    // ------------------------------------------------------------------
    //
    bool    operator > (const AQTime &other) const {
        //
        return modelTime > other.modelTime;
    }
    
    // ------------------------------------------------------------------
    //
    bool    operator >= (const AQTime &other) const {
        //
        return modelTime >= other.modelTime;
    }
};


// ----------------------------------------------------------------------
//
struct AQTimeRange {
    // ------------------------------------------------------------------
    //
    AQTime  from = AQTime::T0();
    AQTime  to = AQTime::TINF();
    
    // ------------------------------------------------------------------
    //
    AQTimeRange(const AQTime &_f = AQTime::T0(),
                const AQTime &_t = AQTime::TINF()) {
        //
        from = _f; to = _t;
    }
    
    // ------------------------------------------------------------------
    //
    static  AQTimeRange TO(const AQTime &t) {
        //
        return AQTimeRange(AQTime::T0(), t);
    }
};

// ----------------------------------------------------------------------
//
enum class Drugs : int {
    //
    SNP = 0, Rocuronium = 1, NTG = 2,
    
    //
    __last = 3
};
    

// ----------------------------------------------------------------------
// Priorities for correct ordering in the simulation kernel
enum class SimElementPrio : int {
    // inputs...
    infusionBolus = 10,
    infusionFlow = 9,
    noises = 8,
    
    // models
    models = 7,
    
    // outputs
    logging = 2, noPrio = 0
};
    

#endif /* BasicDefs_hpp */
