//
//  BP-Models.cpp
//  ModelAQUAS-gen2
//
//  Created by Martin Hruby on 09/09/2019.
//  Copyright © 2019 Martin Hruby. All rights reserved.
//

#include "BP-SNP-Model.h"

// ----------------------------------------------------------------------
// ...
// ----------------------------------------------------------------------
BPSNPBasicModel::BPSNPBasicModel(const BPSNPBasicModelCoeffs &cfs) : AQSimElement(SimElementPrio::models)
{
    //
    _COEFFs = cfs;
}

// ----------------------------------------------------------------------
//
SIMEL_ptr   BPSNPBasicModel::BP_SNP(const Patient &forPatient)
{
    // format: K, Tau, Ti, Tc, Alpha
    static BPSNPBasicModelCoeffs _lowSensi(0.25, 50, 35, 40, 0.4);
    static BPSNPBasicModelCoeffs _normalSensi(1, 50, 35, 40, 0.4);
    static BPSNPBasicModelCoeffs _highSensi(8, 50, 35, 40, 0.4);
    
    //
    BPSNPBasicModelCoeffs _coefs = _normalSensi;
    
    //
    switch (forPatient.SNPsensitivity) {
        case SNPSensitivity::low:
            //
            _coefs = _lowSensi;
            break;
            
        case SNPSensitivity::normal:
            _coefs = _normalSensi;
            break;
            
        case SNPSensitivity::high:
            _coefs = _highSensi;
            break;
            
        case SNPSensitivity::userdef:
            _coefs = forPatient.SNPuserDefined;
            break;
            
        default:
            break;
    }
    
    //
    return std::make_shared<BPSNPBasicModel>(_coefs);
}

// ----------------------------------------------------------------------
//
SIMEL_ptr   BPSNPBasicModel::ActivateInSim(const SIM_ptr &simulator)
{
    //
    auto _model = BPSNPBasicModel::BP_SNP(*simulator->simContext()->patient);
    
    //
    return ActivateModel(simulator, _model);
}

// ----------------------------------------------------------------------
// BP/SNP process, activated in 1s steps
// ----------------------------------------------------------------------
// Mathematical background:
// dMAP'(t) = 1/Tau * dMAP(t) + K/Tau * INFUSION(t - Ti) + K*Alpha/Tau * INFUSION(t - Ti - Tc)
// MAP(t) = MAP_0 - dMAP(t)
// ----------------------------------------------------------------------
// Discretization (1s step)
// dMAP(t+1) = 1/Tau*dMAP(t) + ...
// generaly it is some A*dMAP + B0 * INFUSION(t-Ti) + B1 * INFUSION(t-Ti-Tc)
// INFUSION inputs are delayed -Ti and -(Ti + Tc)
// A-Element causes dMAP lowering (MAP gets back towards MAP0)
// B0-Element - main effect of SNP, delayed (-Ti)
// B0 = K / Tau, "magic constants" from the previous version of the model
// were (K * 15 [s]) / Tau, A = 1*15/Tau.
// B1 = B0 * Alpha (recirculation constant)
// ----------------------------------------------------------------------
// Total effect of SNP in lowering MAP is (in steady state): K * (1+Alpha)
// e.g. K=1, infusion 20 mL/h, alpha=0.4 => 1.4*20 mmHg
SimResponse BPSNPBasicModel::behavior(const SIMEL_ptr &selfPTR)
{
    // ------------------------------------------------------------------
    // comput
    double _deltaMAPnow = 0;
    
    // ------------------------------------------------------------------
    //
    auto   _Tnow = this->activationTime;
    
    // ------------------------------------------------------------------
    //
    auto _hist1 = partOf->simcHistory(_Tnow.minusSeconds(1));
    auto _hist2 = partOf->simcHistory(_Tnow.minusSeconds(_COEFFs.T1()));
    auto _hist3 = partOf->simcHistory(_Tnow.minusSeconds(_COEFFs.T2()));
    
    // ------------------------------------------------------------------
    //
    if (_hist1 != nullptr) {
        //
        _deltaMAPnow += (1 - (1.0/_COEFFs.Tau)) * _hist1->_deltaMAP;
    }
    
    //
    if (_hist2 != nullptr) {
        // [ml/h]
        auto __infu2 = _hist2->infusion(Drugs::SNP);
        
        //
        _deltaMAPnow += (_COEFFs.K / _COEFFs.Tau) * __infu2;
    }
    
    //
    if (_hist3 != nullptr) {
        // [ml/h]
        auto __infu3 = _hist3->infusion(Drugs::SNP);
        
        //
        _deltaMAPnow += ((_COEFFs.K * _COEFFs.alpha) / _COEFFs.Tau) * __infu3;
    }
    
    // ------------------------------------------------------------------
    //
    partOf->simContext()->_deltaMAP = _deltaMAPnow;
    
    // ------------------------------------------------------------------
    // next reactivation of the process/event
    // ------------------------------------------------------------------
    // 1s
    activationTime = activationTime.plusSeconds(1);
    
    //
    return SimResponse::reactivate;
}
