import time

import networkx as nx
import numpy as np
from skimage.measure import label
from scipy.spatial import Voronoi, cKDTree
from skimage.color import label2rgb
from skimage.segmentation import random_walker
from tensorflow.keras.models import load_model, Model


def unique_colors(n):
    """Get N unique RGB colors"""
    n2 = 2*n
    colors = np.random.randn(n2,3)
    colors /= np.linalg.norm(colors, axis=1, keepdims=True)
    colors *= np.sign(colors)
    colors *= np.random.uniform(0.2,1,(n2,1))
    #colors = colors.astype("f")
    colors = np.unique(colors,axis=0)
    idx = np.random.choice(colors.shape[0], n, replace=False)
    colors = colors[idx]
    colors[0,...] = 0
    return colors

    
def get_field_graph(eds, bse, points, embedding_model):
    """Calculate graph representation of EDS measurements in the field"""
    
    # Calculate embedding for each measurement
    #embedding = embedding_model.predict(eds / eds.sum(axis=1,keepdims=True), batch_size=256)
    embedding = embedding_model.predict(eds, batch_size=256)

    # Weighting can be applied here - e.g. if we know that something is more important
    # we can multiply the corresponding column by factor of 2. Or if something should
    # have no effect in segmentation, we can set it to 0.
    # e.g.: embedding[:,i] *= weight
    # embedding[:,j] = 0
    # or
    # embedding *=  weights  where weights is (1,N) vector of per-column factors
    # However it might be more convenient to move the wighting to label_graph phase which
    # can be done with small changes in the code.

    # Field subdivision
    vor = Voronoi(points)
    edges = vor.ridge_points
    nodes = np.arange(points.shape[0])

    # Spatial distance of points    
    pt_a = points[edges[:,0]]
    pt_b = points[edges[:,1]]
    spatial_dist = np.linalg.norm(pt_a-pt_b, axis=-1)

    # Embedding distance
    emb_a = embedding[edges[:,0]]
    emb_b = embedding[edges[:,1]]
    d = emb_a-emb_b
    eds_dist = np.linalg.norm(d, axis=-1)
    #eds_dist = np.abs(emb_a - emb_b).sum(axis=1)
    #eds_dist = np.abs(emb_a - emb_b).max(axis=1)

    # BSE distance
    bse_a = bse[edges[:,0]]
    bse_b = bse[edges[:,1]]
    bse_dist = np.abs(bse_a-bse_b)
    
    return (nodes, edges, spatial_dist, eds_dist, bse_dist), embedding


from sklearn.neighbors import NearestNeighbors

def label_graph(nodes, edges):
    """Label graph nodes by removing high cost edges"""

    # Compose graph from low cost edges
    G = nx.Graph()
    G.add_nodes_from(nodes)
    G.add_edges_from(edges)

    # Get the connected components
    lbl = np.full(nodes.size, -1, "i")
    for i,cpt in enumerate(nx.connected_components(G)):
        n = np.array(list(cpt),"i")
        lbl[n] = i

    return lbl
    

def segment_image(field, model, band, threshold=0.5, l=1):
    """Segment the given field"""
    band = slice(*band)
    S = GraphEmbeddingSegmentation(field, band, model)
    S.label_nodes(threshold=threshold, l=l)
    return S.field_labels()


def softmax(x, axis=-1):
    y = np.exp(-x)
    return y / y.sum(axis=axis,keepdims=1)


def load_embedding_network(filename):
    model = load_model(filename, compile=False)    
    if len(model.input_shape) != 3:
        return model
    from tensorflow.keras.layers import Input, Reshape
    _,v,w = model.input_shape
    in_data = Input(shape=(v,))
    x = Reshape([v,w])(in_data)
    y = model(x)
    return Model(in_data, y)


class GraphEmbeddingSegmentation:
    """
    Field segmentation with Graph and MRF.

    During initialization, a slow pre-processing step is executed. The pre-processing include:
    * Building field graph
    * Lookup table for each pixel containing nearest pixel with EDS measurement (for fast previews)
    Initialization can be very slow since each EDS goes through neural network calculating
    its embedding. And graph of all pixels must be built. On DOT fields, this is done in 5-10 seconds
    (depending on the number of measurements). On HIRES fields, this can take even few minutes.

    Segmentation is executed in two steps:
    1. Node labeling - label_nodes will get labels for each EDS. After this step, a preview
       labeling of full image can be generated by preview_image()

    2. MRF for labeling of unknown pixels -

    Note that step 1 is very fast and segmentation parameters can be set quicly with
    visual feedback (via preview_image). The step 2 can be slow. But on standard
    fields with lower tens of thousands measurements it takes around 2 seconds.

    """
    def __init__(self, field, spectrum_band, embedding_model):
        self.field = field
        self.points = field.pixels
        
        self.graph, self.eds_embedding = get_field_graph(
            field.eds[:,spectrum_band].astype("f"),
            field.normalized_bse[field.r, field.c],
            self.points,
            embedding_model)
        
        nbrs = NearestNeighbors(n_neighbors=4, radius=5).fit(self.points[:,[1,0]])
        x = np.array(np.nonzero(np.ones_like(field.bse))).T
        self.w, self.y  = nbrs.kneighbors(x, n_neighbors=3, return_distance=True)
        
        n_dist, n_idx = nbrs.kneighbors(self.points[:,[1,0]], n_neighbors=20)
        self.close_samples_dist = n_dist
        self.close_samples_idx = n_idx
        
    def label_nodes(self, theta_e, theta_b):
        nodes, edges, sd, ed, bd = self.graph

        # valid_edge_idx = np.logical_and(np.logical_and(
        #     sd < 2*self.field.sample_distance,
        #     ed < theta_e
        # ), bd < theta_b)

        valid_edge_idx = np.logical_and.reduce( (bd<theta_b, ed<theta_e, sd<1.5*self.field.sample_distance) )

        self.labels_ = label_graph(nodes, edges[valid_edge_idx])

        C = nx.Graph()
        C.add_nodes_from(np.arange(self.labels_.max()+1))
        for i in range(1, self.close_samples_dist.shape[1]):
            edges = self.close_samples_idx[:,[0,i]]
            C.add_edges_from(self.labels_[edges])
        lbl_dict = nx.greedy_color(C)
        new_lbl = np.full_like(self.labels_, -1, "i")
        for src,dst in lbl_dict.items():
            k = self.labels_ == src
            new_lbl[k] = dst
        #print(f"{new_lbl.max()} labels for MRF")
        self.labels_ = new_lbl
        
    def preview_image(self):
        if hasattr(self, "labels_"):
            preview_labels = (self.labels_[self.y[:,0]]+2).reshape(self.field.normalized_bse.shape)
            preview_labels[self.field.valid_mask()==0] = 1
            return label(preview_labels, 4 , background=1)

    def field_labels(self):
        if hasattr(self, "labels_"):
            dense_labels = np.squeeze(self.field.as_dense_array(self.labels_+2))
            dense_labels[self.field.valid_mask()==0] = 1
            labels = random_walker(self.field.normalized_bse*10, dense_labels, mode='bf', beta=0.01)
            labels = label(labels, 4, background=1)
            return labels

    def peak_map(self, peak, n=1):
        assert 0 <= peak < self.eds_embedding.shape[1]
        indices = self.y[:,:n]
        weights = np.exp(-self.w[:,:n] / 5)
        weights /= weights.sum(1,keepdims=True)  # can be precomputed
        map_data = self.eds_embedding[:,peak]
        m = (map_data[indices] * weights).sum(1)
        return m.reshape(self.field.bse.shape)
