#include <iostream>
#include <algorithm>
#include <cmath>
#include <cstdlib>
#include <opencv2/core/core.hpp>
#include <opencv2/highgui/highgui.hpp>
#include <opencv2/imgproc/imgproc.hpp>

#include <ctime>

long wall_clock()
{
	struct timespec t;

	if (clock_gettime(CLOCK_REALTIME, &t) < 0) {
		fprintf(stderr, "wall-clock error\n");
		return 0;
	}

	return t.tv_sec * 1000000000L + t.tv_nsec;
}

extern "C" {
#	include "common.h"
#	include "plane.h"
#	include "dwt97.h"
}

cv::Size enlargeSize(const cv::Size &size)
{
	int n = static_cast<int>(ceil_multiple(
		static_cast<size_t>(cvCeil(
			std::max(size.width, size.height) * sqrt(2)
		))
	));

	return cv::Size(n, n);
}

cv::Mat enlargeImage(const cv::Mat &src)
{
	cv::Size size = src.size();
	cv::Size size2 = enlargeSize(size);

	cv::Mat dst;

	cv::Size d0 = (size2 - size) / 2;
	cv::Size d1 = (size2 - size) - d0;

	// top, bottom, left, right
	copyMakeBorder(src, dst, d0.height, d1.height, d0.width, d1.width, cv::BORDER_REFLECT_101);

	return dst;
}

cv::Mat cropImage(const cv::Mat &ref, const cv::Mat &src)
{
	cv::Size size = ref.size();
	cv::Size size2 = enlargeSize(size);

	cv::Mat dst;

	cv::Size d0 = (size2 - size) / 2;

	return cv::Mat(src, cv::Rect(d0.width, d0.height, size.width, size.height));
}

cv::Mat rotate(const cv::Mat &src, int dir, int D)
{
	double angle = 90 * abs(dir) / D;

	cv::Size size = src.size();
	cv::Point2f center((float)size.width / 2.f, (float)size.height / 2.f);

	cv::Mat R = cv::getRotationMatrix2D(center, angle, 1);

	cv::Mat dst;

	#pragma omp critical
	cv::warpAffine(src, dst, R, size, cv::INTER_CUBIC + (dir < 0 ? cv::WARP_INVERSE_MAP : 0) + cv::WARP_FILL_OUTLIERS, /*cv::BORDER_WRAP*/0);

	return dst;
}

cv::Mat denoise(const cv::Mat &source, int levels = 6, int directions = 4, float lambda = 40.f)
{
	set_block_size(1 << levels);

	cv::Size size = source.size();

	std::cout << "Size " << size << std::endl;

	cv::Mat large = enlargeImage(source);

	std::cout << "...extended" << std::endl;

	cv::Mat output(size, CV_32FC1, cv::Scalar(0.f));

	#pragma omp parallel for
	for (int i = 0; i < directions; ++i) {
		cv::Mat tmp = rotate(large, +i, directions);
		std::cout << "...rotated" << std::endl;

		struct plane plane;
		plane.height = tmp.size().height,
		plane.width = tmp.size().width,
		plane.stride_x = 1,
		plane.stride_y = tmp.step1(),
		plane.data = reinterpret_cast<float *>(tmp.data);

		dwt97_encode(&plane, levels);

		dwt97_threshold(&plane, levels, lambda);

		dwt97_decode(&plane, levels);

		tmp = rotate(tmp, -i, directions);
		std::cout << "...rotated back" << std::endl;

		output += cropImage(source, tmp);
	}

	output /= directions;

	return output;
}

int main(int argc, char **argv)
{
	const char *path = argc > 1 ? argv[1] : "Lenna.ppm";
	int levels = 6;
	float lambda = argc > 2 ? (float)atof(argv[2]) : 40.f;
	int directions = argc > 3 ? atoi(argv[3]) : 4;

	std::cout << "Usage: " << argv[0] << " [input-image] [threshold=40] [directions=4]" << std::endl;

	std::cout << "Loading " << path << std::endl;

	cv::Mat source = cv::Mat_<float>(cv::imread(path, cv::ImreadModes::IMREAD_GRAYSCALE));
	if (!source.data) {
		std::cerr << "Failed do load image" << std::endl;
		return -1;
	}

	long start = wall_clock();

	cv::Mat output = denoise(source, levels, directions, lambda);

	printf("elapsed time: %f secs\n", (float)(wall_clock() - start) / (float)1000000000L);

	cv::imwrite("output.pgm", cv::Mat_<uchar>(output));

	return 0;
}
