#!/usr/bin/env python3
# -*- coding: utf-8 -*-

""" Test package for serializable.py """
# Copyright (C) 2018 Libor Polčák
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import unittest
import json
import parametrizable_tc as ptc
import consensus_parser as cp

import serializable as s

# Beautiful soup https://www.crummy.com/software/BeautifulSoup/bs4/doc/
bs_required_text = "BeautifulSoup not installed on this computer"
try:
    from bs4 import BeautifulSoup
    bs_failed = False
except ImportError as e:
    import sys
    sys.stderr.write(str(e))
    bs_failed = True

class test_files(unittest.TestCase, metaclass=ptc.parametrizable_tc_meta):

    def test_base_init(self):
        try:
            a = s.serializable_list([])
        except NotImplementedError:
            return
        except:
            self.assertTrue(False, str(e))
        self.assertTrue(False, str(a.__class__))

    @unittest.skipIf(bs_failed, bs_required_text)
    def test_htmlize_lines_links(self):
        data = [("a", "b"), ("c", "d")]
        soup = BeautifulSoup("\n".join(s.serializable_links(data).htmlize_lines()), "html.parser")
        ul = soup.find("ul")
        self.assertTrue(ul)
        links = [(link.string, link.get('href')) for link in ul.find_all('a')]
        for l in data:
            self.assertIn(l, links)

    @unittest.skipIf(bs_failed, bs_required_text)
    @ptc.parametrizable_test([
        ("ipv4", ["testfiles/preprocessed/147/229/147.229.13.223",
            ['pkYCSI2r79XMzeFFVLbF+73Sr3w', '+LzaWDVh0IrUPzGyrGHCXijdAE0']]),
        ("ipv6", ["testfiles/preprocessed/2001/67c/2001:67c:15e4:5be:216:3eff:fe15:74cb",
            ["nff8qIHEDx0ZxyCz2xrUbPhS9Tk"]]),
        ])
    def test_htmlize_ors(self, path, digests):
        ors = cp.parse_ors_from_file(path)
        urlbase = "/addresses"
        urlip = "1"
        urldate = "date"
        urltime = "urltime"
        html = "\n".join(s.serializable_ors(ors, urlbase, urlip, urldate, urltime).htmlize_lines())
        soup = BeautifulSoup(html, "html.parser")
        or_records = soup.find_all("section", recursive=False)
        self.assertEqual(len(or_records), len(digests), or_records)
        check_keys = ['version', 'Valid in consensuses after ', 'port', 'Dirport',
                'Nickname', 'Digest', 'published at', 'Bandwidth',
                'Tor exit policy', 'valid until', 'IPv4 address', 'Identity',
                'fresh until', 'flags']
        for r in or_records:
            for k in check_keys:
                self.assertIn(k, r.text)
        for ul, d in zip(or_records, digests):
            self.assertIn(d, ul.text)

    def test_jsonify_links(self):
        data = [("a", "b"), ("c", "d")]
        self.assertEqual(s.serializable_links(data).jsonify(), data)

    def test_jsonify_links_with_extend(self):
        data1 = [("a", "b")]
        data2 = [("c", "d")]
        data = data1 + data2
        links = s.serializable_links(data1)
        links.extend(s.serializable_links(data2))
        self.assertEqual(links.jsonify(), data)

    @ptc.parametrizable_test([
        ("ipv4", ["testfiles/preprocessed/147/229/147.229.13.223",
            ['pkYCSI2r79XMzeFFVLbF+73Sr3w', '+LzaWDVh0IrUPzGyrGHCXijdAE0']]),
        ("ipv6", ["testfiles/preprocessed/2001/67c/2001:67c:15e4:5be:216:3eff:fe15:74cb",
            ["nff8qIHEDx0ZxyCz2xrUbPhS9Tk"]]),
        ])
    def test_jsonify_ors(self, path, digests):
        ors = cp.parse_ors_from_file(path)
        sors = s.serializable_ors(ors, "", "", "", "").jsonify()
        self.assertTrue(isinstance(sors, list))
        self.assertEqual(len(sors), len(digests))
        check_keys = ['version', 'inconsensus_val_after', 'orport', 'dirport',
                'nickname', 'digest', 'publication', 'bandwidth',
                'allow_ports', 'inconsensus_val_until', 'ip', 'identity',
                'inconsensus_fresh_until', 'flags']
        for r in sors:
            self.assertTrue(isinstance(r, dict))
            for k in check_keys:
                self.assertIn(k, r)
        for r, d in zip(sors, digests):
            self.assertEqual(r["digest"], d)

def suite():
        test = unittest.makeSuite(test_files, "test")
        return unittest.TestSuite(tuple(test))

def test(verbosity=2, failfast=False):
        runner = unittest.TextTestRunner(verbosity=verbosity,failfast=failfast)
        runner.run(suite())

if __name__ == '__main__':
    test(verbosity=2)
