#!/usr/bin/env python3
# -*- coding: utf-8 -*-

""" Test package for Flask - REST API and related functions """
# Copyright (C) 2018 Libor Polčák
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import unittest
import json

import parametrizable_tc as ptc
import serializable

# Flask section
# See https://pythonhosted.org/Flask-Testing/, http://flask.pocoo.org/docs/1.0/testing/#testing
# Note that we want to allow running tests also on computers without Flask
try:
    import flask
    import flask_testing
    import rest
    flask_failed = False
    super_class = flask_testing.TestCase
except ImportError as e:
    import sys
    sys.stderr.write(str(e))
    flask_failed = True
    super_class = unittest.TestCase
    class fake:
        def __getattr__(self, attr):
            return None
    rest = fake()

# Beautiful soup https://www.crummy.com/software/BeautifulSoup/bs4/doc/
bs_required_text = "BeautifulSoup not installed on this computer"
try:
    from bs4 import BeautifulSoup
    bs_failed = False
except ImportError as e:
    import sys
    sys.stderr.write(str(e))
    bs_failed = True

class test_rest(super_class, metaclass=ptc.parametrizable_tc_meta):

    def create_app(self):
        self.app = rest.app
        self.app.config['TESTING'] = True
        return self.app

    def setUp(self):
        rest.PREPORCESSED_DIR = "testfiles/preprocessed/"
        rest.GEOLITE_DIR = "testfiles/geolite2/"

    def make_request(self, route, headers = []):
        return self.client.get(route, headers = headers)

    def make_request_json(self, route, headers = []):
        return self.make_request(route, headers + [('Accept', 'application/json')])

    def make_request_html(self, route, headers = []):
        return self.make_request(route, headers + [('Accept', 'text/html')])

    def test_htmlize_doctype(self):
        ret = rest.htmlize(None, None, lambda x: x)
        self.assertTrue(ret.startswith("<!DOCTYPE html>"))

    @unittest.skipIf(bs_failed, bs_required_text)
    @ptc.parametrizable_test([
            ("main_entry", [rest.entry_point, "Entry point"]),
            ("show_known_address_ranges", [rest.show_known_address_ranges, "Show known address ranges"]),
            ("show_known_address_subranges", [rest.show_known_address_subranges, "Show known address subranges"]),
            ("show_all_ip_address_information", [rest.show_all_ip_address_information, "Show all ip address information"]),
            ("show_ip_address_activity_dates",
                [rest.show_ip_address_activity_dates, "Show ip address activity dates"]),
            ("show_ip_address_activity_on_specific_date",
                [rest.show_ip_address_activity_on_specific_date, "Show ip address activity on specific date"]),
            ("show_ip_address_activity_at_specific_time",
                [rest.show_ip_address_activity_at_specific_time, "Show ip address activity at specific time"]),
            ("show_ip_address_active_months",
                [rest.show_ip_address_active_months, "Show ip address active months"]),
            ("show_ip_address_activity_during_specific_month",
                [rest.show_ip_address_activity_during_specific_month, "Show ip address activity during specific month"]),
        ])
    def test_htmlize_title(self, func, expected_subtitle):
        ret = rest.htmlize(None, None, func)
        soup = BeautifulSoup(ret, 'html.parser')
        self.assertEqual(soup.title.string, "Toreator REST API - %s" % expected_subtitle)

    @unittest.skipIf(bs_failed, bs_required_text)
    def test_htmlize_list_aka_links(self):
        ret = rest.htmlize(serializable.serializable_links([("a", "b")]), None, lambda x: x)
        soup = BeautifulSoup(ret, 'html.parser')
        self.assertTrue(("a", "b") in [ref for ref in
            [(link.string, link.get('href')) for link in soup.main.find_all('a')]])

    @ptc.parametrizable_test([
            ("all_some_old_browsers", ['*/*', False]),
            ("json", ['application/json', True]),
            ("seamonkey", ['text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8', False]),
            ("prefers_html_implicit", ['text/html,application/json;q=0.9', False]),
            ("prefers_html_explicit", ['text/html;q=0.6,application/json;q=0.2', False]),
            ("prefers_json_implicit", ['text/html;q=0.9,application/json', True]),
            ("prefers_json_explicit", ['text/html;q=0.6,application/json;q=0.8', True]),
        ])
    def test_request_wants_json(self, accept_content, expected):
        with self.app.test_request_context('/', headers = [('Accept', accept_content)]):
            self.assertEqual(rest.request_wants_json(), expected)

    def test_route_entry_point_json(self):
        response = self.make_request_json("/")
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.headers["Content-Type"], "application/json; charset=utf-8")
        res = json.loads(response.data.decode("utf-8"))
        self.assertIn("related", res, res)
        self.assertTrue("/addresses/" in [v for _,v in res["related"]])

    @unittest.skipIf(bs_failed, bs_required_text)
    def test_route_entry_point_html(self):
        response = self.make_request_html("/")
        self.assertEqual(response.status_code, 200)
        self.assertEqual(response.headers["Content-Type"], "text/html; charset=utf-8")
        soup = BeautifulSoup(response.data.decode("utf-8"), 'html.parser')
        self.assertTrue("/addresses/" in [ref for ref in [link.get('href') for link in soup.find_all('a')]])

    def test_rest_redirect(self):
        response = self.make_request_html("/addresses")
        self.assertEqual(response.status_code, 301) # Moved permanently

    @ptc.parametrizable_test([
            ("ipv4_collection", ["/addresses/", "147.0.0.0/8", "/addresses/147.0.0.0-8/"]),
            ("ipv6_collection", ["/addresses/", "2001::/16", "/addresses/2001::-16/"]),
            ("ipv4_subrange", ["/addresses/147.0.0.0-8/", "147.229.0.0/16", "/addresses/147.229.0.0-16/"]),
            ("ipv6_subrange", ["/addresses/2001::-16/", "2001:67c::/32", "/addresses/2001:67c::-32/"]),
            ("ipv4_addresses", ["/addresses/147.229.0.0-16/", "147.229.13.223", "/addresses/147.229.13.223/"]),
            ("ipv6_addresses", ["/addresses/2001:67c::-32/", "2001:67c:15e4:5be:216:3eff:fe15:74cb", "/addresses/2001:67c:15e4:5be:216:3eff:fe15:74cb/"]),
        ])
    def test_route_addresses(self, endpoint, expected_range, expected_endpoint):
        response = self.make_request_json(endpoint)
        self.assertEqual(response.status_code, 200)
        res = json.loads(response.data.decode("utf-8"))
        try:
            self.assertTrue([expected_range, expected_endpoint] in res["result"])
        except KeyError: # Provides a meaningful output
            self.assertTrue(False, res)

    @ptc.parametrizable_test([
            ("ipv4", ["147.229.13.223", 2, "pkYCSI2r79XMzeFFVLbF+73Sr3w"]),
            ("ipv6", ["2001:67c:15e4:5be:216:3eff:fe15:74cb", 1, "nff8qIHEDx0ZxyCz2xrUbPhS9Tk"]),
        ])
    def test_route_addresses_specific(self, addr, records, digest0):
        response = self.make_request_json("/addresses/%s/" % addr)
        self.assertEqual(response.status_code, 200)
        res = json.loads(response.data.decode("utf-8"))
        self.assertEqual(len(res["result"]), records)
        self.assertEqual(res["result"][0]["digest"], digest0)

    @ptc.parametrizable_test([
            ("ipv4", ["147.229.13.223", ["2013-04-02", "2013-04-03"]]),
            ("ipv6", ["2001:67c:15e4:5be:216:3eff:fe15:74cb", ["2018-04-21"]]),
        ])
    def test_route_addresses_date(self, addr, dates):
        response = self.make_request_json("/addresses/%s/date/" % addr)
        self.assertEqual(response.status_code, 200)
        res = json.loads(response.data.decode("utf-8"))
        self.assertEqual(len(res["result"]), len(dates))
        for date, link in zip(dates, res["result"]):
            self.assertEqual(link[1], "/addresses/%s/date/%s/" % (addr, date))

    @ptc.parametrizable_test([
            ("ipv4_1", ["147.229.13.223", "2013-04-02", "2013-04-01", "2013-04-03",
                2, "pkYCSI2r79XMzeFFVLbF+73Sr3w"]),
            ("ipv4_2", ["147.229.13.223", "2013-04-03", "2013-04-02", "2013-04-04",
                2, "pkYCSI2r79XMzeFFVLbF+73Sr3w"]),
            ("ipv4_none_exist", ["147.229.13.223", "2018-01-01", "2017-12-31", "2018-01-02", 0, None]),
            ("ipv4_none_other", ["10.10.10.10", "2017-12-31", "2017-12-30", "2018-01-01", 0, None]),
            ("ipv6", ["2001:67c:15e4:5be:216:3eff:fe15:74cb",
                "2018-04-21", "2018-04-20", "2018-04-22", 1, "nff8qIHEDx0ZxyCz2xrUbPhS9Tk"]),
            ("ipv6_none_exist_after", ["2001:67c:15e4:5be:216:3eff:fe15:74cb",
                "2018-04-20", "2018-04-19", "2018-04-21", 0, None]),
            ("ipv6_none_exist_before", ["2001:67c:15e4:5be:216:3eff:fe15:74cb",
                "2018-04-22", "2018-04-21", "2018-04-23", 0, None]),
            ("ipv6_none_other", ["2001:67c:15e4:5be:216:3eff:fe15:74cc",
                "2018-04-21", "2018-04-20", "2018-04-22",0, None]),
        ])
    def test_route_addresses_specific_date(self, addr, date, prev_date, next_date, items, digest):
        response = self.make_request_json("/addresses/%s/date/%s/" % (addr, date))
        self.assertEqual(response.status_code, 200)
        res = json.loads(response.data.decode("utf-8"))
        self.assertEqual(len(res["result"]), items)
        if items > 1:
            self.assertEqual(res["result"][0]["digest"], digest)
        self.assertTrue("/addresses/%s/date/%s/" % (addr, prev_date) in [v for _, v in res["related"]])
        self.assertTrue("/addresses/%s/date/%s/" % (addr, next_date) in [v for _, v in res["related"]])
        self.assertTrue("/addresses/%s/month/%s/" % (addr, date[:7]) in [v for _, v in res["related"]])
        self.assertTrue("/addresses/%s/year/%s/" % (addr, date[:4]) in [v for _, v in res["related"]])

    def test_show_ip_address_activity_times_redirect(self):
        response = self.make_request_json("/addresses/147.229.1.1/time/")
        self.assertEqual(response.status_code, 301)

    @ptc.parametrizable_test([
            ("ipv4_1", ["147.229.13.223", "2013-04-02 23:00", "2013-04-02",
                2, "pkYCSI2r79XMzeFFVLbF+73Sr3w"]),
            ("ipv4_2", ["147.229.13.223", "4.2.2013 23:00", "2013-04-02",
                2, "pkYCSI2r79XMzeFFVLbF+73Sr3w"]),
            ("ipv4_1", ["147.229.13.223", "2013-04-02 23:15:41", "2013-04-02",
                2, "pkYCSI2r79XMzeFFVLbF+73Sr3w"]),
            ("ipv4_none_exist", ["147.229.13.223", "2013-04-02 16:00", "2013-04-02", 0, None]),
            ("ipv4_none_other", ["10.10.10.10", "2017-12-31", "2017-12-31", 0, None]),
            ("ipv6", ["2001:67c:15e4:5be:216:3eff:fe15:74cb",
                "2018-04-21 16:14:11", "2018-04-21", 1, "nff8qIHEDx0ZxyCz2xrUbPhS9Tk"]),
        ])
    def test_route_addresses_specific_time(self, addr, time, date, items, digest):
        response = self.make_request_json("/addresses/%s/time/%s/" % (addr, time))
        self.assertEqual(response.status_code, 200)
        res = json.loads(response.data.decode("utf-8"))
        self.assertEqual(len(res["result"]), items)
        if items > 1:
            self.assertEqual(res["result"][0]["digest"], digest)
        self.assertTrue("/addresses/%s/date/%s/" % (addr, date) in [v for _, v in res["related"]])
        self.assertTrue("/addresses/%s/month/%s/" % (addr, date[:7]) in [v for _, v in res["related"]])
        self.assertTrue("/addresses/%s/year/%s/" % (addr, date[:4]) in [v for _, v in res["related"]])

    @ptc.parametrizable_test([
            ("ipv4", ["147.229.13.223", ["2013-04"]]),
            ("ipv6", ["2001:67c:15e4:5be:216:3eff:fe15:74cb", ["2018-04"]]),
        ])
    def test_route_addresses_month(self, addr, months):
        response = self.make_request_json("/addresses/%s/month/" % addr)
        self.assertEqual(response.status_code, 200)
        res = json.loads(response.data.decode("utf-8"))
        self.assertEqual(len(res["result"]), len(months))
        for month, link in zip(months, res["result"]):
            self.assertEqual(link[1], "/addresses/%s/month/%s/" % (addr, month))

    @ptc.parametrizable_test([
            ("ipv4", ["147.229.13.223", "2013-04", "2013-03", "2013-05",
                2, "pkYCSI2r79XMzeFFVLbF+73Sr3w"]),
            ("ipv4_none_exist", ["147.229.13.223", "2018-01", "2017-12", "2018-02", 0, None]),
            ("ipv4_none_other", ["10.10.10.10", "2017-12", "2017-11", "2018-01", 0, None]),
            ("ipv6", ["2001:67c:15e4:5be:216:3eff:fe15:74cb",
                "2018-04", "2018-03", "2018-05", 1, "nff8qIHEDx0ZxyCz2xrUbPhS9Tk"]),
            ("ipv6_none_exist_after", ["2001:67c:15e4:5be:216:3eff:fe15:74cb",
                "2018-05", "2018-04", "2018-06", 0, None]),
            ("ipv6_none_exist_before", ["2001:67c:15e4:5be:216:3eff:fe15:74cb",
                "2018-03", "2018-02", "2018-04", 0, None]),
            ("ipv6_none_other", ["2001:67c:15e4:5be:216:3eff:fe15:74cc",
                "2018-04", "2018-03", "2018-05",0, None]),
        ])
    def test_route_addresses_specific_month(self, addr, month, prev_month, next_month, items, digest):
        response = self.make_request_json("/addresses/%s/month/%s/" % (addr, month))
        self.assertEqual(response.status_code, 200)
        res = json.loads(response.data.decode("utf-8"))
        self.assertEqual(len(res["result"]), items)
        if items > 0:
            self.assertEqual(res["result"][0]["digest"], digest)
        self.assertTrue("/addresses/%s/month/%s/" % (addr, prev_month) in [v for _, v in res["related"]])
        self.assertTrue("/addresses/%s/month/%s/" % (addr, next_month) in [v for _, v in res["related"]])
        self.assertTrue("/addresses/%s/year/%s/" % (addr, month[:4]) in [v for _, v in res["related"]])

    @ptc.parametrizable_test([
            ("ipv4", ["147.229.13.223", ["2013"]]),
            ("ipv6", ["2001:67c:15e4:5be:216:3eff:fe15:74cb", ["2018"]]),
        ])
    def test_route_addresses_year(self, addr, years):
        response = self.make_request_json("/addresses/%s/year/" % addr)
        self.assertEqual(response.status_code, 200)
        res = json.loads(response.data.decode("utf-8"))
        self.assertEqual(len(res["result"]), len(years))
        for year, link in zip(years, res["result"]):
            self.assertEqual(link[1], "/addresses/%s/year/%s/" % (addr, year))

    @ptc.parametrizable_test([
            ("ipv4", ["147.229.13.223", "2013", "2012", "2014",
                2, "pkYCSI2r79XMzeFFVLbF+73Sr3w"]),
            ("ipv4_none_exist", ["147.229.13.223", "2011", "2010", "2012", 0, None]),
            ("ipv4_none_other", ["10.10.10.10", "2018", "2017", "2019", 0, None]),
            ("ipv6", ["2001:67c:15e4:5be:216:3eff:fe15:74cb",
                "2018", "2017", "2019", 1, "nff8qIHEDx0ZxyCz2xrUbPhS9Tk"]),
            ("ipv6_none_exist_after", ["2001:67c:15e4:5be:216:3eff:fe15:74cb",
                "2019", "2018", "2020", 0, None]),
            ("ipv6_none_exist_before", ["2001:67c:15e4:5be:216:3eff:fe15:74cb",
                "2017", "2016", "2018", 0, None]),
            ("ipv6_none_other", ["2001:67c:15e4:5be:216:3eff:fe15:74cc",
                "2018", "2017", "2019",0, None]),
        ])
    def test_route_addresses_specific_year(self, addr, year, prev_year, next_year, items, digest):
        response = self.make_request_json("/addresses/%s/year/%s/" % (addr, year))
        self.assertEqual(response.status_code, 200)
        res = json.loads(response.data.decode("utf-8"))
        self.assertEqual(len(res["result"]), items)
        if items > 0:
            self.assertEqual(res["result"][0]["digest"], digest)
        self.assertTrue("/addresses/%s/year/%s/" % (addr, prev_year) in [v for _, v in res["related"]])
        self.assertTrue("/addresses/%s/year/%s/" % (addr, next_year) in [v for _, v in res["related"]])

def suite():
    if flask_failed:
        import sys
        sys.stderr.write("Please install Flask to test REST module.\n")
        return unittest.TestSuite(())
    else:
        test = unittest.makeSuite(test_rest, "test")
        return unittest.TestSuite(tuple(test))

def test(verbosity=2, failfast=False):
        runner = unittest.TextTestRunner(verbosity=verbosity,failfast=failfast)
        runner.run(suite())

if __name__ == '__main__':
    test(verbosity=2)
