# serializable.py: Functions transforming data to HTML
# Copyright (C) 2018 Libor Polčák <ipolcak@fit.vutbr.cz>
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

class serializable_list():
    """ This is a base class intended to be subclassed by specialized serializers. """

    def __init__(self, lst):
        if self.__class__ == serializable_list:
            raise NotImplementedError("Cannot create instance of abstract class")
        self._wrapped = lst

    def htmlize_lines(self):
        """ Returns a list of lines that represents the iterable. """
        raise NotImplementedError("Cannot call abstract method")

    def jsonify(self):
        """ Returns an object to be JSONified. """
        raise NotImplementedError("Cannot call abstract method")

    def append(self, item):
        self._wrapped.append(item)

    def extend(self, other):
        self._wrapped.extend(other._wrapped)

    @staticmethod
    def _make_html_link(url, desc):
        return "<a href='%s'>%s</a>" % (url, desc)

class serializable_ors(serializable_list):
    """ Provides functionality to HTMLize and JSONify ORs. """

    def __init__(self, lst, addressbase, ipstr, datestr, timestr):
        serializable_list.__init__(self, lst)
        self.__addressbase = addressbase
        self.__ipstr = ipstr
        self.__datestr = datestr
        self.__timestr = timestr
        self.__maxmind = ""

    def __htmlize_or(self, r, res):
        def make_timestamp(datestr, timestr):
            return self._make_html_link("%s/%s/%s/%s %s/" % (self.__addressbase,
                        self.__ipstr, self.__timestr, datestr, timestr), "%s %s" % (datestr,
                            timestr)),
        def append_item(itemstr):
            res.append("<li>%s</li>" % itemstr)
        def print_ipv4():
            append_item("IPv4 address %s port %s" % (r.get_ip(), r.get_orport()))
        def print_ipv6():
            for item in r.get_ipv6():
                append_item("IPv6 address %s port %s" % item)
        def print_reverse_name():
            name, queried = r.get_dns_reverse()
            html = ["<ul>"]
            for q in queried:
                html.append("<li>%s</li>" % make_timestamp(*q))
            html.append("</ul>")
            append_item("DNS reverse name %s queried at %s" % (name, "".join(html)))
        def print_roles():
            try:
                exit_policy = ", Tor exit policy %s" % r.get_allow_ports()
            except:
                exit_policy = ""
            append_item("Server flags in Tor network %s%s" % (", ".join(r.get_flags()), exit_policy))
        def print_geolocation():
            geolocation = r.get_maxmind_geolocation()
            res.append("<li>MaxMind Geolocation:<ul>")
            for item in geolocation.values():
                list_items = []
                for subdesc, value in item:
                    if subdesc == "network":
                        net, plen = value.split("/")
                        list_items.append("<li>%s: %s</li>" % (subdesc,
                            self._make_html_link("%s/%s-%s/" % (self.__addressbase, net, plen), value)))
                    elif subdesc == "timestamp":
                        d, t = value.split(" ")
                        validity = "from %s:</li>" % make_timestamp(d, t)
                    elif subdesc == "geolocation source":
                        self.__maxmind = value
                    else:
                        list_items.append("<li>%s: %s</li>" % (subdesc, value))
                append_item("%s<ul>%s</ul>" % (validity, "".join(list_items)))
            res.append("</ul></li>")
        def print_asn():
            geolocation = r.get_maxmind_asn()
            res.append("<li>MaxMind autonomous system number:<ul>")
            for item in geolocation.values():
                list_items = []
                for subdesc, value in item:
                    if subdesc == "AS network":
                        net, plen = value.split("/")
                        list_items.append("<li>%s: %s</li>" % (subdesc,
                            self._make_html_link("%s/%s-%s/" % (self.__addressbase, net, plen), value)))
                    elif subdesc == "timestamp":
                        d, t = value.split(" ")
                        validity = "from %s:</li>" % make_timestamp(d, t)
                    elif subdesc == "geolocation source":
                        self.__maxmind = value
                    else:
                        list_items.append("<li>%s: %s</li>" % (subdesc, value))
                append_item("%s<ul>%s</ul>" % (validity, "".join(list_items)))
            res.append("</ul></li>")
        def print_timestamps():
            val_after = make_timestamp(*r.get_inconsensus_val_after())
            fresh_until = make_timestamp(*r.get_inconsensus_fresh_until())
            val_until = make_timestamp(*r.get_inconsensus_val_until())
            append_item("Valid in consensuses after %s, fresh until %s, valid until %s" %
                    (val_after, fresh_until, val_until))
        def print_tor_router_parameters():
            for name, getter in (
                    ("Tor software version", r.get_version),
                    ("Node bandwidth", r.get_bandwidth),
                    ("Identity", r.get_identity),
                    ("Digest", r.get_digest),
                    ("Dirport", r.get_dirport),
                    ("Supported protocols", r.get_supported_proto),
                    ):
                try:
                    append_item("%s: %s" % (name, getter()))
                except:
                    pass
        print_functions = [print_ipv4, print_ipv6, print_reverse_name,
                print_roles, print_geolocation, print_asn, print_timestamps,
                print_tor_router_parameters]
        res.append("<section>")
        try:
            pub = r.get_publication()
            published = " published at %s" % make_timestamp(pub[0], pub[1])
        except:
            published = ""
        try:
            res.append("<h2>Nickname %s %s</h2>" % (r.get_nickname(), published))
        except:
            res.append("<h2>Onion router item</h2>")
        res.append("<ul>")
        for f in print_functions:
            try:
                f()
            except:
                pass
        res.append("</ul>")
        res.append("</section>")

    def htmlize_lines(self):
        ret = []
        for r in self._wrapped:
            self.__htmlize_or(r, ret)
        ret.append(self.__maxmind)
        return ret

    def jsonify(self):
        return [r.get_json_dict() for r in self._wrapped]

class serializable_links(serializable_list):
    """ Provides functionality to HTMLize and JSONify links.
    
    Stores a list of two-tuples - (Link description, URI).
    """

    def __init__(self, lst):
        serializable_list.__init__(self, lst)

    def htmlize_lines(self):
        ret = ["<ul>"]
        for desc, url in self._wrapped:
            ret.append("<li>%s</li>" % self._make_html_link(url, desc))
        ret.append("<ul>")
        return ret

    def jsonify(self):
        return self._wrapped
