"""
"""
from os.path import dirname, abspath, join
import sys
sys.path.append(abspath(join(dirname(__file__), '..', '')))

import json
import logging
import unittest

from dm.ConnectionUtil import ConnectionUtil as cu
from dm.models.ModelsUtil import ModelsUtil
from dm.DateTimeUtil import DateTimeUtil
from dm.WundergroundCom import WundergroundCom

__author__ = ''
__email__ = ''


def to_timestamp(time_str):
    return int(DateTimeUtil.local_time_str_to_utc(time_str).timestamp())


class Test0301Estimate(unittest.TestCase):
    def setUp(self):
        self.cls = cu.setup_clients()
        cu.setup_logging(logging.ERROR)

    def tearDown(self):
        cu.setup_clients_logout(self.cls)

    def test_0300_estimate_open_co2_notification_ok(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'co2_in_ppm',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 2,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.estimate_open_co2(devs, cls, lat, lon, actual_time, w)
        expected = {
            "data": {
                "estimate_open": False,
                "type": "co2_open"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": True,
            "raise_catch": False,
            "readable": "2019-02-20 03:00:00",
            "timestamp": 1550628000
        }
        self.assertEqual(expected, notification)

    def test_0300_estimate_open_co2_notification_fail(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/10/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'co2_in_ppm',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 2,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.estimate_open_co2(devs, cls, lat, lon, actual_time, w)
        expected = {
            "data": {
                "estimate_open": None,
                "type": "co2_open"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": True,
            "readable": "2019-10-20 03:00:00",
            "timestamp": 1571533200
        }
        self.assertEqual(expected, notification)

    def test_0300_estimate_open_t_h_notification_ok(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'temperature_in2_celsius',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            },
            {
                'db_column_name': 'rh_in2_percentage',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.estimate_open_t_h(devs, cls, lat, lon, actual_time, w)
        expected = {
            "data": {
                "estimate_open": True,
                "type": "t_h_open"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": True,
            "raise_catch": False,
            "readable": "2019-02-20 03:00:00",
            "timestamp": 1550628000
        }
        self.assertEqual(expected, notification)

    def test_0300_estimate_open_t_h_notification_fail(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/10/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'temperature_in2_celsius',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            },
            {
                'db_column_name': 'rh_in2_percentage',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.estimate_open_t_h(devs, cls, lat, lon, actual_time, w)
        expected = {
            "data": {
                "estimate_open": None,
                "type": "t_h_open"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": True,
            "readable": "2019-10-20 03:00:00",
            "timestamp": 1571533200
        }
        self.assertEqual(expected, notification)

    def test_0301_estimate_co2_notification_ok_no_send_notification_none_estimate_as_zero(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 03:00:00').timestamp())

        devs = [
            {
                'db_column_name': 'co2_in_ppm',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 2,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.estimate_co2(devs, cls, actual_time)
        expected = {
            "data": {
                "current_value": 2000,
                "es_level": 2000,
                "es_time": 60,
                "estimate_time": 0,
                "level": 2,
                "level_value": 2500,
                "type": "co2_estimate"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": False,
            "readable": "2019-02-20 03:00:00",
            "timestamp": 1550628000
        }
        self.assertEqual(expected, notification)

    def test_0301_estimate_co2_notification_ok_no_send_notification(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 06:00:00').timestamp())

        devs = [
            {
                'db_column_name': 'co2_in_ppm',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 2,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.estimate_co2(devs, cls, actual_time)
        expected = {
            "data": {
                "current_value": 2000,
                "es_level": 2000,
                "es_time": 60,
                "estimate_time": 0,
                "level": 2,
                "level_value": 2500,
                "type": "co2_estimate"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": False,
            "readable": "2019-02-20 06:00:00",
            "timestamp": 1550638800
        }
        self.assertEqual(expected, notification)

    def test_0301_estimate_co2_notification_ok_send_notification(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 09:00:00').timestamp())

        devs = [
            {
                'db_column_name': 'co2_in_ppm',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 2,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.estimate_co2(devs, cls, actual_time)
        expected = {
            "data": {
                "current_value": 1492,
                "es_level": 1619,
                "es_time": 60,
                "estimate_time": 22,
                "level": 1,
                "level_value": 1500,
                "type": "co2_estimate"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": True,
            "raise_catch": False,
            "readable": "2019-02-20 09:00:00",
            "timestamp": 1550649600
        }
        self.assertEqual(expected, notification)

    def test_0301_estimate_co2_notification_fail(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/10/20 03:00:00').timestamp())

        devs = [
            {
                'db_column_name': 'co2_in_ppm',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 2,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.estimate_co2(devs, cls, actual_time)
        expected = {
            "data": {
                "current_value": None,
                "es_level": None,
                "es_time": 60,
                "estimate_time": None,
                "level": None,
                "level_value": None,
                "type": "co2_estimate"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": True,
            "readable": "2019-10-20 03:00:00",
            "timestamp": 1571533200
        }
        self.assertEqual(expected, notification)

    def test_0301_estimate_t_h_notification_ok_no_send_notification_none_estimate_as_zero(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 11:00:00').timestamp())

        devs = [
            {
                'db_column_name': 'temperature_in2_celsius',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            },
            {
                'db_column_name': 'rh_in2_percentage',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.estimate_t_h(devs, cls, actual_time)
        expected = {
            "data": {
                "current_value": 34,
                "es_level": 34,
                "es_time": 60,
                "estimate_time": 0,
                "level": 1,
                "level_value": 40,
                "type": "co2_estimate"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": False,
            "timestamp": 1550656800,
            "readable": "2019-02-20 11:00:00"
        }
        self.assertEqual(expected, notification)

    def test_0301_estimate_t_h_notification_ok_no_send_notification(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 08:00:00').timestamp())

        devs = [
            {
                'db_column_name': 'temperature_in2_celsius',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            },
            {
                'db_column_name': 'rh_in2_percentage',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.estimate_t_h(devs, cls, actual_time)
        expected = {
            "data": {
                "current_value": 32,
                "es_level": 36,
                "es_time": 60,
                "estimate_time": 75,
                "level": 1,
                "level_value": 40,
                "type": "co2_estimate"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": False,
            "timestamp": 1550646000,
            "readable": "2019-02-20 08:00:00"
        }
        self.assertEqual(expected, notification)

    def test_0301_estimate_t_h_notification_ok_send_notification(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 07:30:00').timestamp())

        devs = [
            {
                'db_column_name': 'temperature_in2_celsius',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            },
            {
                'db_column_name': 'rh_in2_percentage',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.estimate_t_h(devs, cls, actual_time)
        expected = {
            "data": {
                "current_value": 30,
                "es_level": 10,
                "es_time": 60,
                "estimate_time": 12,
                "level": 1,
                "level_value": 40,
                "type": "co2_estimate"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": True,
            "raise_catch": False,
            "timestamp": 1550644200,
            "readable": "2019-02-20 07:30:00"
        }
        self.assertEqual(expected, notification)

    def test_0301_estimate_t_h_notification_fail(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/10/20 03:00:00').timestamp())

        devs = [
            {
                'db_column_name': 'temperature_in2_celsius',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            },
            {
                'db_column_name': 'rh_in2_percentage',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.estimate_t_h(devs, cls, actual_time)
        expected = {
            "data": {
                "current_value": None,
                "es_level": None,
                "es_time": 60,
                "estimate_time": None,
                "level": None,
                'level_value': None,
                "type": "co2_estimate"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": True,
            "readable": "2019-10-20 03:00:00",
            "timestamp": 1571533200
        }
        self.assertEqual(expected, notification)

    def test_0302_predictor_co2_notification_ok(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'co2_in_ppm',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 2,
                'server_name': 'ant-work',
            },
            {
                "db_column_name": "temperature_in_celsius",
                "gateway_id": "1816820318180747",
                "device_id": "0xa900811026800001",
                "module_id": 0,
                "server_name": "ant-work"
            },
            {
                "db_column_name": "rh_in_percentage",
                "gateway_id": "1816820318180747",
                "device_id": "0xa900811026800001",
                "module_id": 1,
                "server_name": "ant-work"
            }
        ]

        notification = ModelsUtil.predictor(devs, cls, lat, lon, actual_time, 'co2', w)
        expected = {
            "data": {
                "current_value": 2000,
                "estimate_time": 10,
                'final_value': 400,
                "type": "co2_predictor"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": True,
            "raise_catch": False,
            "readable": "2019-02-20 03:00:00",
            "timestamp": 1550628000
        }
        self.assertEqual(expected, notification)

    def test_0302_predictor_co2_notification_fail(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/10/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'co2_in_ppm',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 2,
                'server_name': 'ant-work',
            },
            {
                "db_column_name": "temperature_in_celsius",
                "gateway_id": "1816820318180747",
                "device_id": "0xa900811026800001",
                "module_id": 0,
                "server_name": "ant-work"
            },
            {
                "db_column_name": "rh_in_percentage",
                "gateway_id": "1816820318180747",
                "device_id": "0xa900811026800001",
                "module_id": 1,
                "server_name": "ant-work"
            }
        ]

        notification = ModelsUtil.predictor(devs, cls, lat, lon, actual_time, 'co2', w)
        expected = {
            "data": {
                "current_value": None,
                "estimate_time": None,
                'final_value': 400,
                "type": "co2_predictor"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": True,
            "readable": "2019-10-20 03:00:00",
            "timestamp": 1571533200
        }
        self.assertEqual(expected, notification)

    def test_0302_predictor_t_h_notification_ok(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'temperature_in2_celsius',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            },
            {
                'db_column_name': 'rh_in2_percentage',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.predictor(devs, cls, lat, lon, actual_time, 't_h', w, 40)
        expected = {
            "data": {
                "current_value": 41,
                "estimate_time": 10,
                "final_value": 40,
                "type": "t_h_predictor",
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": True,
            "raise_catch": False,
            "timestamp": 1550628000,
            "readable": "2019-02-20 03:00:00"
        }
        self.assertEqual(expected, notification)

    def test_0302_predictor_t_h_notification_fail(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/10/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'temperature_in2_celsius',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            },
            {
                'db_column_name': 'rh_in2_percentage',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.predictor(devs, cls, lat, lon, actual_time, 't_h', w, 40)
        expected = {
            "data": {
                "current_value": None,
                "estimate_time": None,
                "final_value": 40,
                "type": "t_h_predictor",
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": True,
            "timestamp": 1571533200,
            "readable": "2019-10-20 03:00:00"
        }
        self.assertEqual(expected, notification)

    def test_0303_when_ventilate_predictor_notification_ok(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'temperature_in_celsius',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            },
            {
                'db_column_name': 'rh_in_percentage',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.when_ventilate_summer(devs, cls, lat, lon, actual_time, w, temperature_diff=4)
        expected = {
            "data": {
                "temp_in": 26.0,
                "temp_out": 3.0,
                "type": "when_ventilate_summer",
                "ventilate": 2
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": True,
            "raise_catch": False,
            "readable": "2019-02-20 03:00:00",
            "timestamp": 1550628000
        }
        self.assertEqual(expected, notification)

    def test_0303_when_ventilate_predictor_notification_fail(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/10/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'temperature_in_celsius',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            },
            {
                'db_column_name': 'rh_in_percentage',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.when_ventilate_summer(devs, cls, lat, lon, actual_time, w, temperature_diff=4)
        expected = {
            "data": {
                "temp_in": None,
                "temp_out": None,
                "type": "when_ventilate_summer",
                "ventilate": None
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": True,
            "readable": "2019-10-20 03:00:00",
            "timestamp": 1571533200
        }
        self.assertEqual(expected, notification)

    def test_0304_dew_point_notification_ok(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'temperature_in_celsius',
                'name': 'BeeeOn sensor',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            },
            {
                'db_column_name': 'rh_in_percentage',
                'name': 'BeeeOn sensor',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.dew_point(devs, cls, lat, lon, actual_time, w)
        expected = {
            "data": {
                "dew_point": 11.75,
                "dewing": None,
                "hum_in": 41.0,
                "temp_in": 26.0,
                "type": "dew_point"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": True,
            "raise_catch": False,
            "readable": "2019-02-20 03:00:00",
            "timestamp": 1550628000
        }
        self.assertEqual(expected, notification)

    def test_0304_dew_point_notification_fail(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/10/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'temperature_in_celsius',
                'name': 'BeeeOn sensor',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            },
            {
                'db_column_name': 'rh_in_percentage',
                'name': 'BeeeOn sensor',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]

        notification = ModelsUtil.dew_point(devs, cls, lat, lon, actual_time, w)
        expected = {
            "data": {
                "dew_point": None,
                "dewing": None,
                "hum_in": None,
                "temp_in": None,
                "type": "dew_point"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": True,
            "readable": "2019-10-20 03:00:00",
            "timestamp": 1571533200
        }
        self.assertEqual(expected, notification)

    def test_0305_anomaly_diff_t_notification_ok_send_notification(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 23:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'temperature_in_celsius',
                'name': 'BeeeOn sensor',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            }
        ]
        notification = ModelsUtil.anomaly_diff(devs, cls, lat, lon, actual_time, w, 1, 7200, 'temperature')
        expected = {
            "data": {
                "actual_diff": 1.1,
                "anomaly": True,
                "min_anomaly_diff": 1,
                "min_anomaly_time": 120,
                "type": "anomaly_temperature"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": True,
            "raise_catch": False,
            "readable": "2019-02-20 23:00:00",
            "timestamp": 1550700000
        }
        self.assertEqual(expected, notification)

    def test_0305_anomaly_diff_t_notification_ok_no_send_notification(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 23:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'temperature_in_celsius',
                'name': 'BeeeOn sensor',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            }
        ]
        notification = ModelsUtil.anomaly_diff(devs, cls, lat, lon, actual_time, w, 2, 7200, 'temperature')
        expected = {
            "data": {
                "actual_diff": 1.1,
                "anomaly": False,
                "min_anomaly_diff": 2,
                "min_anomaly_time": 120,
                "type": "anomaly_temperature"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": False,
            "readable": "2019-02-20 23:00:00",
            "timestamp": 1550700000
        }
        self.assertEqual(expected, notification)

    def test_0305_anomaly_diff_t_notification_fail(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/10/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'temperature_in_celsius',
                'name': 'BeeeOn sensor',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 0,
                'server_name': 'ant-work',
            }
        ]
        notification = ModelsUtil.anomaly_diff(devs, cls, lat, lon, actual_time, w, 2, 7200, 'temperature')
        expected = {
            "data": {
                "actual_diff": None,
                "anomaly": None,
                'min_anomaly_diff': 2,
                'min_anomaly_time': 120,
                "type": "anomaly_temperature"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": True,
            "readable": "2019-10-20 03:00:00",
            "timestamp": 1571533200
        }
        self.assertEqual(expected, notification)

    def test_0305_anomaly_diff_h_notification_ok_send_notification(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'rh_in_percentage',
                'name': 'BeeeOn sensor',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]
        notification = ModelsUtil.anomaly_diff(devs, cls, lat, lon, actual_time, w, 1, 7200, 'humidity')
        expected = {
            "data": {
                "actual_diff": 1.5,
                "anomaly": True,
                "min_anomaly_diff": 1,
                "min_anomaly_time": 120,
                "type": "anomaly_humidity"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": True,
            "raise_catch": False,
            "readable": "2019-02-20 03:00:00",
            "timestamp": 1550628000
        }
        self.assertEqual(expected, notification)

    def test_0305_anomaly_diff_h_notification_ok_no_send_notification(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'rh_in_percentage',
                'name': 'BeeeOn sensor',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]
        notification = ModelsUtil.anomaly_diff(devs, cls, lat, lon, actual_time, w, 2, 7200, 'humidity')
        expected = {
            "data": {
                "actual_diff": 1.5,
                "anomaly": False,
                "min_anomaly_diff": 2,
                "min_anomaly_time": 120,
                "type": "anomaly_humidity"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": False,
            "readable": "2019-02-20 03:00:00",
            "timestamp": 1550628000
        }
        self.assertEqual(expected, notification)

    def test_0305_anomaly_diff_h_notification_fail(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/10/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'rh_in_percentage',
                'name': 'BeeeOn sensor',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 1,
                'server_name': 'ant-work',
            }
        ]
        notification = ModelsUtil.anomaly_diff(devs, cls, lat, lon, actual_time, w, 2, 7200, 'humidity')
        expected = {
            "data": {
                "actual_diff": None,
                "anomaly": None,
                "min_anomaly_diff": 2,
                "min_anomaly_time": 120,
                "type": "anomaly_humidity"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": True,
            "readable": "2019-10-20 03:00:00",
            "timestamp": 1571533200
        }
        self.assertEqual(expected, notification)

    def test_0305_anomaly_diff_co2_notification_ok_send_notification(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 23:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'co2_in_ppm',
                'name': '',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 2,
                'server_name': 'ant-work',
            }
        ]
        notification = ModelsUtil.anomaly_diff(devs, cls, lat, lon, actual_time, w, 100, 7200, 'co2')
        expected = {
            "data": {
                "actual_diff": 882.2,
                "anomaly": True,
                "min_anomaly_diff": 100,
                "min_anomaly_time": 120,
                "type": "anomaly_co2"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": True,
            "raise_catch": False,
            "readable": "2019-02-20 23:00:00",
            "timestamp": 1550700000
        }
        self.assertEqual(expected, notification)

    def test_0305_anomaly_diff_co2_notification_ok_no_send_notification(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 23:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'co2_in_ppm',
                'name': '',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 2,
                'server_name': 'ant-work',
            }
        ]
        notification = ModelsUtil.anomaly_diff(devs, cls, lat, lon, actual_time, w, 1000, 7200, 'co2')
        expected = {
            "data": {
                "actual_diff": 882.2,
                "anomaly": False,
                "min_anomaly_diff": 1000,
                "min_anomaly_time": 120,
                "type": "anomaly_co2"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": False,
            "readable": "2019-02-20 23:00:00",
            "timestamp": 1550700000
        }
        self.assertEqual(expected, notification)

    def test_0305_anomaly_diff_co2_notification_fail(self):
        cls = cu.setup_clients()

        actual_time = int(DateTimeUtil.local_time_str_to_utc('2019/10/20 03:00:00').timestamp())
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'co2_in_ppm',
                'name': '',
                'gateway_id': '1816820318180747',
                'device_id': '0xa900811026800001',
                'module_id': 2,
                'server_name': 'ant-work',
            }
        ]
        notification = ModelsUtil.anomaly_diff(devs, cls, lat, lon, actual_time, w, 2, 7200, 'co2')
        expected = {
            "data": {
                "actual_diff": None,
                "anomaly": None,
                "min_anomaly_diff": 2,
                "min_anomaly_time": 120,
                "type": "anomaly_co2"
            },
            "device_id": "0xa900811026800001",
            "event": "env-notification-pre",
            "gateway_id": "1816820318180747",
            "raise": False,
            "raise_catch": True,
            "readable": "2019-10-20 03:00:00",
            "timestamp": 1571533200
        }
        self.assertEqual(expected, notification)

    def test_0305_anomaly_diff_co2_notification_fail_2(self):
        cls = cu.setup_clients()

        actual_time = 1572855383
        lat = 49.1649894
        lon = 16.562262499999974

        w = WundergroundCom()
        w.api_key = cu.wunderground_api_key()

        devs = [
            {
                'db_column_name': 'co2_in_ppm',
                'name': '',
                'gateway_id': '1821675233441538',
                'device_id': '0xa9008110c8f715af',
                'module_id': 2,
                'server_name': 'rehivetech',
            }
        ]
        notification = ModelsUtil.anomaly_diff(devs, cls, lat, lon, actual_time, w, 2, 7200, 'co2')
        expected = {
            "data": {
                "actual_diff": None,
                "anomaly": None,
                "min_anomaly_diff": 2,
                "min_anomaly_time": 120,
                "type": "anomaly_co2"
            },
            "device_id": "0xa9008110c8f715af",
            "event": "env-notification-pre",
            "gateway_id": "1821675233441538",
            "raise": False,
            "raise_catch": True,
            "readable": "2019-11-04 09:16:23",
            "timestamp": 1572855383
        }
        self.assertEqual(expected, notification)


if __name__ == '__main__':
    unittest.main()
