"""Cached selector for one value from database.

Selector can select value based on given time from a table in database and
selected values are stored in cache.
"""
from dm.DateTimeUtil import DateTimeUtil
from dm.Storage import Storage
from dm.selectors.row.AbstractTableRowSelector import AbstractRowSelector

__author__ = 'Peter Tisovčík'
__email__ = 'xtisov00@stud.fit.vutbr.cz'


class SimpleCachedRowSelector(AbstractRowSelector):
    def row(self, column_name, time):
        """It selects one row from cache or database.

        If the method is called for the first time, cache will be initialized.
        The second call of the method returns loaded data stored in cache.

        :param column_name: name of column that contains required values
        :param time: timestamp of required data
        :return: data from the column in given time
        """
        if column_name not in self.cache:
            self.cache[column_name] = {}

        value = None
        if time in self.cache[column_name]:
            value = self.cache[column_name][time]
        else:
            res = Storage.one_row(self.con, self.table_name, column_name, time)

            if res is not None and res[0] is not None:
                self.cache[column_name][time] = float(res[0])
                value = float(res[0])

        if value is None:
            t = DateTimeUtil.utc_timestamp_to_str(time, '%Y/%m/%d %H:%M:%S')
            raise ValueError('empty value at %s' % t)

        return value
