"""Selector that gets data from server.
"""
import logging
from dm.selectors.AbstractSelector import AbstractSelector
from dm.ValueConversionUtil import ValueConversionUtil as vc

__author__ = ''
__email__ = ''

MAXIMUM_FEATURE_TIME = 60


class CachedDataFromServer(AbstractSelector):
    def __init__(self):
        super(CachedDataFromServer, self).__init__()

    def init_cache(self, measured, weather):
        """It initializes cache using measured data and weather data.

        :param measured: list of measured data
        :param weather: list of weather data
        :return: None
        """
        for item in measured + weather:
            at = None
            for key, value in item.items():
                if key == 'at' or key == 'measured_time':
                    at = value
                    continue

                if key not in self.cache:
                    self.cache[key] = {}

            for key, value in item.items():
                if key == 'at' or key == 'measured_time':
                    continue

                self.cache[key][at] = value

    def row(self, column_name, time):
        """It selects one row from cache.

        :param column_name: name of column that contains required values
        :param time: timestamp of required data
        :return: data from the column in given time
        """
        if column_name in self.cache:
            if time in self.cache[column_name]:
                return self.cache[column_name][time]

            maximum = max(self.cache[column_name])
            if time - maximum > MAXIMUM_FEATURE_TIME:
                raise ValueError('value in feature not found')

            if time > maximum:
                logging.debug('return {0} value from feature: {1}s'.format(column_name, time - maximum))
                return self.cache[column_name][maximum]

        if 'diff' in column_name:
            t_out = self.cache['temperature_out'][time]
            h_out = self.cache['humidity_out'][time]

            if column_name == 'temperature_in_celsius_diff':
                v_in = self.cache['temperature_in_celsius'][time]
                return v_in - t_out

            if column_name == 'temperature_in2_celsius_diff':
                v_in = self.cache['temperature_in2_celsius'][time]
                return v_in - t_out

            if column_name == 'rh_in_specific_g_kg_diff':
                v_in = self.cache['rh_in_specific_g_kg'][time]
                v_out = vc.rh_to_specific_g_kg(t_out, h_out)
                return v_in - v_out

            if column_name == 'rh_in2_specific_g_kg_diff':
                v_in = self.cache['rh_in2_specific_g_kg'][time]
                v_out = vc.rh_to_specific_g_kg(t_out, h_out)
                return v_in - v_out

            if column_name == 'rh_in_absolute_g_m3_diff':
                v_in = self.cache['rh_in_absolute_g_m3'][time]
                v_out = vc.rh_to_absolute_g_m3(t_out, h_out)
                return v_in - v_out

            if column_name == 'rh_in2_absolute_g_m3_diff':
                v_in = self.cache['rh_in2_absolute_g_m3'][time]
                v_out = vc.rh_to_absolute_g_m3(t_out, h_out)
                return v_in - v_out

            if column_name == 'co2_in_ppm_diff':
                if 'co2_in_ppm' in self.cache:
                    v_in = self.cache['co2_in_ppm'][time]
                elif 'co2_in' in self.cache:
                    v_in = self.cache['co2_in'][time]

                return v_in - 300

        raise ValueError(column_name + ' value not found')

    def interval(self, column_name, start, end):
        """It selects an interval from cache.

        :param column_name: name of column that contains required values
        :param start: timestamp that denotes start of the required interval
        :param end: timestamp that denotes end of the required interval
        :return: data from the column in given interval (start, end)
        """
        out = []

        for time in range(start, end):
            out.append(self.row(column_name, time))
        return out
