"""Creates adapted models for window opening detection.
"""
from pathlib import Path
import logging

from .create_attrs import func_co2, func_t_h
from dm.AttributeUtil import AttributeUtil
from dm.CSVUtil import CSVUtil
from dm.ConnectionUtil import ConnectionUtil as cu
from dm.Storage import Storage
from dm.PreProcessing import PreProcessing
from dm.selectors.from_server.CachedDataFromServer import CachedDataFromServer
from dm.DateTimeUtil import DateTimeUtil
from dm.models.open_detector.create_attrs import ColumnMapper

__author__ = 'Peter Tisovčík'
__email__ = 'xtisov00@stud.fit.vutbr.cz'


def training_data(json_f, cls, devs, model_type, no_event_time_shift, interval_extension,
                  mapper, func, weather):
    """It creates training dataset.

    The training dataset contains events when a window was open.

    :param json_f: dictionary that contains information about events
    :param cls: list of clients
    :param devs: list of devices
    :param model_type: type of model - based on CO2 concentration or temperature and humidity
    :param no_event_time_shift: number of seconds subtracted from start of event to define event when a window
                                was closed
    :param interval_extension: time shift that is subtracted or added to start or end of an interval respectively
    :param mapper: dictionary containing mapping of attribute name in
                   database to the name used in dataset
    :param func: object that determines function used for attribute calculation
    :param weather: object used to get information about weather
    :return: training dataset
    """
    cache_before = int(cu.open_detector('selector.cache.before'))
    cache_after = int(cu.open_detector('selector.cache.after'))

    input_json = []
    for item in json_f:
        start = item['e_start']['timestamp'] - cache_before
        start_dp_data = int(DateTimeUtil.local_time_str_to_utc('2018/09/20 01:00:00').timestamp())

        if start > start_dp_data:
            input_json.append(item)

    # for travis
    if cu.is_testable_system():
        json_f = input_json[:cu.MAX_TESTABLE_EVENTS]

    out = []
    for i in range(0, len(json_f)):
        start = json_f[i]['e_start']['timestamp'] - cache_before
        end = json_f[i]['e_start']['timestamp'] + cache_after

        start_dp_data = int(DateTimeUtil.local_time_str_to_utc('2018/09/20 01:00:00').timestamp())
        if start < start_dp_data:
            continue

        devices = None
        if model_type == 'co2':
            devices = devs['peto']
            middle = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 03:00:00').timestamp())
            if json_f[i]['e_start']['timestamp'] > middle:
                devices = devs['peto2']
        elif model_type == 't_h':
            devices = devs['klarka']

        suffix = ' ok'
        try:
            t = []

            selector = CachedDataFromServer()
            _, history = PreProcessing.prepare(cls, devices, start, end + 1, 0, interval_extension)
            if model_type == 'co2':
                history = PreProcessing.ppm_filter(history)

            w = weather.weather_by_coordinates(['humidity_out', 'temperature_out'],
                                               start, end, 49.1649894, 16.5622624)

            selector.init_cache(history, w)
            t += AttributeUtil.testing_one_row(func, json_f[i]['e_start']['timestamp'],
                                               selector, selector,
                                               'open', mapper, json_f[i]['e_start']['timestamp'])

            selector = CachedDataFromServer()
            start += no_event_time_shift
            end += no_event_time_shift
            _, history = PreProcessing.prepare(cls, devices, start, end + 1, 0, interval_extension)
            if model_type == 'co2':
                history = PreProcessing.ppm_filter(history)

            w = weather.weather_by_coordinates(['humidity_out', 'temperature_out'],
                                               start, end, 49.1649894, 16.5622624)

            selector.init_cache(history, w)
            t += AttributeUtil.testing_one_row(func, json_f[i]['e_start']['timestamp'] +
                                               no_event_time_shift, selector, selector,
                                               'nothing', mapper, None)
            out += t
        except Exception as e:
            suffix = ' error: ' + str(e)

        logging.debug('tr %d/%d: %s%s' % (i, len(json_f), json_f[i]['e_start']['readable'], suffix))
    return out


def add_func(events, cls, devs, model_type, interval_extension, mapper, func, weather):
    """It adds events to training dataset.

    The training dataset contains events when a window was closed.

    :param events: list of events
    :param cls: list of clients
    :param devs: list of devices
    :param model_type: type of model - based on CO2 concentration or temperature and humidity
    :param interval_extension: time shift that is subtracted or added to start or end of an interval respectively
    :param mapper: dictionary containing mapping of attribute name in
                   database to the name used in dataset
    :param func: object that determines function used for attribute calculation
    :param weather: object used to get information about weather
    :return: records containing information about given events
    """
    cache_before = int(cu.open_detector('selector.cache.before'))
    cache_after = int(cu.open_detector('selector.cache.after'))

    out = []
    for i in range(0, len(events)):
        ev = events[i]

        t = int(DateTimeUtil.local_time_str_to_utc(ev[0]).timestamp())
        start = t - cache_before
        end = t + cache_after

        devices = None
        if model_type == 'co2':
            devices = devs['peto']
            middle = int(DateTimeUtil.local_time_str_to_utc('2019/02/20 03:00:00').timestamp())
            if t > middle:
                devices = devs['peto2']
        elif model_type == 't_h':
            devices = devs['klarka']

        _, history = PreProcessing.prepare(cls, devices, start, end + 1, 0, interval_extension)
        if model_type == 'co2':
            history = PreProcessing.ppm_filter(history)

        w = weather.weather_by_coordinates(['humidity_out', 'temperature_out'],
                                           start, end, 49.1649894, 16.5622624)

        selector = CachedDataFromServer()
        selector.init_cache(history, w)

        suffix = ' ok'
        try:
            out += AttributeUtil.testing_one_row(func, t, selector, selector,
                                                 ev[1], mapper, None)
        except Exception as e:
            suffix = ' error'

        logging.debug('ad %d/%d: %s%s' % (i, len(events), ev[0], suffix))
    return out


def generic_training_file(events_file, no_event_time_shift, model_type,
                          columns_map, cls, devs, weather):
    """It creates generic training dataset.

    The training dataset contains the same number of events when a window was open and when it was closed.

    :param events_file: file containing events
    :param no_event_time_shift: number of seconds subtracted from start of event to define event when a window
                                was closed
    :param model_type: type of model - based on CO2 concentration or temperature and humidity
    :param columns_map: mapping of column names
    :param cls: list of clients
    :param devs: list of devices
    :param weather: object used to get information about weather
    :return: generic training dataset
    """
    logging.info('start')
    interval_extension = int(cu.open_detector('attrs.interval_extension'))

    p = Path(cu.open_detector('generic.directory'))
    if not p.is_dir():
        p.mkdir()
        logging.debug('created generic directory: ' + p.name)

    if model_type == 'co2':
        func = func_co2
        output_file = cu.open_detector('generic.co2.data_file.name')
        no_ev_records = ColumnMapper.NO_EVENTS_RECORDS_CO2
    elif model_type == 't_h':
        func = func_t_h
        output_file = cu.open_detector('generic.t_h.data_file.name')
        no_ev_records = ColumnMapper.NO_EVENTS_RECORDS_T_H
    else:
        raise ValueError('model type is not supported')

    # for travis
    if cu.is_testable_system():
        events_file = events_file[(cu.MAX_TESTABLE_EVENTS * (-1)):]
        no_ev_records = no_ev_records[(cu.MAX_TESTABLE_EVENTS * (-1)):]

    # stiahnutie dat
    storage = Storage(events_file, no_event_time_shift, '')
    data = training_data(storage.read_meta(), cls, devs, model_type, no_event_time_shift,
                         interval_extension, columns_map, func, weather)

    logging.info('downloaded events: %d' % len(events_file))
    logging.info('training set contains: %d records' % len(data))

    additional = add_func(no_ev_records, cls, devs, model_type, interval_extension,
                          columns_map, func, weather)
    logging.info('additional training set contains %d records' % len(additional))

    balanced = AttributeUtil.balance_set(data, additional)
    CSVUtil.create_csv_file(balanced, output_file)
    logging.info('end')

    return data
