"""Creates models for window opening detection.
"""
from distutils.util import strtobool
from pathlib import Path
import logging

from dm.AttributeUtil import AttributeUtil
from dm.ConnectionUtil import ConnectionUtil as cu
from dm.PreProcessing import PreProcessing
from dm.models.ModelsUtil import ModelsUtil
from dm.models.open_detector.create_attrs import func_co2, func_t_h, ColumnMapper
from dm.selectors.from_server.CachedDataFromServer import CachedDataFromServer

__author__ = ''
__email__ = ''


def filter_only_open(values, attr_name='value'):
    """It gets events when a window was open.

    :param values: dictionary of values
    :param attr_name: name of attribute
    :return: list of events when a window was open
    """
    out = []

    for item in values:
        if int(float(item[attr_name])) == 1:
            out.append(item)

    return out


def prepare_adapted_data_co2(devs, cls, start, end, lat, lon, weather):
    """It creates a model on the basis of CO2 concentration.

    :param devs: list of devices
    :param cls: list of clients
    :param start: timestamp that denotes start of required time interval
    :param end: timestamp that denotes end of required time interval
    :param lat: latitude where data was gathered
    :param lon: longitude where data was gathered
    :param weather: object used to get information about weather
    :return: model on the basis of CO2 concentration
    """
    model_data = []

    dev_co2 = devs[1]
    dev_open_close = devs[0]

    cache_before = int(cu.open_detector('selector.cache.before'))
    cache_after = int(cu.open_detector('selector.cache.after'))

    no_ev_shift = int(cu.open_detector('attrs.no_event.time_shift'))

    s = start
    e = end

    for t in range(s, e, 24 * 60 * 60):
        start = t - cache_before
        end = t + cache_after

        open_close = PreProcessing.download_data(cls, [dev_open_close], start, end + 1)
        open_close = filter_only_open(open_close[0], 'value')

        for event in open_close:
            interval_extension = int(cu.open_detector('attrs.interval_extension'))

            try:
                # nothing
                _, history = PreProcessing.prepare(cls, [dev_co2],
                                                   event['at'] - cache_before + no_ev_shift,
                                                   event['at'] + cache_after + no_ev_shift + 1,
                                                   0, interval_extension)

                w = weather.weather_by_coordinates(['humidity_out', 'temperature_out'],
                                                   event['at'] - cache_before + no_ev_shift,
                                                   event['at'] + cache_after + no_ev_shift + 1,
                                                   lat, lon)

                selector = CachedDataFromServer()
                selector.init_cache(history, w)
                record = AttributeUtil.testing_one_row(func_co2,
                                                       event['at'] + no_ev_shift,
                                                       selector, selector,
                                                       'nothing', ColumnMapper.OPEN_CO2, None)
                # open
                _, history = PreProcessing.prepare(cls, [dev_co2],
                                                   event['at'] - cache_before,
                                                   event['at'] + cache_after + 1,
                                                   0, interval_extension)

                w = weather.weather_by_coordinates(['humidity_out', 'temperature_out'],
                                                   start, end, lat, lon)

                selector = CachedDataFromServer()
                selector.init_cache(history, w)

                record += AttributeUtil.testing_one_row(func_co2, event['at'],
                                                        selector, selector,
                                                        'open', ColumnMapper.OPEN_CO2, None)

                model_data += record
            except Exception as e:
                logging.debug(e)
                # logging.exception(e)

    directory = cu.open_detector('adapted.directory')
    filename = 'co2_{0}_{1}.bin'.format(dev_co2['gateway_id'], dev_co2['device_id'],)

    p = Path(directory)
    if not p.is_dir():
        p.mkdir()

    combined = cu.open_detector('adapted.combined_with_generic_data')
    if strtobool(combined):
        model_data += ModelsUtil.read_generic_data('co2')

    output_filename = '{0}/{1}'.format(directory, filename)

    msg = 'created co2 model for GW: {0}, Dev: {1} '.format(dev_co2['gateway_id'],
                                                            dev_co2['device_id'])
    msg += 'from {0} records (open + nothing) '.format(len(model_data))
    msg += 'file: {0}'.format(output_filename)
    logging.debug(msg)

    return ModelsUtil.write_model(model_data, output_filename, ModelsUtil.replace_nothing_open)


def prepare_adapted_data_t_h(devs, cls, start, end, lat, lon, weather):
    """It creates a model on the basis of temperature and humidity.

    :param devs: list of devices
    :param cls: list of clients
    :param start: timestamp that denotes start of required time interval
    :param end: timestamp that denotes end of required time interval
    :param lat: latitude where data was gathered
    :param lon: longitude where data was gathered
    :param weather: object used to get information about weather
    :return: model on the basis of temperature and humidity
    """
    model_data = []

    dev_open_close = devs[0]
    dev_t = devs[1]
    dev_h = devs[2]

    cache_before = int(cu.open_detector('selector.cache.before'))
    cache_after = int(cu.open_detector('selector.cache.after'))

    no_ev_shift = int(cu.open_detector('attrs.no_event.time_shift'))

    for t in range(start, end, 24 * 60 * 60):
        start = t - cache_before
        end = t + cache_after

        open_close = PreProcessing.download_data(cls, [dev_open_close], start, end + 1)
        open_close = filter_only_open(open_close[0], 'value')

        for event in open_close:
            interval_extension = int(cu.open_detector('attrs.interval_extension'))

            try:
                # nothing
                _, history = PreProcessing.prepare(cls, [dev_t, dev_h],
                                                   event['at'] - cache_before + no_ev_shift,
                                                   event['at'] + cache_after + no_ev_shift + 1,
                                                   0, interval_extension)

                w = weather.weather_by_coordinates(['humidity_out', 'temperature_out'],
                                                   start + no_ev_shift, end + no_ev_shift,
                                                   lat, lon)

                selector = CachedDataFromServer()
                selector.init_cache(history, w)

                record = AttributeUtil.testing_one_row(func_t_h,
                                                       event['at'] + no_ev_shift,
                                                       selector, selector,
                                                       'nothing', ColumnMapper.OPEN_T_H, None)

                # open
                _, history = PreProcessing.prepare(cls, [dev_t, dev_h],
                                                   event['at'] - cache_before,
                                                   event['at'] + cache_after + 1,
                                                   0, interval_extension)

                w = weather.weather_by_coordinates(['humidity_out', 'temperature_out'],
                                                   start, end, lat, lon)

                selector = CachedDataFromServer()
                selector.init_cache(history, w)

                record += AttributeUtil.testing_one_row(func_t_h, event['at'],
                                                        selector, selector,
                                                        'open', ColumnMapper.OPEN_T_H, None)

                model_data += record
            except Exception as e:
                logging.debug(e)
                logging.exception(e)

    directory = cu.open_detector('adapted.directory')
    filename = 't_h_{0}_{1}.bin'.format(dev_t['gateway_id'], dev_t['device_id'],)

    p = Path(directory)
    if not p.is_dir():
        p.mkdir()

    combined = cu.open_detector('adapted.combined_with_generic_data')
    if strtobool(combined):
        model_data += ModelsUtil.read_generic_data('co2')

    output_filename = '{0}/{1}'.format(directory, filename)

    msg = 'created t_h model for GW: {0}, Dev: {1} '.format(dev_t['gateway_id'],
                                                            dev_t['device_id'])
    msg += 'from {0} records (open + nothing)'.format(len(model_data))
    msg += 'file: {0}'.format(output_filename)
    logging.debug(msg)

    return ModelsUtil.write_model(model_data, output_filename, ModelsUtil.replace_nothing_open)
