"""Estimates course of quantities.
"""
from dm.ValueConversionUtil import ValueConversionUtil as conv
from scipy import stats

__author__ = ''
__email__ = ''


class Estimate:
    @staticmethod
    def compute_min_win_size(last_close_event_time, actual_time):
        """It computes minimal window size.

        :param last_close_event_time: last time when window was closed
        :param actual_time: current time
        :return: minimal window size
        """
        return actual_time - last_close_event_time

    @staticmethod
    def compute_lin_reg(values):
        """It computes linear regression from given values.

        :param values: list of values
        :return: pair of intercept and slope
        """
        x = []
        y = []

        for i in range(0, len(values)):
            x.append(i)
            y.append(values[i])

        slope, intercept, r_value, p_value, std_err = stats.linregress(x, y)

        return intercept, slope

    @staticmethod
    def co2_level(co2_value):
        """It determines level of CO2 concentration.

        :param co2_value: value of CO2
        :return: level of CO2 concentration and order of level
        """
        co2_level = None
        level = None

        if co2_value < 500:
            co2_level = 500
            level = 0
        elif co2_value < 1500:
            co2_level = 1500
            level = 1
        elif co2_value < 2500:
            co2_level = 2500
            level = 2
        elif co2_value < 4000:
            co2_level = 4000
            level = 3
        else:
            level = 4

        return co2_level, level

    @staticmethod
    def t_h_level(temp, rh):
        """It determines level of humidity.

        :param temp: value of temperature
        :param rh: value of humidity
        :return: level of absolute humidity, level of relative humidity and order of level
        """
        t_h_level = None
        t_h_level_rh = None
        level = None

        if rh < 30:
            t_h_level = conv.rh_to_absolute_g_m3(temp, 30)
            t_h_level_rh = 30
            level = 0
        elif rh < 40:
            t_h_level = conv.rh_to_absolute_g_m3(temp, 40)
            t_h_level_rh = 40
            level = 1
        elif rh < 60:
            t_h_level = conv.rh_to_absolute_g_m3(temp, 60)
            t_h_level_rh = 60
            level = 2
        elif rh < 70:
            t_h_level = conv.rh_to_absolute_g_m3(temp, 70)
            t_h_level_rh = 70
            level = 3
        else:
            t_h_level = conv.rh_to_absolute_g_m3(temp, 100)
            t_h_level_rh = 100
            level = 4

        return t_h_level, t_h_level_rh, level
