"""
"""
from collections import OrderedDict
from pathlib import Path
from sklearn.svm import SVC
import json
import csv
import logging
import pandas as pd
import pickle
import math

from dm.AttributeUtil import AttributeUtil
from dm.CSVUtil import CSVUtil
from dm.ConnectionUtil import ConnectionUtil
from dm.PreProcessing import PreProcessing
from dm.models.open_detector.create_attrs import ColumnMapper
from dm.models.open_detector.create_attrs import func_co2, func_t_h
from dm.models.open_detector.create_attrs import func_predict_t_h, func_predict_co2
from dm.selectors.from_server.CachedDataFromServer import CachedDataFromServer
from dm.models.estimate.Estimate import Estimate
from dm.DateTimeUtil import DateTimeUtil
from dm.ValueConversionUtil import ValueConversionUtil as conv
from dm.coefficients.CenterLineSlope import CenterLineSlope
from dm.ValueConversionUtil import ValueConversionUtil
import math
from dm.co2regression.SimpleExpRegression import SimpleExpRegression

from dm.models.predictor.THPredictorUtil import training_testing_data
from dm.models.predictor.THPredictorUtil import training_testing_data_with_distance
from dm.models.predictor.THPredictorUtil import training_testing_data_only_distance
from dm.models.predictor.THPredictorUtil import training_testing_data_without_distance


__author__ = ''
__email__ = ''


class ModelsUtil:
    @staticmethod
    def json_to_file(json_data, filename, log_notification):
        """It saves JSON structure into the file.

        :param json_data: JSON structure containing data
        :param filename: name of output file
        :param log_notification: true if notification is logged
        :return: None
        """
        with open(filename, 'w') as f:
            json.dump(json_data, f, indent=4, sort_keys=True)

        if log_notification:
            logging.debug(json.dumps(json_data, indent=4, sort_keys=True))

    @staticmethod
    def replace_nothing_open(data):
        """It converts value of event attribute to boolean.

        Attribute value open is represented by 1 and close by 0.

        :param data: list of data
        :return: modified list of data
        """
        for i in range(0, len(data)):
            if data[i]['event'] == 'open':
                data[i]['event'] = 1
                continue
            data[i]['event'] = 0

        return data

    @staticmethod
    def replace_ventilation_length(data):
        """It removes quotation marks in value of ventilation length attribute.

        :param data: list of data
        :return: modified list of data
        """
        for i in range(0, len(data)):
            data[i]['event'] = data[i]['VentilationLength_event__'].replace('"', '')
            del data[i]['VentilationLength_event__']

        return data

    @staticmethod
    def replace_co2_ventilation_len(data):
        """It moves value of CO2 concentration after regression to event attribute.

        :param data: list of data
        :return: modified list of data
        """

        for i in range(0, len(data)):
            data[i]['event'] = data[i]['Regression_co2_in_before_0']
            del data[i]['Regression_co2_in_before_0']

        return data

    @staticmethod
    def write_model(attrs, filename, replace, event_column='event'):
        """It saves model to the output file.

        :param attrs: list of attributes
        :param filename: name of output file
        :param replace: function that replaces an attribute
        :param event_column: name of attribute that denotes class
        :return: None
        """
        attrs = replace(attrs)
        data = pd.DataFrame(attrs)

        # pre-processing of training set
        X = data.drop('event', axis=1)
        X = X.drop('datetime', axis=1)
        y = data[event_column]

        # training phase (SVM - Gaussian Kernel)
        svclassifier = SVC(kernel='rbf', gamma='scale')
        svclassifier.fit(X, y)

        with open(filename, 'wb') as f:
            model = pickle.dumps(svclassifier)
            f.write(model)
            f.close()

    @staticmethod
    def load_model(type, device):
        """It loads model for window opening detector from file.

        :param type: type of model, t_h or co2
        :param device: information about device
        :return: loaded model
        """
        file = '{0}/{1}_{2}_{3}.bin'.format(ConnectionUtil.open_detector('adapted.directory'),
                                            type, device['gateway_id'], device['device_id'])
        if not Path(file).exists():
            generic_key = 'generic.{0}.model.name'.format(type)
            file = ConnectionUtil.open_detector(generic_key)

        logging.debug('load model from: ' + file)
        return pickle.load(open(file, 'rb'))

    @staticmethod
    def load_model_predictor(type, device):
        """It loads model for predictor from file.

        :param type: type of model, t_h or co2
        :param device: information about device
        :return: loaded model
        """
        file = '{0}/{1}_{2}_{3}.bin'.format(ConnectionUtil.predictor('adapted.directory'),
                                            type, device['gateway_id'], device['device_id'])
        if not Path(file).exists():
            generic_key = 'generic.{0}.model.name'.format(type)
            file = ConnectionUtil.predictor(generic_key)

        logging.debug('load model from: ' + file)
        return pickle.load(open(file, 'rb'))

    @staticmethod
    def read_generic_data(model_type):
        """It gets list of attributes from file.

        The attributes are used in model for window opening detector.

        :param model_type: type of model, t_h or co2
        :return: list of attributes
        """
        if model_type == 'co2':
            output_file = ConnectionUtil.open_detector('generic.co2.data_file.name')
        elif model_type == 't_h':
            output_file = ConnectionUtil.open_detector('generic.t_h.data_file.name')
        else:
            raise ValueError('model type is not supported')

        attrs = []
        with open(str(output_file), 'r') as f:
            csv_reader = csv.DictReader(f, delimiter=',')
            for row in csv_reader:
                attrs.append(OrderedDict(row))
        return attrs

    @staticmethod
    def estimate_open_co2(device, cls, lat, lon, actual_time, weather):
        """It estimates if window was open on the basis of CO2 concentration.

        :param device: information about device
        :param cls: list of clients
        :param lat: latitude of locality
        :param lon: longitude of locality
        :param actual_time: current timestamp
        :param weather: class that obtains information about weather
        :return: notification that contains information about estimation
        """
        def g_notification(dev, estimate_open, current_time, enable_notification, raise_catch):
            return {
                'data': {
                    'estimate_open': estimate_open,
                    'type': 'co2_open',
                },
                'device_id': dev['device_id'],
                'event': 'env-notification-pre',
                'gateway_id': dev['gateway_id'],
                'raise': enable_notification,
                'raise_catch': raise_catch,
                'readable': DateTimeUtil.utc_timestamp_to_str(current_time),
                'timestamp': current_time,
            }

        start = actual_time - int(ConnectionUtil.open_detector('selector.cache.before'))
        end = actual_time + int(ConnectionUtil.open_detector('selector.cache.after'))

        w = weather.weather_by_coordinates(['humidity_out', 'temperature_out'],
                                           start, end, lat, lon)

        interval_extension = int(ConnectionUtil.open_detector('attrs.interval_extension'))
        _, history = PreProcessing.prepare(cls, device, start, end + 1, 0, interval_extension)

        for item in history:
            if item['co2_in_ppm'] is None:
                logging.error('too small amount of data for co2 detector')
                return g_notification(device[0], None, actual_time, False, True)

        selector = CachedDataFromServer()
        selector.init_cache(history, w)

        delay = int(ConnectionUtil.open_detector('co2.detection.delay'))
        res = AttributeUtil.testing_one_row(func_co2, actual_time - delay,
                                            selector, selector,
                                            '1', ColumnMapper.OPEN_CO2, None)

        io_res = CSVUtil.create_csv_io(res)

        testing_set = pd.read_csv(io_res)
        testing_set = testing_set.drop('datetime', axis=1)
        testing_set = testing_set.drop('event', axis=1)
        model = ModelsUtil.load_model('co2', device[0])

        return g_notification(
            device[0],
            bool(model.predict(testing_set)[0]),
            actual_time,
            True,
            False,
        )

    @staticmethod
    def estimate_open_t_h(device, cls, lat, lon, actual_time, weather):
        """It estimates if window was open on the basis of temperature and humidity.

        :param device: information about device
        :param cls: list of clients
        :param lat: latitude of locality
        :param lon: longitude of locality
        :param actual_time: current timestamp
        :param weather: class that obtains information about weather
        :return: notification that contains information about estimation
        """
        def g_notification(dev, estimate_open, current_time, enable_notification, raise_catch):
            return {
                'data': {
                    'estimate_open': estimate_open,
                    'type': 't_h_open',
                },
                'device_id': dev['device_id'],
                'event': 'env-notification-pre',
                'gateway_id': dev['gateway_id'],
                'raise': enable_notification,
                'raise_catch': raise_catch,
                'readable': DateTimeUtil.utc_timestamp_to_str(current_time),
                'timestamp': current_time,
            }

        start = actual_time - int(ConnectionUtil.open_detector('selector.cache.before'))
        end = actual_time + int(ConnectionUtil.open_detector('selector.cache.after'))

        w = weather.weather_by_coordinates(['humidity_out', 'temperature_out'],
                                           start, end, lat, lon)

        interval_extension = int(ConnectionUtil.open_detector('attrs.interval_extension'))
        _, history = PreProcessing.prepare(cls, device, start, end + 1, 0, interval_extension)

        for item in history:
            if item['temperature_in2_celsius'] is None:
                logging.error('too small amount of data for t_h detector')
                return g_notification(device[0], None, actual_time, False, True)

        selector = CachedDataFromServer()
        selector.init_cache(history, w)

        delay = int(ConnectionUtil.open_detector('t_h.detection.delay'))
        res = AttributeUtil.testing_one_row(func_t_h, actual_time - delay,
                                            selector, selector,
                                            '1', ColumnMapper.OPEN_T_H, None)
        io_res = CSVUtil.create_csv_io(res)

        testing_set = pd.read_csv(io_res)
        testing_set = testing_set.drop('datetime', axis=1)
        testing_set = testing_set.drop('event', axis=1)
        model = ModelsUtil.load_model('t_h', device[0])

        return g_notification(
            device[0],
            bool(model.predict(testing_set)[0]),
            actual_time,
            True,
            False,
        )

    @staticmethod
    def estimate_co2(device, cls, actual_time):
        """It estimates values in given time on the basis of CO2 concentration.

        :param device: information about device
        :param cls: list of clients
        :param actual_time: current timestamp
        :return: notification that contains information about estimation
        """
        def g_notification(dev, estimate_time, es_level, max_es_time, current_value, ntf_level, lev_value,
                           current_time, enable_notification, raise_catch):
            return {
                'data': {
                    'current_value': None if current_value is None else int(current_value),
                    'es_level': es_level,
                    'es_time': max_es_time,
                    'estimate_time': estimate_time,
                    'level': ntf_level,
                    'level_value': lev_value,
                    'type': 'co2_estimate',
                },
                'device_id': dev['device_id'],
                'event': 'env-notification-pre',
                'gateway_id': dev['gateway_id'],
                'raise': enable_notification,
                'raise_catch': raise_catch,
                'readable': DateTimeUtil.utc_timestamp_to_str(current_time),
                'timestamp': current_time,
            }

        win_size = int(ConnectionUtil.estimate('co2.windows.size'))
        interval_extension = int(ConnectionUtil.estimate('attrs.interval_extension'))
        max_es_time = int(ConnectionUtil.estimate('estimate.time'))

        start = actual_time - win_size
        end = actual_time + 1
        _, history = PreProcessing.prepare(cls, device, start, end, 0, interval_extension)
        values, times = PreProcessing.extract_items(history, 'co2_in_ppm')

        if len(values) < win_size // 2:
            logging.warning('window contain %s values from %s required' %
                            (len(values), win_size // 2))
            return None

        for item in history:
            if item['co2_in_ppm'] is None:
                logging.error('too small amount of data for co2 detector')
                return g_notification(device[0], None, None, int(max_es_time / 60), None, None, None, actual_time, False, True)

        intercept, slope = Estimate.compute_lin_reg(values)
        last_value = values[-1]
        co2_level, ntf_level = Estimate.co2_level(last_value)

        es_time = (co2_level - intercept) // slope
        es_level = int(slope * max_es_time + intercept)

        if not math.isnan(es_time):
            logging.debug('last co2 level: ' + str(last_value))
            logging.debug("estimation time %s (%s) for co2 level %s" %
                          (es_time, DateTimeUtil.utc_timestamp_to_str(actual_time + es_time),
                           co2_level))
            logging.debug("estimated co2 level after %d seconds: %s ppm" %
                          (max_es_time, es_level))

        return g_notification(
            device[0],
            0 if (es_time is None or math.isnan(es_time)) else int(es_time / 60),
            es_level,
            int(max_es_time / 60),
            last_value,
            ntf_level,
            co2_level,
            actual_time,
            bool(es_time is not None and es_time <= max_es_time),
            False
        )

    @staticmethod
    def estimate_t_h(device, cls, actual_time):
        """It estimates values in given time on the basis of humidity.

        :param device: information about device
        :param cls: list of clients
        :param actual_time: current timestamp
        :return: notification that contains information about estimation
        """
        def g_notification(dev, estimate_time, es_level, max_es_time, current_value, ntf_level, level_value,
                           current_time, enable_notification, raise_catch):
            return {
                'data': {
                    'current_value': None if current_value is None else int(current_value),
                    'es_level': es_level,
                    'es_time': max_es_time,
                    'estimate_time': estimate_time,
                    'level': ntf_level,
                    'level_value': level_value,
                    'type': 'co2_estimate',
                },
                'device_id': dev['device_id'],
                'event': 'env-notification-pre',
                'gateway_id': dev['gateway_id'],
                'raise': enable_notification,
                'raise_catch': raise_catch,
                'readable': DateTimeUtil.utc_timestamp_to_str(current_time),
                'timestamp': current_time,
            }

        win_size = int(ConnectionUtil.estimate('t_h.windows.size'))
        interval_extension = int(ConnectionUtil.estimate('attrs.interval_extension'))
        max_es_time = int(ConnectionUtil.estimate('estimate.time'))

        start = actual_time - win_size
        end = actual_time + 1
        _, history = PreProcessing.prepare(cls, device, start, end, 0, interval_extension)
        for item in history:
            if 'rh_in2_absolute_g_m3' not in item:
                logging.error('\'rh_in2_absolute_g_m3\' not in history')
                return g_notification(device[0], None, None, int(max_es_time / 60), None, None, None,
                                      actual_time, False, True)

        values, times = PreProcessing.extract_items(history, 'rh_in2_absolute_g_m3')
        values_rh, _ = PreProcessing.extract_items(history, 'rh_in2_percentage')
        values_temp, _ = PreProcessing.extract_items(history, 'temperature_in2_celsius')

        if len(values) < win_size // 2:
            logging.warning('window contains %s values from %s required' %
                            (len(values), win_size // 2))
            return None

        for item in history:
            if item['temperature_in2_celsius'] is None:
                logging.error('too small amount of data for co2 detector')
                return g_notification(device[0], None, None, int(max_es_time / 60), None, None, None,
                                      actual_time, False, True)

        intercept, slope = Estimate.compute_lin_reg(values)
        last_value = round(values[-1], 2)
        t_h_level, t_h_level_rh, ntf_level = Estimate.t_h_level(values_temp[-1], values_rh[-1])

        # easy hack if: current RH < intercent
        if intercept > t_h_level:
            t_h_level, t_h_level_rh, ntf_level = Estimate.t_h_level(values_temp[-1], values_rh[-1]+5)

        es_time = (t_h_level - intercept) // slope
        es_level = round(slope * max_es_time + intercept, 2)
        es_level_rh = int(conv.ah_to_relative_percent(values_temp[-1], es_level))

        logging.debug('last abs hum level: %s (rel hum: %s %%)' % (str(last_value), str(round(values_rh[-1], 2))))
        logging.debug("estimation time %s (%s) for abs hum (rel hum) level %s (%s %%)" %
                      (es_time, DateTimeUtil.utc_timestamp_to_str(actual_time + es_time),
                       round(t_h_level, 2), round(t_h_level_rh, 2)))
        logging.debug("estimated hum level after %d seconds: %s (%s %%)" %
                      (max_es_time, es_level, es_level_rh))

        return g_notification(
            device[0],
            0 if (es_time is None or math.isnan(es_time) or es_time < 0) else int(es_time / 60),
            es_level_rh,
            int(max_es_time / 60),
            values_rh[-1],
            ntf_level,
            t_h_level_rh,
            actual_time,
            bool(es_time is not None and 0 < es_time <= max_es_time),
            False
        )

    @staticmethod
    def predictor(device, cls, lat, lon, actual_time, module_type, weather, decrease=None):
        """It predicts length of ventilation to decrease quantity to required level.

        :param device: information about device
        :param cls: list of clients
        :param lat: latitude of locality
        :param lon: longitude of locality
        :param actual_time: current timestamp
        :param module_type: type of module, t_h or CO2
        :param weather: class that obtains information about weather
        :param decrease: value that defines how much quantity value should be decreased
        :return: notification that contains information about estimation
        """
        def g_notification(dev, estimate_time, current_value, decrease,
                           current_time, enable_notification, raise_catch):
            notification = {
                'data': {
                    'current_value': None if current_value is None else int(current_value),
                    'estimate_time': estimate_time,
                    'type': module_type + '_predictor',
                },
                'device_id': dev['device_id'],
                'event': 'env-notification-pre',
                'gateway_id': dev['gateway_id'],
                'raise': enable_notification,
                'raise_catch': raise_catch,
                'readable': DateTimeUtil.utc_timestamp_to_str(current_time),
                'timestamp': current_time,
            }

            if decrease is not None:
                notification['data']['final_value'] = decrease

            if module_type == 'co2':
                notification['data']['final_value'] = 400

            return notification

        interval_extension = int(ConnectionUtil.estimate('attrs.interval_extension'))
        start = actual_time - int(ConnectionUtil.predictor('selector.cache.before'))
        end = actual_time

        if module_type == 'co2':
            func = func_predict_co2
            mapper = ColumnMapper.PREDICTOR_CO2
        elif module_type == 't_h':
            func = func_predict_t_h
            mapper = ColumnMapper.PREDICTOR_T_H
        else:
            raise ValueError('invalid module type: ' + module_type)

        w = weather.weather_by_coordinates(['humidity_out', 'temperature_out'],
                                           start, end, lat, lon)

        _, history = PreProcessing.prepare(cls, device, start, end + 1, 0, interval_extension)

        is_none = False
        if module_type == 'co2':
            for item in history:
                if item['co2_in_ppm'] is None:
                    is_none = True
                    break

        if module_type == 't_h':
            for item in history:
                if item['temperature_in2_celsius'] is None:
                    is_none = True
                    break

        if is_none:
            logging.error('too small amount of data for ' + module_type + ' predictor')
            return g_notification(device[0], None, None, decrease, actual_time, False, True)

        selector = CachedDataFromServer()
        selector.init_cache(history, w)

        res = AttributeUtil.testing_one_row(func, actual_time, selector, selector,
                                            '1', mapper, None)

        output = None
        if module_type == 't_h':
            training = []
            with open(ConnectionUtil.predictor('generic.t_h.raw_csv.training_data'), 'r') as f:
                csv_reader = csv.DictReader(f, delimiter=',')
                for row in csv_reader:
                    training.append(row)

            res[0]['InLinear_rh_in_specific_after_1200'] = ValueConversionUtil.rh_to_specific_g_kg(
                history[-1]['temperature_in2_celsius'], decrease)

            _, res = training_testing_data_with_distance(training, res,
                          CenterLineSlope(), 'center_', False, False, False, False,
                          None, None)

        io_res = CSVUtil.create_csv_io(res)

        testing_set = pd.read_csv(io_res)
        testing_set = testing_set.drop('datetime', axis=1)
        testing_set = testing_set.drop('event', axis=1)
        model = ModelsUtil.load_model_predictor(module_type, device[0])

        if module_type == 't_h':
            output = int(model.predict(testing_set)[0].replace('\'', ''))
            output = math.ceil(output/60)
            current_val = history[-1]['rh_in2_percentage']
        elif module_type == 'co2':
            output = model.predict(testing_set)[0]/3600
            tmp = (history[-1]['co2_in_ppm'] - 400)
            tmp *= math.e**(-output * 1200)
            tmp += 400
            output = math.ceil(tmp / 60)
            current_val = history[-1]['co2_in_ppm']

        return g_notification(device[0], output, current_val, decrease, actual_time, True, False)

    @staticmethod
    def when_ventilate_summer(devs, cls, lat, lon, actual_time, weather, temperature_diff):
        """It determines if ventilation should be performed on the basis of temperature difference.

        :param devs: information about device
        :param cls: list of clients
        :param lat: latitude of locality
        :param lon: longitude of locality
        :param actual_time: current timestamp
        :param weather: class that obtains information about weather
        :param temperature_diff: difference between inside and outside temperature
        :return: notification that contains information about estimation
        """
        def g_notification(dev, v_ventilate, v_temp_in, v_temp_out,
                           current_time, enable_notification, raise_catch):
            return {
                'data': {
                    'temp_in': v_temp_in,
                    'temp_out': v_temp_out,
                    'type': 'when_ventilate_summer',
                    'ventilate': v_ventilate,
                },
                'device_id': dev['device_id'],
                'event': 'env-notification-pre',
                'gateway_id': dev['gateway_id'],
                'raise': enable_notification,
                'raise_catch': raise_catch,
                'readable': DateTimeUtil.utc_timestamp_to_str(current_time),
                'timestamp': current_time,
            }

        start = actual_time - 1
        end = actual_time
        interval_extension = int(ConnectionUtil.package('interval_extension'))

        w = weather.weather_by_coordinates(['humidity_out', 'temperature_out'],
                                           start, end, lat, lon)

        _, history = PreProcessing.prepare(cls, devs, start, end, 0, interval_extension)
        for item in history:
            if item['temperature_in_celsius'] is None:
                logging.error('too small amount of data for co2 detector')
                return g_notification(devs[0], None, None, None, actual_time, False, True)

        selector = CachedDataFromServer()
        selector.init_cache(history, w)

        temp_in = selector.row('temperature_in_celsius', actual_time)
        temp_out = selector.row('temperature_out', actual_time)
        diff = temp_in - temp_out

        if math.fabs(diff) < temperature_diff:
            ventilate = 1
        elif diff < 0:
            ventilate = 0
        else:
            ventilate = 2

        return g_notification(devs[0], ventilate, temp_in, temp_out, actual_time, True, False)

    @staticmethod
    def dew_point(devs, cls, lat, lon, actual_time, weather):
        """It calculates dew point.

        :param devs: information about device
        :param cls: list of clients
        :param lat: latitude of locality
        :param lon: longitude of locality
        :param actual_time: current timestamp
        :param weather: class that obtains information about weather
        :return: notification that contains information about dew point
        """
        # https://www.planetaoken.cz/clanek/rosny-bod-a-jeho-vypocet-pro-eliminaci-tvorby-plisni/
        def g_notification(dev, tau, v_temp_in, v_hum_in, dewing,
                           current_time, enable_notification, raise_catch):
            return {
                'data': {
                    'dew_point': tau,
                    'temp_in': v_temp_in,
                    'hum_in': v_hum_in,
                    'dewing': dewing,
                    'type': 'dew_point',
                },
                'device_id': dev['device_id'],
                'event': 'env-notification-pre',
                'gateway_id': dev['gateway_id'],
                'raise': enable_notification,
                'raise_catch': raise_catch,
                'readable': DateTimeUtil.utc_timestamp_to_str(current_time),
                'timestamp': current_time,
            }

        start = actual_time - 1
        end = actual_time
        interval_extension = int(ConnectionUtil.package('interval_extension'))

        try:
            w = weather.weather_by_coordinates(['humidity_out', 'temperature_out'],
                                               start, end, lat, lon)
        except Exception as e:
            logging.error('too small amount of data for co2 detector')
            return g_notification(devs[0], None, None, None, None, actual_time, False, True)

        _, history = PreProcessing.prepare(cls, devs, start, end, 0, interval_extension)
        for item in history:
            if item['temperature_in_celsius'] is None:
                logging.error('too small amount of data for co2 detector')
                return g_notification(devs[0], None, None, None, None, actual_time, False, True)

        selector = CachedDataFromServer()
        selector.init_cache(history, w)

        temp_in = selector.row('temperature_in_celsius', actual_time)
        hum_in = selector.row('rh_in_percentage', actual_time)

        ai = 7.45
        bi = 235

        t = temp_in * 1
        f = hum_in * 1
        z1 = (ai * t) / (bi + t)
        es = 6.1 * math.exp(z1 * 2.3025851)
        e = es * f / 100
        z2 = e / 6.1
        z3 = 0.434292289 * math.log(z2)
        dru = e * 100
        dru = math.floor(dru) / 100
        tau = (235 * z3) / (7.45 - z3) * 100
        tau = math.floor(tau) / 100
        feu = (216.7 * e) / (273.15 + t) * 100
        feu = round(feu, 0) / 100

        return g_notification(devs[0], tau, t, f, None, actual_time, True, False)

    @staticmethod
    def anomaly_diff(devs, cls, lat, lon, actual_time, weather, min_diff, diff_time, a_type):
        """It detect an anomaly within given time interval.

        :param devs: information about device
        :param cls: list of clients
        :param lat: latitude of locality
        :param lon: longitude of locality
        :param actual_time: current timestamp
        :param weather: class that obtains information about weather
        :param min_diff: minimal difference between measured values
        :param diff_time: window size for anomaly detection
        :param a_type: type of anomaly detection (temperature, humidity or co2)
        :return: notification that contains information about anomaly
        """
        def g_notification(dev, v_diff, v_anomaly, min_diff, min_time,
                           current_time, enable_notification, raise_catch):
            return {
                'data': {
                    'actual_diff': v_diff,
                    'anomaly': v_anomaly,
                    'min_anomaly_diff': min_diff,
                    'min_anomaly_time': min_time if min_time is None else int(min_time/60),
                    'type': 'anomaly_{}'.format(a_type),
                },
                'device_id': dev['device_id'],
                'event': 'env-notification-pre',
                'gateway_id': dev['gateway_id'],
                'raise': False if v_anomaly is None else v_anomaly,
                'raise_catch': raise_catch,
                'readable': DateTimeUtil.utc_timestamp_to_str(current_time),
                'timestamp': current_time,
            }

        start = actual_time - 1 - diff_time
        end = actual_time
        interval_extension = int(ConnectionUtil.package('interval_extension'))

        _, history = PreProcessing.prepare(cls, devs, start, end, 0, interval_extension)
        selector = CachedDataFromServer()
        selector.init_cache(history, [])

        if a_type == 'temperature':
            start_value = selector.row('temperature_in_celsius', actual_time - diff_time)
            end_value = selector.row('temperature_in_celsius', actual_time)
            for item in history:
                if item['temperature_in_celsius'] is None:
                    logging.error('too small amount of data for co2 detector')
                    return g_notification(devs[0], None, None, min_diff, diff_time, actual_time, False, True)
        elif a_type == 'humidity':
            start_value = selector.row('rh_in_percentage', actual_time - diff_time)
            end_value = selector.row('rh_in_percentage', actual_time)
            for item in history:
                if item['rh_in_percentage'] is None:
                    logging.error('too small amount of data for co2 detector')
                    return g_notification(devs[0], None, None, min_diff, diff_time, actual_time, False, True)
        elif a_type == 'co2':
            start_value = selector.row('co2_in_ppm', actual_time - diff_time)
            end_value = selector.row('co2_in_ppm', actual_time)
            for item in history:
                if item['co2_in_ppm'] is None:
                    logging.error('too small amount of data for co2 detector')
                    return g_notification(devs[0], None, None, min_diff, diff_time, actual_time, False, True)
        else:
            raise ValueError('invalid anomaly type: {}'.format(a_type))

        diff = round(math.fabs(start_value - end_value), 2)

        anomaly = False
        if diff >= min_diff:
            anomaly = True

        return g_notification(devs[0], diff, anomaly, min_diff, diff_time, actual_time, True, False)
