"""Calculates exponential regression from given CO2 values.
"""
from dm.attrs.AbstractPrepareAttr import AbstractPrepareAttr
import numpy as np

__author__ = 'Peter Tisovčík'
__email__ = 'xtisov00@stud.fit.vutbr.cz'


class Regression(AbstractPrepareAttr):
    def __init__(self, row_selector, interval_selector, method):
        self._method = method
        super(Regression, self).__init__(row_selector, interval_selector)

    def execute(self, timestamp_start, timestamp_end, column, precision, prefix,
                enable_error, new_column_name):
        """It computes exponential regression from given CO2 values.

        :param timestamp_start: timestamp that denotes the start of time interval
                                from which the regression is computed
        :param timestamp_end: timestamp that denotes the end of time interval
                              from which the regression is computed
        :param column: name of column that contains required values
        :param precision: precision of calculation
        :param prefix: prefix of attribute name
        :param enable_error: if an error of exponential regression is in the output
        :param new_column_name: name of attribute
        :return: pair of lists, the first list contains pair of attribute name and
                 exponential regression from given CO2 values and pair of attribute
                 name and error of exponential regression (optional)
        """
        x = []
        y = []
        for timestamp in range(timestamp_start, timestamp_end):
            y.append(self.row_selector.row(column, timestamp))
            x.append(timestamp - timestamp_start)

        x = np.asarray(x)
        y = np.asarray(y)

        param, err = self._method.compute_parameter(x, y)
        name = self.attr_name(new_column_name, prefix, 'before', 0)
        before = [(name, round(param * 3600, precision))]

        if enable_error:
            before.append(('err', round(float(err), 8)))

        return before, []

    @staticmethod
    def gen_f_lambda(co2_start, co2_out):
        return lambda x, a: co2_out + (co2_start - co2_out) * np.exp(-a * x)

    @staticmethod
    def gen_f_prietok(co2_start, co2_out, volume):
        return lambda x, a: co2_out + (co2_start - co2_out) * np.exp(-a / volume * x)
