"""Calculates quantity values after linearization.
"""
from dm.attrs.AbstractPrepareAttr import AbstractPrepareAttr
from scipy import stats

__author__ = 'Klára Nečasová'
__email__ = 'xnecas24@stud.fit.vutbr.cz'


class InLinear(AbstractPrepareAttr):
    def execute(self, timestamp_before, timestamp_after, column, precision,
                start_before, end_before, start_after, end_after, prefix,
                new_column_name):
        """It computes quantity values after linearization.

        :param timestamp_before: timestamp selected from linearized time interval before event
        :param timestamp_after: timestamp selected from linearized time interval after event
        :param column: name of column that contains required values
        :param precision: precision of calculation
        :param start_before: time shift before event that denotes start of time interval
                             that is linearised (in seconds)
        :param end_before: time shift before event that denotes end of time interval
                           that is linearised (in seconds)
        :param start_after: time shift after event that denotes start of time interval
                            that is linearised (in seconds)
        :param end_after: time shift after event that denotes end of time interval
                          that is linearised (in seconds)
        :param prefix: prefix of attribute name
        :param new_column_name: name of attribute
        :return: pair of lists, each list contains pairs of attribute name and value
         after linearization
        """
        def compute(start, end, timestamp, interval_name):
            res = self.interval_selector.interval(column, start, end)
            x = []
            y = []
            for i in range(0, len(res)):
                x.append(i + start)
                y.append(res[i])

            slope, intercept, _, _, _ = stats.linregress(x, y)
            res = round(intercept + slope * timestamp, precision)

            if interval_name == 'before':
                interval = end_before - start_before
            else:
                interval = end_after - start_after
            name = self.attr_name(new_column_name, prefix, interval_name, str(interval))

            return name, res

        before = []
        after = []

        if start_before != end_before:
            before = [compute(start_before, end_before, timestamp_before, 'before')]

        if start_after != end_after:
            after = [compute(start_after, end_after, timestamp_after, 'after')]

        return before, after
