"""Calculates first differences using quantity values (only successive values).
"""
from dm.attrs.AbstractPrepareAttr import AbstractPrepareAttr

__author__ = 'Klára Nečasová'
__email__ = 'xnecas24@stud.fit.vutbr.cz'


class FirstDifferenceAttrB(AbstractPrepareAttr):
    def execute(self, timestamp, column, precision, intervals_before, intervals_after,
                normalize, enable_count, prefix, selected_before, selected_after,
                new_column_name):
        """It computes the first differences using quantity values.

        Only successive values are used for computation.

        :param timestamp: timestamp when event occurred
        :param column: name of column that contains required values
        :param precision: precision of calculation
        :param intervals_before: list of time points before event
        :param intervals_after: list of time points after event
        :param normalize: if the computed should be normalized
        :param enable_count: if number of positives values should be computed
        :param prefix: prefix of attribute name
        :param selected_before: list of lists of selected time points before event
        :param selected_after: list of lists of selected time points after event
        :param new_column_name: name of attribute
        :return: pair of lists, each list contains pairs of attribute name and
                 the first difference
        """
        before = []
        after = []

        middle = self.row_selector.row(column, timestamp)

        last_value = middle
        last_shift = 0
        for interval in intervals_before:
            value_time = timestamp - interval
            value = self.row_selector.row(column, value_time)

            if normalize:
                derivation = round((last_value - value) / (interval - last_shift), precision)
                name = self.attr_name(new_column_name, prefix, 'norm_before', interval)
            else:
                derivation = round(last_value - value, precision)
                name = self.attr_name(new_column_name, prefix, 'before', interval)

            before.append((name, self.transform(derivation, interval)))
            last_value = value
            last_shift = interval

        last_value = middle
        last_shift = 0
        for interval in intervals_after:
            value_time = timestamp + interval
            value = self.row_selector.row(column, value_time)

            if normalize:
                derivation = round((value - last_value) / (interval - last_shift), precision)
                name = self.attr_name(new_column_name, prefix, 'norm_after', interval)
            else:
                derivation = round(value - last_value, precision)
                name = self.attr_name(new_column_name, prefix, 'after', interval)

            after.append((name, self.transform(derivation, interval)))
            last_value = value
            last_shift = interval

        if enable_count:
            b, a = self._compute_increase(column, intervals_before, intervals_after,
                                          before, after,
                                          selected_before, selected_after, prefix)
            return before + b, after + a

        return before, after
