"""Calculates differences between real and linearized values of quantity in given time points.
"""
from dm.attrs.AbstractPrepareAttr import AbstractPrepareAttr
from scipy import stats

__author__ = 'Klára Nečasová'
__email__ = 'xnecas24@stud.fit.vutbr.cz'


class DifferenceBetweenRealLinear(AbstractPrepareAttr):
    def execute(self, timestamp, column, precision, intervals_before, intervals_after,
                window_size_before, window_size_after, prefix, new_column_name):
        """It computes difference between real and linearised course of given quantity.

        :param timestamp: timestamp which is in the middle of time interval used for calculation
        :param column: name of column that contains real values
        :param precision: precision of calculation
        :param intervals_before: list of time points before event
        :param intervals_after: list of time points after event
        :param window_size_before: time interval before event used to calculate linearised course
        :param window_size_after: time interval after event used to calculate linearised course
        :param prefix: prefix of attribute name
        :param new_column_name: name of attribute
        :return: pair of lists, each list contains pairs of attribute name and difference between
                 real and linearised course of given quantity
        """

        intervals_before = [0] + intervals_before
        before = []
        after = []
        infix = '_windowSize{0}_{1}'.format(window_size_before, window_size_after)

        # compute before
        x = []
        y = []
        values = {}

        start = timestamp - window_size_before
        end = timestamp + 1
        for time in range(start, end, 1):
            value = self.row_selector.row(column, time)
            x.append(time)
            y.append(value)
            values[time] = value
        slope, intercept, _, _, _ = stats.linregress(x, y)

        for interval in intervals_before:
            if interval > window_size_before:
                break

            time = timestamp - interval
            orig_value = values[time]
            linear_value = intercept + slope * time

            diff = round(linear_value - orig_value, precision)
            name = self.attr_name(new_column_name, prefix, infix + '_before', interval)
            before.append((name, self.transform(diff, interval)))

        # compute after
        x = []
        y = []
        values = {}

        start = timestamp
        end = timestamp + window_size_after + 1
        for time in range(start, end, 1):
            value = self.row_selector.row(column, time)
            x.append(time)
            y.append(value)
            values[time] = value
        slope, intercept, _, _, _ = stats.linregress(x, y)

        for interval in intervals_after:
            if interval > window_size_after:
                break

            time = timestamp + interval
            orig_value = values[time]
            linear_value = intercept + slope * time

            diff = round(linear_value - orig_value, precision)
            name = self.attr_name(new_column_name, prefix, infix + '_after', interval)
            after.append((name, self.transform(diff, interval)))

        return before, after
