"""Calculates number of positive differences in given time points, geometric mean,
   arithmetic mean, variance and standard deviation of differences.
"""
from abc import ABC, abstractmethod
from functools import reduce
import math

__author__ = 'Peter Tisovčík'
__email__ = 'xtisov00@stud.fit.vutbr.cz'


# https://www.smartfile.com/blog/abstract-classes-in-python/
# https://code.tutsplus.com/articles/understanding-args-and-kwargs-in-python--cms-29494
# http://homel.vsb.cz/~dor028/Casove_rady.pdf
class AbstractPrepareAttr(ABC):
    def __init__(self, row_selector, interval_selector, tr=None):
        self.name = self.__class__.__name__
        self.row_selector = row_selector
        self.transform = tr
        self.interval_selector = interval_selector

        if self.transform is None:
            self.transform = self.__simple_transform

        super(AbstractPrepareAttr, self).__init__()

    @staticmethod
    def __simple_transform(value, timestamp):
        return value

    @abstractmethod
    def execute(self, **kwargs):
        """It ensures calculation of the required attribute.

        :param kwargs:
        :return: pair of lists, each list contains pair of attribute name and its value
        """
        pass

    def attr_name(self, column_name, prefix, interval_type, interval):
        """ It generates a name of attribute.

        :param column_name: name of attribute
        :param prefix: prefix of attribute name
        :param interval_type: type of interval - before or after
        :param interval: time interval from which the attribute is calculated
        :return: name of attribute
        """
        return '{0}_{1}{2}_{3}_{4}'.format(self.name, column_name, prefix, interval_type,
                                           interval)

    def _compute_increase(self, new_column_name, intervals_before, intervals_after,  before, after,
                          selected_before, selected_after, prefix):
        """It computes number of positives values.

         It computes number of positives values that occurred in certain time points
         before and after event.

        :param new_column_name: name of attribute
        :param intervals_before: list of time points before event
        :param intervals_after: list of time points after event
        :param before: list of values in given time points before event
        :param after: list of values in given time points after event
        :param selected_before: list of lists of selected time points before event
        :param selected_after: list of lists of selected time points after event
        :param prefix: prefix of attribute name
        :return: pair of lists of numbers of positives values
        """
        before_out = []
        after_out = []

        for intervals in selected_before:
            before_increase = 0

            for interval in intervals:
                index = intervals_before.index(interval)
                value = before[index][1]

                if value > 0:
                    before_increase += 1

            suffix = '_'.join(str(x) for x in intervals)
            name = self.attr_name(new_column_name, prefix, 'before_increase', suffix)
            before_out.append((name, before_increase))

        for intervals in selected_after:
            after_increase = 0

            for interval in intervals:
                index = intervals_after.index(interval)
                value = after[index][1]

                if value > 0:
                    after_increase += 1

            suffix = '_'.join(str(x) for x in intervals)
            name = self.attr_name(new_column_name, prefix, 'after_increase', suffix)
            after_out.append((name, after_increase))

        return before_out, after_out

    def _extract_values(self, values):
        out = []
        for row in values:
            out.append(row[1])

        return out

    def geometric_mean(self, new_column_name, precision, values_before, values_after, prefix):
        """It computes geometric mean from given values before and after event.

        :param new_column_name: name of attribute
        :param precision: precision of calculation
        :param values_before: list of values before event
        :param values_after: list of values after event
        :param prefix: prefix of attribute name
        :return: pair of lists, each list contains pairs of attribute name and geometric mean
        """
        def compute(input_values, interval_name):
            values = self._extract_values(input_values)
            vals = values[1:]
            try:
                vals.remove(0.0)
                vals.remove(0)
            except:
                pass

            count = len(input_values)

            # sqrt can be negative
            is_negative = False
            tmp = reduce((lambda x, y: x * y), vals)
            if tmp < 0:
                is_negative = True
                tmp *= -1
            v1 = round(tmp ** (1/(count - 1)), precision)
            if is_negative:
                v1 *= -1

            attr_prefix = '_geometricMean' + prefix
            name = self.attr_name(new_column_name, attr_prefix, interval_name, '')
            return name, v1

        before = []
        after = []
        if len(values_before) > 1:
            before = [compute(values_before, 'before')]

        if len(values_after) > 1:
            after = [compute(values_after, 'after')]

        return before, after

    def arithmetic_mean(self, new_column_name, precision, values_before, values_after, prefix):
        """It computes arithmetic mean from given values before and after event.

        :param new_column_name: name of attribute
        :param precision: precision of calculation
        :param values_before: list of values before event
        :param values_after: list of values after event
        :param prefix: prefix of attribute name
        :return: pair of lists, each list contains pairs of attribute name and arithmetic mean
        """
        def compute(input_values, interval_name):
            count = len(input_values)
            values_sum = sum(self._extract_values(input_values))
            avg = round(values_sum / count, precision)

            attr_prefix = '_arithmeticMean' + prefix
            name = self.attr_name(new_column_name, attr_prefix, interval_name, '')
            return name, avg

        before = []
        after = []
        if len(values_before) > 1:
            before = [compute(values_before, 'before')]

        if len(values_after) > 1:
            after = [compute(values_after, 'after')]

        return before, after

    def variance(self, new_column_name, precision, values_before, values_after, prefix):
        """It computes variance from given values before and after event.

        :param new_column_name: name of column
        :param precision: precision of calculation
        :param values_before: list of values before event
        :param values_after: list of values after event
        :param prefix: prefix of attribute name
        :return: pair of lists, each list contains pairs of attribute name and variance
        """
        def compute(input_values, interval_name):
            count = len(input_values)
            values = self._extract_values(input_values)
            values_sum = sum(values)
            avg = values_sum / count

            res = 0
            for row in values:
                res += (row - avg) ** 2
            res = round(res/count, precision)

            attr_prefix = '_variance' + prefix
            name = self.attr_name(new_column_name, attr_prefix, interval_name, '')
            return name, res

        before = []
        after = []
        if len(values_before) > 1:
            before = [compute(values_before, 'before')]

        if len(values_after) > 1:
            after = [compute(values_after, 'after')]

        return before, after

    def standard_deviation(self, new_column_name, precision, values_before, values_after, prefix):
        """It computes standard deviation from given values before and after event.

        :param new_column_name: name of column
        :param precision: precision of calculation
        :param values_before: list of values before event
        :param values_after: list of values after event
        :param prefix: prefix of attribute name
        :return: pair of lists, each list contains pairs of attribute name and standard deviation
        """
        def compute(value, interval_name):
            attr_prefix = '_standardDeviation' + prefix
            name = self.attr_name(new_column_name, attr_prefix, interval_name, '')
            return name, round(value, precision)

        b, a = self.variance(new_column_name, precision, values_before, values_after, prefix)

        before = []
        after = []
        if len(values_before) > 1:
            before = [compute(math.sqrt(b[0][1]), 'before')]

        if len(values_after) > 1:
            after = [compute(math.sqrt(a[0][1]), 'after')]

        return before, after
