"""Weather data extraction from weather.com.
"""
import copy
import json
import logging
import requests
from dm.DateTimeUtil import DateTimeUtil
from dm.ConnectionUtil import ConnectionUtil as cu

__author__ = ''
__email__ = ''


class WundergroundCom:
    def __init__(self, precision=1):
        self.__precision = precision
        self.__endpoint = 'https://api.weather.com/v1/geocode/'
        self.__api_key = ''
        self.__log = logging.getLogger(self.__class__.__name__)

    def weather_by_coordinates(self, quantities, start, end, lat, lon):
        """It gets information about weather in a locality at required time interval.

        :param quantities: deprecated
        :param start: timestamp that denotes start of time interval
        :param end: timestamp that denotes end of time interval
        :param lat: latitude of a locality
        :param lon: longitude of a locality
        :return: information about weather
        """
        url_req = self.__endpoint
        url_req += str(lat) + '/'
        url_req += str(lon) + '/'
        url_req += 'observations/'
        url_req += 'historical.json?apiKey=' + self.__api_key
        url_req += '&startDate='
        url_req += str(DateTimeUtil.utc_timestamp_to_str(start, '%Y%m%d %H:%M:%S')[:-9])
        url_req += '&endDate='
        url_req += str(DateTimeUtil.utc_timestamp_to_str(end, '%Y%m%d %H:%M:%S')[:-9])
        url_req += '&units=m'

        self.__log.debug(url_req)

        json_data = requests.get(url_req).text

        python_obj = json.loads(json_data)

        if 'observations' not in python_obj:
            logging.debug('empty observations')
            return []

        out_general = []
        for element in python_obj['observations']:
            out_general.append({
                'at': element['valid_time_gmt'],
                'temperature_out': element['temp'],
                'humidity_out': element['rh'],
            })

        act = self.actual_weather(lat, lon)
        t = act['validTimeUtc'] % 1800
        out_general.append({
            'at': act['validTimeUtc']-t,
            'temperature_out': act['temperature'],
            'humidity_out': act['relativeHumidity']
        })

        generate_weather_data = self.__generate_weather_data(out_general, start, end)
        out_detailed = []

        for i in range(0, len(generate_weather_data)):
            weather = generate_weather_data[i]
            if weather['at'] < start or generate_weather_data[i]['at'] > end:
                continue

            out_detailed.append(generate_weather_data[i])

        return out_detailed

    def actual_weather(self, lat, lon):
        """It gets information about current weather in a locality.

        :param lat: latitude of a locality
        :param lon: longitude of a locality
        :return: information about current weather
        """
        url_req = 'https://api.weather.com/v3/wx/observations/current'
        url_req += '?apiKey=' + self.__api_key
        url_req += '&geocode='
        url_req += str(lat) + '%2C'
        url_req += str(lon)
        url_req += '&units=m&language=en-US&format=json'

        json_data = requests.get(url_req).text

        return json.loads(json_data)

    def __generate_weather_data(self, out_general, start, end):
        out_detailed = []

        # odstranenie null hodnot a casov, ktore nie su zarovnane na polhodinu
        out_not_null_value = []
        for item in out_general:
            failed = False

            # ak udaj obsahuje nejaku None hodnotu, vynechame ho
            for key, value in item.items():
                if value is None:
                    failed = True

            # ak udaj obsahuje nezarovnany cas na polhodinu, vynechame ho
            if item['at'] % 1800 != 0:
                failed = True

            if not failed:
                out_not_null_value.append(item)

        tmp = copy.deepcopy(out_not_null_value)
        out_general = []

        last_at = tmp[0]['at']
        for k in range(0, len(tmp)):
            i = copy.deepcopy(tmp[k])

            # ak je rovnaky cas ako predpokladany iba vlozime
            if last_at == i['at']:
                out_general.append(i)
                last_at += 1800
                continue

            while True:
                # ak je last_at mensi ako predpokladany, co znaci, ze chyba polozka
                # tak upravime cas na pozadovany a pridame
                # takto sa duplikuje aktualna polozka, meni sa len cas, hodnoty zostavaju
                if last_at <= tmp[k]['at']:
                    i = copy.deepcopy(tmp[k])
                    i['at'] = last_at
                    out_general.append(i)
                    last_at += 1800
                else:
                    break

        tmp = copy.deepcopy(out_general)
        out_general = []
        for row in tmp:
            if row['at'] < (start - (start % 1800)):
                continue
            if row['at'] > (end - (end % 1800) + 1800):
                continue

            out_general.append(row)

        # duplikujeme poslednu hodnotu, aby bolo mozne
        # generovat aj rozsah v poslednej polhodine dna
        out_general.append(out_general[-1])

        for i in range(0, len(out_general) - 1):
            temp_start = out_general[i]['temperature_out']
            temp_end = out_general[i + 1]['temperature_out']
            if temp_start - temp_end == 0:
                temp_increase = 0
            else:
                temp_diff = temp_end - temp_start
                temp_increase = temp_diff / 1800.0

            rh_start = out_general[i]['humidity_out']
            rh_end = out_general[i + 1]['humidity_out']
            if rh_start - rh_end == 0:
                rh_increase = 0
            else:
                rh_diff = rh_end - rh_start
                rh_increase = rh_diff / 1800.0

            temp = temp_start
            rh = rh_start
            for j in range(0, 1800):
                out_detailed.append({
                    'at': int(out_general[i]['at']) + j,
                    'temperature_out': round(float(temp), self.__precision),
                    'humidity_out': round(float(rh), self.__precision),
                })
                temp = temp + temp_increase
                rh = rh + rh_increase

        return out_detailed

    @property
    def api_key(self):
        return self.api_key

    @api_key.setter
    def api_key(self, api_key):
        self.__api_key = api_key
