"""Converts relative humidity to absolute humidity, relative humidity to specific humidity and
   ppm to milligrams per cubic meter.
"""
from math import exp

__author__ = 'Klára Nečasová'
__email__ = 'xnecas24@stud.fit.vutbr.cz'


class ValueConversionUtil:
    CO_MOLECULAR_WEIGHT = 44.0095  # g / mol

    @staticmethod
    def rh_to_absolute_g_m3(temp: float, rh: float) -> float:
        """It converts relative humidity to absolute humidity.

        :param temp: temperature
        :param rh: relative humidity
        :return: absolute humidity
        """
        result = (6.112 * exp((17.67 * temp) / (temp + 243.5)) * rh * 2.1674)
        result = result / (273.15 + temp)

        return result

    @staticmethod
    def ah_to_relative_percent(temp: float, ah: float) -> float:
        """It converts absolute humidity to relative humidity.

        :param temp: temperature
        :param ah: absolute humidity
        :return: relative humidity
        """
        result = (ah * (273.15 + temp))/(6.112 * exp((17.67 * temp) / (temp + 243.5)) * 2.1674)

        return result

    @staticmethod
    def rh_to_specific_g_kg(temp: float, rh: float) -> float:
        """It converts relative humidity to specific humidity.

        :param temp: temperature
        :param rh: relative humidity
        :return: specific humidity
        """
        saturated_partial_pressure = exp(23.58 - (4044.6 / (235.63 + temp)))
        partial_pressure = (rh * saturated_partial_pressure) / 100
        res = (622 * partial_pressure) / (101500 - partial_pressure)

        return res

    @staticmethod
    # http://www.aresok.org/npg/nioshdbs/calc.htm
    def co2_ppm_to_mg_m3(co2):
        """It converts ppm to milligrams per cubic meter.

        :param co2: ppm
        :return: milligrams per cubic meter
        """
        return co2 * ValueConversionUtil.CO_MOLECULAR_WEIGHT / 24.45
