"""Utils for date pre-processing.
"""
from dm.DBUtil import DBUtil
from dm.DateTimeUtil import DateTimeUtil
from dm.ValueConversionUtil import ValueConversionUtil as conv
from dm.ConnectionUtil import ConnectionUtil as cu
import logging

__author__ = 'Peter Tisovčík'
__email__ = 'xtisov00@stud.fit.vutbr.cz'


class PreProcessing:
    TIME_ATTR_NAME = 'measured_time'
    TIME_STRING_ATTR_NAME = 'measured_time_str'
    OPEN_CLOSE_ATTR_NAME = 'open_close'

    @staticmethod
    def db_name_maps(devices: list) -> list:
        """It gets a list of column names in database that is loaded from list of devices.

        A column name is loaded from a file to assign a given value to the right column in database.

        :param devices: list of information about devices
        :return: list of column names
        """

        names = []
        for device in devices:
            names.append(device['db_column_name'])

        return names

    @staticmethod
    def rename_attribute(values: list, old_attribute: str, new_attribute: str) -> list:
        """It renames an attribute in data from an original name to a new name.

        :param values: list of values
        :param old_attribute: name of an original attribute
        :param new_attribute: name of a new attribute
        :return: list of pairs of a renamed attribute and its value
        """
        out_values = []

        # loop over all values
        for value in values:
            new_val = {}

            # loop over all attributes
            for key, val in value.items():
                if key == old_attribute:
                    new_val[new_attribute] = val
                    continue

                new_val[key] = val
            out_values.append(new_val)

        return out_values

    @staticmethod
    def rename_all_attributes(items: list, devices: list) -> list:
        """It renames all attributes in a list that contains list of values.

        It renames individual attributes that is related to a given module.
        Each downloaded value from list of values with a defined name of attributes by default.
        New names of attributes are derived from list of devices.

        :param items: list of values
        :param devices: list of information about devices
        :return: list that contains a modified list of values including renamed attributes
        """
        out_values = []
        maps = PreProcessing.db_name_maps(devices)

        for i in range(0, len(items)):
            item = items[i]
            key = maps[i]

            item = PreProcessing.rename_attribute(item, 'at', PreProcessing.TIME_ATTR_NAME)
            item = PreProcessing.rename_attribute(item, 'value', key)

            out_values.append(item)

            i += 1

        return out_values

    @staticmethod
    def download_data(clients: list, devices: list, start: int, end: int) -> list:
        """It downloads required data according to a list of devices.

        :param clients: list of clients
        :param devices: list of information about devices
        :param start: timestamp from which data is downloaded
        :param end: timestamp to which data is downloaded
        :return: list of downloaded data
        """
        out_items = []

        for dev in devices:
            client = clients[dev['server_name']]

            values = client.history(
                dev['gateway_id'],
                dev['device_id'],
                dev['module_id'],
                start,
                end
            )['data']

            out_items.append(values)

        return out_items

    @staticmethod
    def generate_open_close(values: list, time_attribute_name: str,
                            open_close_attribute_name: str, start: int, end: int,
                            last_open_close_state: int) -> list:
        """It modifies downloaded data so that an interval between two adjacent records was one second.

        It modifies the data using linearization, it does not consider quantities or their course.

        :param values: list of values
        :param time_attribute_name: name of attribute that contains timestamp
        :param open_close_attribute_name: name of attribute that contains state of window (open/closed)
        :param start: timestamp that denotes start of an interval
        :param end: timestamp that denotes end of an interval
        :param last_open_close_state: last recorded state of a window
        :return: list of values that contains modified downloaded data (with step of one second)
        """
        # prevod hodnot zo stringu do cisiel
        for i in range(0, len(values)):
            values[i][open_close_attribute_name] = float(values[i][open_close_attribute_name])

        out = []

        # vygenerovanie zoznamu hodnot
        for t in range(start, end):
            out.append({
                time_attribute_name: t,
                open_close_attribute_name: None
            })

        out_index = 0
        for i in range(0, len(values)):
            value = values[i]

            for tt in range(out_index, len(out)):
                if out[tt][time_attribute_name] >= value[time_attribute_name]:
                    break
                out[tt][open_close_attribute_name] = last_open_close_state
                out_index += 1

            last_open_close_state = value[open_close_attribute_name]

        # dogenerovanie dat, v pripade, ze akcia s oknom je v ramci zadaneho intervalu
        for tt in range(out_index, len(out)):
            out[tt][open_close_attribute_name] = last_open_close_state

        return out

    @staticmethod
    def generate_data(values: list, value_attribute: str, time_attribute: str,
                      precision: int=7) -> list:
        """It modifies downloaded data so that an interval between two adjacent records was one second.

        It modifies the data using linearization, it does not consider quantities or their course.

        :param values: list of values
        :param value_attribute: name of attribute that contains data
        :param time_attribute: ame of attribute that contains timestamp
        :param precision: precision of calculation
        :return: list of values that contains modified downloaded data (with step of one second)
        """
        new = []

        if len(values) > 1:
            last_item = values[-1]
            default = int(cu.package('shift.last_value'))
            last_item[PreProcessing.TIME_ATTR_NAME] = last_item[PreProcessing.TIME_ATTR_NAME] + default
            values.append(last_item)

        for i in range(0, len(values) - 1):
            act_value = values[i]
            next_value = values[i + 1]

            value_start = act_value[value_attribute]
            value_end = next_value[value_attribute]

            if value_start is None or value_end is None:
                continue

            value_start = float(act_value[value_attribute])
            value_end = float(next_value[value_attribute])

            time_start = act_value[time_attribute]
            time_end = next_value[time_attribute]

            if round(value_start - value_end, precision) == round(0, precision):
                value_increase = 0
            else:
                value_diff = value_end - value_start

                if (time_end - time_start) == 0:
                    value_increase = 0
                else:
                    value_increase = value_diff / (time_end - time_start)

            act_value = value_start
            for j in range(1, time_end - time_start + 1):
                new.append({
                    time_attribute: time_start + j,
                    value_attribute: round(act_value, precision)
                })
                act_value = act_value + value_increase

        return new

    @staticmethod
    def cut_interval(items: list, start: int, end: int, time_attribute: str) -> list:
        """It cuts data to have the same start and end of an interval (starting and end time).

        :param items: list of values
        :param start: timestamp from which data is cut
        :param end: timestamp to which data is cut
        :param time_attribute: shift of start and end of time intervals
        :return: cut list of values
        """
        out_values = []

        for value in items:
            if value[time_attribute] < start or value[time_attribute] >= end:
                continue

            out_values.append(value)

        return out_values

    @staticmethod
    def check_start_end_interval(items: list, time_attribute: str) -> None:
        """It checks if lists of values have the same starting and end time.

        :param items: list of values
        :param time_attribute: name of attributes that contains timestamp
        :return: None
        """
        start = None
        end = None

        for item in items:
            if start is None:
                start = item[0][time_attribute]

            if end is None:
                end = item[-1][time_attribute]

            if not item:
                raise SyntaxError('empty list of values')

            if start != item[0][time_attribute]:
                raise SyntaxError('start time must be equal in all items')

            if end != item[-1][time_attribute]:
                raise SyntaxError('end time must be equal in all items')

    @staticmethod
    def join_items(items: list, time_attribute: str) -> list:
        """It joins several lists of values into one list.

        The resulting list contains only one attribute that includes time because
        times are the same. It contains various values that were included in various lists.

        THe method has to be called only if passed list of values was checked using the method
        check_start_end_interval().

        :param items: list of values
        :param time_attribute: name of attribute that contains timestamp
        :return: list of values including all attributes and time
        """
        item_out = items[0]
        for values in items[1:]:

            for i in range(0, len(values)):
                item = values[i]

                # loop over dictionary items
                for key, value in item.items():
                    if key is time_attribute and item_out[i][time_attribute] != value:
                        raise SyntaxError(
                            'value in `%s` attribute is different' % (time_attribute))

                    item_out[i][key] = value

        return item_out

    @staticmethod
    def value_filter(data):
        """It checks if indoor relative humidity is in an admissible range.

        If indoor relative humidity is out of range, it is logged.

        :param data: dictionary of data
        :return: dictionary of data
        """
        for i in range(0, len(data)):
            row = data[i]
            for k in range(0, len(row)):
                item = row[k]

                key = 'rh_in_percentage'
                if key in item and item[key] is not None and float(item[key]) > 100:
                    t = DateTimeUtil.utc_timestamp_to_str(item['measured_time'])
                    value = float(item[key])
                    logging.error('{0}, {1}: value {2} is out of range, skipped'.format(
                                  t, key, value))
                    item[key] = None

        return data

    @staticmethod
    def prepare_downloaded_data(clients: list, devices: list, start: int, end: int,
                                time_shift: int, last_open_close_state) -> list:
        """It prepares downloaded data.

        :param clients: list of clients
        :param devices: list of information about devices
        :param start: timestamp that denotes start of an interval
        :param end: timestamp that denotes end of an interval
        :param time_shift: time shift that is subtracted or added to start or end of an interval respectively
        :param last_open_close_state: last recorded state of a window
        :return: dictionary of prepared data
        """
        data = PreProcessing.download_data(clients, devices,
                                           start - time_shift,
                                           end + time_shift)
        data = PreProcessing.rename_all_attributes(data, devices)
        data = PreProcessing.value_filter(data)

        new_data = []
        maps = PreProcessing.db_name_maps(devices)
        i = 0
        for key in maps:
            item = data[i]
            val = []

            if key == PreProcessing.OPEN_CLOSE_ATTR_NAME:
                val = PreProcessing.generate_open_close(item, PreProcessing.TIME_ATTR_NAME,
                                                        key, start, end, last_open_close_state)
            else:
                if len(item) != 0:
                    val = PreProcessing.generate_data(item, key, PreProcessing.TIME_ATTR_NAME)
                    val = PreProcessing.generate_data(val, key, PreProcessing.TIME_ATTR_NAME)

            val = PreProcessing.cut_interval(val, start, end, PreProcessing.TIME_ATTR_NAME)

            # v pripade, ze vybrany interval pre danu hodnotu neobsahuje vsetky data tak sa vynuluje
            if (end - start) != len(val):
                val = []

                for k in range(start, end):
                    val.append({'measured_time': k, key: None})

            new_data.append(val)

            i += 1

        return new_data

    @staticmethod
    def prepare_value_conversion(value):
        """It converts relative humidity to absolute and specific humidity.

        :param value: data related to an event
        :return: data related to an event including absolute and specific humidity
        """
        # sensor 1
        exists_in = 'temperature_in_celsius' in value and 'rh_in_percentage' in value
        if exists_in:
            temperature_in_celsius = value['temperature_in_celsius']
            rh_in_percentage = value['rh_in_percentage']

            if temperature_in_celsius is not None and rh_in_percentage is not None:
                # absolute humidity in 1
                value['rh_in_absolute_g_m3'] = conv.rh_to_absolute_g_m3(
                    value['temperature_in_celsius'],
                    value['rh_in_percentage'])

                # specific humidity in 1
                value['rh_in_specific_g_kg'] = conv.rh_to_specific_g_kg(
                    value['temperature_in_celsius'],
                    value['rh_in_percentage'])

        # sensor 2
        exists_in2 = 'temperature_in2_celsius' in value and 'rh_in2_percentage' in value
        if exists_in2:
            temperature_in2_celsius = value['temperature_in2_celsius']
            rh_in2_percentage = value['rh_in2_percentage']

            if temperature_in2_celsius is not None and rh_in2_percentage is not None:
                # absolute humidity in 2
                value['rh_in2_absolute_g_m3'] = conv.rh_to_absolute_g_m3(
                    value['temperature_in2_celsius'],
                    value['rh_in2_percentage'])

                # specific humidity in 2
                value['rh_in2_specific_g_kg'] = conv.rh_to_specific_g_kg(
                    value['temperature_in2_celsius'],
                    value['rh_in2_percentage'])

        # sensor out
        exists_out = 'temperature_out_celsius' in value and 'rh_out_percentage' in value
        if exists_out:
            temperature_out_celsius = value['temperature_out_celsius']
            rh_out_percentage = value['rh_out_percentage']

            if temperature_out_celsius is not None and rh_out_percentage is not None:
                # absolute humidity out
                value['rh_out_absolute_g_m3'] = conv.rh_to_absolute_g_m3(
                    value['temperature_out_celsius'],
                    value['rh_out_percentage'])

                # specific humidity out
                value['rh_out_specific_g_kg'] = conv.rh_to_specific_g_kg(
                    value['temperature_out_celsius'],
                    value['rh_out_percentage'])

        return value

    @staticmethod
    def insert_values(conn, table_name, values, maps, write_each, precision):
        """It inserts values to a table.

        :param conn: connection to a database
        :param table_name: name of table
        :param values: data related to event
        :param maps: list of column names
        :param write_each: number of events - 1 between two adjacent events that are written to a table
        :param precision: precision of inserted values
        :return: None
        """
        for i in range(0, len(values)):
            value = values[i]
            t = ()

            if i % write_each != 0:
                continue

            for column in DBUtil.measured_values_table_column_names():
                if column == PreProcessing.TIME_STRING_ATTR_NAME:
                    t += (DateTimeUtil.utc_timestamp_to_str(
                        value[PreProcessing.TIME_ATTR_NAME]),)
                    continue

                if column in maps and value[column] is not None:
                    t += (round(value[column], precision),)
                else:
                    t += (None,)

            DBUtil.insert_value(conn, t, False, table_name)

    @staticmethod
    def ppm_filter(data, ppm_limit=2000):
        """It sets carbon dioxide concentration to None if its value is out of range.

        :param data: dictionary of data
        :param ppm_limit: maximal admissible value of the concentration
        :return: dictionary of data where excessive concentration is set to None
        """
        for i in range(0, len(data)):
            row = data[i]
            key = 'co2_in_ppm'
            if row[key] is not None and row[key] >= ppm_limit:
                row[key] = None

        return data

    @staticmethod
    def prepare(clients: list, devices: list, start: int, end: int,
                last_open_close_state: int, time_shift: int):
        """It prepares data.

         :param clients: list of clients
         :param devices: list of information about devices
         :param start: timestamp from which data is downloaded
         :param end: timestamp to which data is downloaded
         :param last_open_close_state: last recorded state of a window
         :param time_shift: time shift that is subtracted or added to start or end of an interval respectively.
         :return: list of attribute names and list of values
         """
        values = []
        try:
            values = PreProcessing.prepare_downloaded_data(clients, devices, start, end,
                                                           time_shift, last_open_close_state)

            PreProcessing.check_start_end_interval(values, PreProcessing.TIME_ATTR_NAME)
            values = PreProcessing.join_items(values, PreProcessing.TIME_ATTR_NAME)
        except Exception as e:
            maps = [
                PreProcessing.TIME_ATTR_NAME,
                PreProcessing.TIME_STRING_ATTR_NAME,
                PreProcessing.OPEN_CLOSE_ATTR_NAME,
            ]

            return maps, values[0]

        for i in range(0, len(values)):
            value = values[i]
            PreProcessing.prepare_value_conversion(value)

        maps = PreProcessing.db_name_maps(devices)
        for value in values:
            for key, _ in value.items():
                maps.append(key)
            break

        # odstranenie duplicit zo zonamu
        return list(set(maps)), values

    @staticmethod
    def extract_items(data, value_column):
        """It prepares data.

         :param data: dictionary of data
         :param value_column: name of column
         :return: list of values and list of timestamps
         """
        values = []
        times = []

        for item in data:
            values.append(item[value_column])
            times.append(item[PreProcessing.TIME_ATTR_NAME])

        return values, times
