"""Gets information about weather from OpenWeather.org.
"""
from dm.DateTimeUtil import DateTimeUtil
import logging
import requests
import json

__author__ = ''
__email__ = ''


class OpenWeatherOrg:
    def __init__(self, precision=2):
        self.__precision = precision
        self.__log = logging.getLogger(self.__class__.__name__)
        self.__endpoint = 'https://api.openweathermap.org/data/2.5/forecast?'
        self.__api_key = ''

    def weather_by_city_name(self, quantities, start, end, name):
        """It gets weather from a given city during certain time interval.

        :param quantities: list of required quantities
        :param start: timestamp that denotes start of time interval
        :param end: timestamp that denotes end of time interval
        :param name: name of city
        :return: weather data
        """
        url_req = self.__endpoint
        url_req += 'q=' + name
        url_req += '&appid=' + self.__api_key
        self.__log.debug(url_req)

        data = json.loads(requests.get(url_req).text)
        extracted = self._extract_quantities(quantities, data)
        every_hour = self._generate_every_hour(extracted, quantities)

        return self._generate_values(every_hour, quantities, start, end)

    def weather_by_city_id(self, quantities, start, end, city_id: int):
        """It gets weather from city based on its identifier during certain time interval.

        :param quantities: list of required quantities
        :param start: timestamp that denotes start of time interval
        :param end: timestamp that denotes end of time interval
        :param city_id: identifier of city
        :return: weather data
        """
        url_req = self.__endpoint
        url_req += 'id=' + str(city_id)
        url_req += '&appid=' + self.__api_key
        self.__log.debug(url_req)

        data = json.loads(requests.get(url_req).text)
        extracted = self._extract_quantities(quantities, data)
        every_hour = self._generate_every_hour(extracted, quantities)

        return self._generate_values(every_hour, quantities, start, end)

    def weather_by_coordinates(self, quantities, start, end, lat, lon):
        """It gets weather from city based on its coordinates during certain time interval.

        :param quantities: list of required quantities
        :param start: timestamp that denotes start of time interval
        :param end: timestamp that denotes end of time interval
        :param lat: latitude
        :param lon: longitude
        :return: weather data
        """
        url_req = self.__endpoint
        url_req += 'lat=' + str(lat)
        url_req += '&lon=' + str(lon)
        url_req += '&appid=' + self.__api_key
        self.__log.debug(url_req)

        data = json.loads(requests.get(url_req).text)
        extracted = self._extract_quantities(quantities, data)
        every_hour = self._generate_every_hour(extracted, quantities)

        return self._generate_values(every_hour, quantities, start, end)

    def _extract_quantities(self, quantities, data):
        """It extracts required quantities.

        A simple selection of required quantities and conversion to a simple structure.

        :param quantities: dictionary representing quantities
        :param data: dictionary containing downloaded data from website about weather
        :return: list of quantities
        """
        out = []

        limit = 5

        for i in range(0, limit):
            if limit > len(data['list']):
                break

            row = data['list'][i]

            c_time = DateTimeUtil.local_time_str_to_utc(row['dt_txt'],
                                                        format='%Y-%m-%d %H:%M:%S')
            item = {
                'at': int(c_time.timestamp()),
            }

            main = row['main']
            for quantity in quantities:
                if quantity == 'temperature_out':
                    item[quantity] = int(main['temp'] - 273.15)
                elif quantity == 'humidity_out':
                    item[quantity] = main['humidity']
                else:
                    item[quantity] = main[quantity]

            out.append(item)

        return out

    def _generate_every_hour(self, data, quantities):
        """It generates data gathered during one hour with the step of one second to make generation more accurate.

        It creates a structure that contains data gathered during one hour, pre-calculates quantities
        and step used to get more data. The aim is to achieve higher precision.
        Value of the step corresponds with one second.

        :param data: dictionary containing downloaded data from website about weather
        :param quantities: dictionary representing quantities
        :return: dictionary of quantities containing generated values
        """
        out = []

        for i in range(0, len(data)-1):
            item = data[i]

            new_item = {'at': item['at'] - 2 * 3600}

            for quantity in quantities:
                if i == 0:
                    new_item[quantity] = item[quantity]
                    new_item['next_hour_step_' + quantity] = 0
                else:
                    step = data[i-1][quantity] - data[i][quantity]
                    new_item[quantity] = round(item[quantity] + step/3*2, 1)
                    new_item['next_hour_step_' + quantity] = step / (3600*3)

            out.append(new_item)

            new_item = {'at': item['at'] - 3600}
            for quantity in quantities:
                if i == 0:
                    new_item[quantity] = item[quantity]
                    new_item['next_hour_step_' + quantity] = 0
                else:
                    step = data[i-1][quantity] - data[i][quantity]
                    new_item[quantity] = round(item[quantity] + step/3, 1)
                    new_item['next_hour_step_' + quantity] = step / (3600*3)
            out.append(new_item)

            out.append(item)

        # dogenerovanie hodnot kroku, pre zaznamy, ktore sluzili ako pivot
        for i in range(2, len(out) - 1, 3):
            item = out[i]

            for quantity in quantities:
                item['next_hour_step_' + quantity] = out[i + 1]['next_hour_step_' + quantity]

        # odstranenie poslednej polozky, ktora nemala step
        out.pop()
        return out

    def _generate_values(self, data, quantities, start, end):
        """It generates required interval with a precision of one second.

        :param data: dictionary containing downloaded data from website about weather
        :param quantities: dictionary representing quantities
        :param start: timestamp that denotes start of time interval
        :param end: timestamp that denotes end of time interval
        :return: dictionary of quantities containing generated values
        """
        out = []

        index = 0
        for at in range(start, end):
            while at - 3600 > data[index]['at']:
                index += 1

            item = data[index]
            new_item = {
                'at': at,
            }

            for quantity in quantities:
                diff = at - data[index]['at']
                val = item[quantity] - data[index]['next_hour_step_' + quantity] * diff
                new_item[quantity] = round(val, 1)

            out.append(new_item)
        return out

    @property
    def api_key(self):
        return self.api_key

    @api_key.setter
    def api_key(self, api_key):
        self.__api_key = api_key
