"""Defines various filters to select data that meets a required condition.
"""
__author__ = 'Klára Nečasová'
__email__ = 'xnecas24@stud.fit.vutbr.cz'


class FilterUtil:
    @staticmethod
    def only_valid_events(events):
        """It gets only valid events.

        :param events: list of events
        :return: list of valid events
        """
        out = []

        for event in events:
            if event['valid_event']:
                out.append(event)
        return out

    @staticmethod
    def min_time_interval(events, time_interval_length):
        """It gets events that lasted at least the required time.

        :param events: list of events
        :param time_interval_length: length of time interval
        :return: list of events that lasted at least the required time
        """
        out = []

        for event in events:
            if (event['e_end']['timestamp'] - event['e_start']['timestamp']) >= time_interval_length:
                out.append(event)
        return out

    @staticmethod
    def min_max_time_interval(events, min_time, max_time):
        """It gets events that lasted admissible time.

        :param events: list of events
        :param min_time: minimal length of event
        :param max_time: maximal length of event
        :return: list of events that lasted admissible time
        """
        out = []

        for event in events:
            time_diff = event['e_end']['timestamp'] - event['e_start']['timestamp']
            if min_time <= time_diff < max_time:
                out.append(event)
        return out

    @staticmethod
    def temperature_diff(events, min_value, max_value):
        """It gets events in which temperature difference was in admissible range.

        :param events: list of events
        :param min_value: minimal difference of temperature
        :param max_value: maximal difference of temperature
        :return: list of events in which temperature difference was in admissible range
        """
        out = []

        for event in events:
            if len(event['measured']['temperature_in2_celsius']) == 0:
                continue
            if len(event['measured']['temperature_out_celsius']) == 0:
                continue

            temp_in = event['measured']['temperature_in2_celsius'][0]
            temp_out = event['measured']['temperature_out_celsius'][0]

            if min_value <= abs(temp_in - temp_out) < max_value:
                out.append(event)

        return out

    @staticmethod
    def temperature_out_max(events, max_value):
        """It gets events in which temperature was not higher than given value.

        :param events: list of events
        :param max_value: maximal temperature
        :return: list of events in which temperature was not higher than given value
        """
        out = []

        for event in events:
            temp_out = event['measured']['temperature_out_celsius'][0]

            if temp_out < max_value:
                out.append(event)

        return out

    @staticmethod
    def humidity(events, min_out_specific_humidity, min_diff, max_diff):
        """It gets events that fulfill the conditions.

        It gets events in which specific humidity difference was in admissible range
        and outside specific humidity was not higher than given value.

        :param events: list of events
        :param min_out_specific_humidity:
        :param min_diff:
        :param max_diff:
        :return: list of events
        """
        out = []

        for event in events:
            specific_in = event['measured']['rh_in2_specific_g_kg'][0]
            specific_out = event['measured']['rh_out_specific_g_kg'][0]

            if specific_out < min_out_specific_humidity:
                out.append(event)
                continue

            if min_diff <= abs(specific_out - specific_in) <= max_diff:
                out.append(event)

        return out

    @staticmethod
    def attribute(events, attribute_name, value):
        """It gets events in which attribute contains given value.

        :param events: list of events
        :param attribute_name: name of attribute
        :param value: required value of attribute
        :return: list of events in which attribute contains given value
        """
        out = []

        for event in events:
            if event[attribute_name] == value:
                out.append(event)

        return out

    @staticmethod
    def attribute_exclude(events, attribute_name, value):
        """It gets events in which attribute does not contain given value.

        :param events: list of events
        :param attribute_name: name of attribute
        :param value: not required value of attribute
        :return: list of events in which attribute does not contain given value
        """
        out = []

        for event in events:
            if event[attribute_name] != value:
                out.append(event)

        return out

    @staticmethod
    def measured_values_not_empty(events, attributes):
        """It gets events that include valid measured data in given attributes.

        :param events: list of events
        :param attributes: names of attributes
        :return: list of events that include valid measured data in given attributes
        """
        out = []

        for event in events:
            valid = True
            for key, value in event['measured'].items():
                if key in attributes and value == []:
                    valid = False

            if valid:
                out.append(event)

        return out

    @staticmethod
    def derivation_not_zero(events):
        """It gets events in which derivation is not zero.

        :param events: list of events
        :return: list of events in which derivation is not zero
        """
        out = []

        derivation_key = [
            'after',
            'before',
            'no_event_after',
            'no_event_before',
        ]

        for event in events:
            valid = True

            for key in derivation_key:
                values = event['derivation'][key]
                if not valid:
                    break

                for value in values:
                    if value == 0:
                        valid = False
                        break
            if valid:
                out.append(event)

        return out

    @staticmethod
    def min_timestamp(events, timestamp):
        """It gets events that started after given time.

        :param events: list of events
        :param timestamp: timestamp from which events are considered
        :return: list of events that started after given time.
        """
        out = []

        for event in events:
            if event['e_start']['timestamp'] > timestamp:
                out.append(event)

        return out

    @staticmethod
    def min_length(events, length):
        """It gets events that lasted at least given time.

        :param events: list of events
        :param length: minimal length of event
        :return: list of events that lasted at least given time
        """
        out = []

        for event in events:
            if (event['e_end']['timestamp'] - event['e_start']['timestamp']) > length:
                out.append(event)

        return out
