"""Converts various time formats (one to another).
"""
import datetime
import pytz

__author__ = 'Klára Nečasová'
__email__ = 'xnecas24@stud.fit.vutbr.cz'


class DateTimeUtil:
    @staticmethod
    def local_time_str_to_utc(date_str, timezone='Europe/Prague', format='%Y/%m/%d %H:%M:%S'):
        """It converts given local time in string form to UTC form (number).

        :param date_str: local time in string form
        :param timezone: timezone of local time
        :param format: output format of time
        :return: object containing local time in UTC form
        """
        # https://www.saltycrane.com/blog/2009/05/converting-time-zones-datetime-objects-python/

        datetime_obj_naive = datetime.datetime.strptime(date_str, format)
        datetime_obj_pacific = pytz.timezone(timezone).localize(datetime_obj_naive)

        return datetime_obj_pacific

    @staticmethod
    def utc_timestamp_to_local_time(timestamp, timezone='Europe/Prague'):
        """It converts given UTC timestamp to local time according to timezone.

        :param timestamp: UTC timestamp
        :param timezone: timezone of local time
        :return: local time
        """
        utc = datetime.datetime.fromtimestamp(timestamp, pytz.timezone('UTC'))
        local_time = utc.astimezone(pytz.timezone(timezone))

        return local_time

    @staticmethod
    def utc_timestamp_to_str(timestamp, format='%Y-%m-%d %H:%M:%S'):
        """It converts UTC timestamp to local time in string form.

        :param timestamp: UTC timestamp
        :param format: output format of time
        :return: local time in string form
        """
        local_time = DateTimeUtil.utc_timestamp_to_local_time(timestamp, 'Europe/Prague')

        return local_time.strftime(format)

    @staticmethod
    def create_interval_str(start, end):
        """It converts given interval expressed using UTC timestamp to string form.

        :param start: start of interval
        :param end: end of interval
        :return: interval expressed in string form
        """
        out = DateTimeUtil.utc_timestamp_to_str(start)
        out += ' - '
        out += DateTimeUtil.utc_timestamp_to_str(end, '%H:%M:%S')

        return out
