"""Utils to create connections to db, REST client and utils for environment setting.
"""
from configparser import ConfigParser, ExtendedInterpolation
from dm.BeeeOnClient import BeeeOnClient
import logging
import mysql.connector
import matplotlib
import os

__author__ = 'Peter Tisovčík'
__email__ = 'xtisov00@stud.fit.vutbr.cz'

CONFIG_FILE = '/etc/dp/config.ini'


class ConnectionUtil:
    MAX_TESTABLE_EVENTS = 75

    @staticmethod
    def create_con(config_file=CONFIG_FILE):
        """It creates connection to the database.

        :param config_file: path to the configuration file
        :return: connection to the database
        """
        if ConnectionUtil.is_testable_system():
            return mysql.connector.connect(
                host='localhost',
                user='root',
                passwd='',
                database='demo'
            )

        config = ConfigParser()
        config.read(config_file)

        return mysql.connector.connect(
            host=config['db']['host'],
            user=config['db']['user'],
            passwd=config['db']['passwd'],
            database=config['db']['database']
        )

    @staticmethod
    def rapid_miner(config_file=CONFIG_FILE):
        """It gets information about RapidMiner.

        :param config_file: path to the configuration file
        :return: information about RapidMiner
        """
        config = ConfigParser()
        config.read(config_file)

        return config['rapidminer']

    @staticmethod
    def is_testable_system():
        """It gets information if system is travis for testing.

        :return: true if system is travis for testing, otherwise false
        """
        # example travis hostname: travis-job-d072bd30-f722-4d10-*
        return 'travis' in os.uname()[1]

    @staticmethod
    def get_value(key, config_file):
        """It gets value using given key from configuration file.

        :param key: key for obtaining of data from information
        :param config_file: path to the configuration file
        :return: required data obtained using the key
        """
        config = ConfigParser(interpolation=ExtendedInterpolation())
        config.read(config_file)

        return config[key]

    @staticmethod
    def ant_work_api_key(server_type, config_file=CONFIG_FILE):
        """It gets information about api key from configuration file.

        :param server_type: type of server - dm
        :param config_file: path to configuration file
        :return: information about api key
        """
        key = '{0}.{1}'.format('api.key', server_type)
        return ConnectionUtil.get_value('ant-work', config_file)[key]

    @staticmethod
    def rehivetech_api_key(server_type, config_file=CONFIG_FILE):
        """It gets information about api key from configuration file.

        :param server_type: type of server - dm or acontroller
        :param config_file: path to configuration file
        :return: information about api key
        """
        key = '{0}.{1}'.format('api.key', server_type)
        return ConnectionUtil.get_value('rehivetech', config_file)[key]

    @staticmethod
    def open_weather_api_key(config_file=CONFIG_FILE):
        """It gets information about OpenWeather.org api key from configuration file.

        :param config_file: path to configuration file
        :return: information about api key
        """
        key = '{0}'.format('api.key')
        return ConnectionUtil.get_value('OpenWeatherOrg', config_file)[key]

    @staticmethod
    def wunderground_api_key(config_file=CONFIG_FILE):
        """It gets information about WunderGround.com api key from configuration file.

        :param config_file: path to configuration file
        :return: information about api key
        """
        key = '{0}'.format('api.key')
        return ConnectionUtil.get_value('WundergroundCom', config_file)[key]

    @staticmethod
    def open_detector(key, config_file=CONFIG_FILE):
        """It gets information about open detector from configuration file.

        :param key: key for obtaining of data from information about open detector
        :param config_file: path to configuration file
        :return: information about open detector obtained using the key
        """
        return ConnectionUtil.get_value('open-detector', config_file)[key]

    @staticmethod
    def estimate(key, config_file=CONFIG_FILE):
        """It gets information about estimation from configuration file.

        :param key: key for obtaining of data from information about estimation
        :param config_file: path to configuration file
        :return: information about estimation obtained using the key
        """
        return ConnectionUtil.get_value('estimate', config_file)[key]

    @staticmethod
    def package(key, config_file=CONFIG_FILE):
        """It gets information about package from configuration file.

        :param key: key for obtaining of data from information about package
        :param config_file: path to configuration file
        :return: information about package obtained using the key
        """
        return ConnectionUtil.get_value('package', config_file)[key]

    @staticmethod
    def predictor(key, config_file=CONFIG_FILE):
        """It gets information about OpenWeather.org api key from configuration file.

        :param key: key for obtaining of data from information about predictor
        :param config_file: path to configuration file
        :return: information about predictor obtained using the key
        """
        return ConnectionUtil.get_value('predictor', config_file)[key]

    @staticmethod
    def setup_clients():
        """It sets clients used for communication with servers.

        :return: dictionary of set clients
        """
        cls = {
            'ant-work': BeeeOnClient('ant-work.fit.vutbr.cz', 8010),
            'rehivetech': BeeeOnClient('beeeon.rehivetech.com', 8010),
        }
        cls['ant-work'].api_key = ConnectionUtil.ant_work_api_key('dm')
        cls['rehivetech'].api_key = ConnectionUtil.rehivetech_api_key('acontroller')

        return cls

    @staticmethod
    def setup_clients_logout(cls):
        """It log outs clients.

        :param cls: list of clients for communication with server
        :return: None
        """
        for key, client in cls.items():
            del client

    @staticmethod
    def setup_logging(log_level=logging.DEBUG):
        """It sets loggers for logging.

        :param log_level: type of logging level
        :return: None
        """
        logging.basicConfig(level=log_level,
                            format='%(asctime)s %(levelname)s %(message)s')

        matplotlib_logger = logging.getLogger('matplotlib')
        matplotlib_logger.setLevel(logging.ERROR)

        requests_logger = logging.getLogger('requests')
        requests_logger.setLevel(logging.ERROR)
        
        urllib_logger = logging.getLogger('urllib3')
        urllib_logger.setLevel(logging.ERROR)

        if ConnectionUtil.is_testable_system():
            logging.getLogger('WundergroundCom').setLevel(logging.ERROR)
