"""Creates a csv file from OrderedDict.
"""
import csv
import io
import logging
import os

__author__ = 'Klára Nečasová'
__email__ = 'xnecas24@stud.fit.vutbr.cz'


class CSVUtil:
    @staticmethod
    def create_csv_file(data: list, filename: str, enable_append=False):
        """It creates a CSV file.

        :param data: list of pairs containing attribute name and its value
        :param filename: output filename
        :param enable_append: if it is true, data is appended to the end of file
        :return: None
        """
        if data == []:
            logging.warning('empty data to write')
            return

        write_header = True
        if os.path.isfile(filename) and enable_append:
            write_header = False

        field_names = []
        for key, _ in data[0].items():
            field_names.append(key)

        mode = 'w'
        if enable_append:
            mode = 'a'

        with open(filename, mode) as f:
            csv_writer = csv.DictWriter(f, fieldnames=field_names)

            if write_header:
                csv_writer.writeheader()

            for item in data:
                csv_writer.writerow(item)

            f.close()

    @staticmethod
    def create_csv_io(data: list):
        """It creates StringIO from given data.

        :param data: list of pairs containing attribute name and its value
        :return: StringIO containing given data
        """
        if data == []:
            logging.warning('empty data to write')
            return

        out = io.StringIO()
        field_names = []
        for key, _ in data[0].items():
            field_names.append(key)

        csv_writer = csv.DictWriter(out, fieldnames=field_names)
        csv_writer.writeheader()

        for item in data:
            csv_writer.writerow(item)

        out.seek(0)
        return out
