"""REST API client for communication with BeeeOn server.
"""
from dm.HTTPClient import HTTPClient
import json
import logging

__author__ = 'Klára Nečasová'
__email__ = 'xnecas24@stud.fit.vutbr.cz'


class BeeeOnClient:
    """Client for communication with server supporting BeeeOn api."""

    def __init__(self, host, port):
        self.__host = host
        self.__port = port
        self.__api_key = ""
        self.__token_id = ""
        self.__log = logging.getLogger(self.__class__.__name__)

    def refresh_token(self):
        """It refreshes token.

        :return: token id
        """
        logging.debug('obtaining token id')
        data = {'key': self.__api_key, 'provider': 'apikey'}
        try:
            req = HTTPClient(self.__host, self.__port, "/auth", "POST", False)
            req.body(json.dumps(data))
            res, body = req.perform()
        except Exception as e:
            self.__log.error(e)
            return

        json_res = json.loads(body)

        if 'code' not in json_res:
            raise LookupError('return code not found')

        if int(json_res['code']) != 200:
            self.__log.warning("invalid token_id")
            exit(1)
            return

        return json_res['data']['id']

    def history(self, gateway, device, sensor, start, end, interval=1, aggregation='avg'):
        """It gets historical data from server.

        :param gateway: identifier of gateway
        :param device: identifier of device
        :param sensor: identifier of sensor
        :param start: start of interval from which data is generated
        :param end: end of interval from which data is generated
        :param interval: time interval from which data are aggregated
        :param aggregation: type of aggregation
        :return: json that contains historical data from given time interval
        """
        if not self.__token_id:
            self.__token_id = self.refresh_token()

        endpoint = '/gateways/' + gateway
        endpoint += '/devices/' + device
        endpoint += '/sensors/' + str(sensor)
        endpoint += '/history'
        endpoint += '?range=' + str(start) + ',' + str(end)
        endpoint += '&interval=' + str(interval)
        endpoint += '&aggregation=' + aggregation

        req = HTTPClient(self.__host, self.__port, endpoint, "GET", False)
        req.authorize(self.__token_id)

        res, body = req.perform()

        if res.code != 200:
            logging.debug('request: %s' % endpoint)
            logging.debug(body)

        return json.loads(body)

    def sensors_info(self, gateway_id, device_id):
        """It gets information about sensor.

        :param gateway_id: identifier of gateway
        :param device_id: identifier of device
        :return: json that contains information about sensor
        """
        if not self.__token_id:
            self.__token_id = self.refresh_token()

        endpoint = '/gateways/' + str(gateway_id) + '/devices/' + str(
            device_id) + '/sensors'

        req = HTTPClient(self.__host, self.__port, endpoint, "GET", False)
        req.authorize(self.__token_id)
        res, body = req.perform()

        return json.loads(body)['data']

    def device_info(self, gateway_id, device_id):
        """It gets information about device.

        :param gateway_id: identifier of gateway
        :param device_id: identifier of device
        :return: json that contains information about device
        """
        if not self.__token_id:
            self.__token_id = self.refresh_token()

        endpoint = '/gateways/' + str(gateway_id) + '/devices/' + str(
            device_id)

        req = HTTPClient(self.__host, self.__port, endpoint, "GET", False)
        req.authorize(self.__token_id)
        res, body = req.perform()

        return json.loads(body)['data']

    def __logout(self):
        if not self.__token_id:
            self.__log.warning('token is not set')
            return

        endpoint = '/auth'

        req = HTTPClient(self.__host, self.__port, endpoint, "DELETE", False)
        req.authorize(self.__token_id)

        req.perform()

        self.__log.debug('logout was successful')

    def __del__(self):
        try:
            self.__logout()
        except:
            logging.error('problem with logout')

    @property
    def api_key(self):
        return self.__api_key

    @api_key.setter
    def api_key(self, key):
        self.__api_key = key

    @property
    def token_id(self):
        if not self.__token_id:
            self.__token_id = self.refresh_token()

        return self.__token_id

    @token_id.setter
    def token_id(self, token):
        if self.__token_id:
            logging.warning('token is set')
        else:
            self.__token_id = token
