package org.ndx.pcap;

import org.apache.hadoop.fs.Seekable;
import org.apache.hadoop.io.LongWritable;
import org.apache.hadoop.io.ObjectWritable;
import org.apache.hadoop.mapred.RecordReader;
import org.apache.hadoop.mapred.Reporter;
import org.ndx.model.pcap.PacketModel.RawFrame;

import java.io.DataInputStream;
import java.io.IOException;
import java.util.Iterator;

public class PcapRecordReader implements RecordReader<LongWritable, ObjectWritable> {
    private PcapReader pcapReader;
    private Iterator<RawFrame> pcapReaderIterator;
    private Seekable baseStream;
    private DataInputStream stream;
    private Reporter reporter;

    private long packetCount = 0;
    private long start, end;

    public PcapRecordReader(PcapReader pcapReader, long start, long end, Seekable baseStream, DataInputStream stream,
                            Reporter reporter) {
        this.pcapReader = pcapReader;
        this.baseStream = baseStream;
        this.stream = stream;
        this.start = start;
        this.end = end;
        this.reporter = reporter;

        pcapReaderIterator = pcapReader.iterator();
    }

    @Override
    public void close() throws IOException {
        stream.close();
    }

    @Override
    public boolean next(LongWritable key, ObjectWritable value) throws IOException {
        if (!pcapReaderIterator.hasNext())
            return false;

        key.set(++packetCount);
        value.set(pcapReaderIterator.next());

        reporter.setStatus("Read " + getPos() + " of " + end + " bytes");
        reporter.progress();

        return true;
    }

    @Override
    public LongWritable createKey() {
        return new LongWritable();
    }

    @Override
    public ObjectWritable createValue() {
        return new ObjectWritable();
    }

    @Override
    public long getPos() throws IOException {
        return baseStream.getPos();
    }

    @Override
    public float getProgress() throws IOException {
        if (start == end)
            return 0;
        return Math.min(1.0f, (getPos() - start) / (float) (end - start));
    }

}