/**
 *  HLOC - Horizon LOCate
 *
 *  Author: Jan Brejcha <ibrejcha@fit.vutbr.cz>, <brejchaja@gmail.com>
 *  Copyright (C) 2014-2018  Jan Brejcha
 *
 *  OPEN SOURCE LICENCE VUT V BRNĚ
 *  Verze 1.
 *  Copyright (c) 2010, Vysoké učení technické v Brně, Antonínská 548/1, PSČ 601 90
 *  -------------------------------------------------------------------------------
 */

#include <iomanip>
#include <cstdio>
#include <vector>
#include <sstream>
#include <fstream>
#include <cmath>
#include <stdexcept>
#include <limits>
#include <string.h>
#include <algorithm>

#include <QString>
#include <QDir>
#include <QList>
#include <QApplication>
//#include <QDesktopWidget>
#include <QImage>
#include <QXmlStreamReader>
#include <QDebug>
#include <QFile>
#include <QThreadPool>
#include <QFuture>
#include <QtConcurrentRun>
#include <QTime>
#include <QHash>
#include <QMutex>
#include <QMutexLocker>


#include "PlaceSet.h"
#include "ContourlettExtractor.h"
#include "Contourlett.h"
#include "DatabaseEntry.h"
#include "BOWDatabase.h"
#include "BOWQueryResult.h"


#include "bytearray.h"

#include "pointmatcher/PointMatcher.h"

#include "boost/filesystem.hpp"
#include "boost/bind/bind.hpp"

#include <cassert>
#include <fstream>
#include <sstream>
#include "ResultInfo.h"

#include <chrono>
#include <thread>

#include <argumentparser.h>
#include "fieldofviewutil.h"

#include <qjson-qt5/serializer.h>
#include <qjson-qt5/parser.h>

#include <QAtomicInt>

#ifdef USE_LIBHTTPSERVER
    #include "stringzip.h"
    #include <httpserver.hpp>
#endif

#define MAX_FEAT_CNT 1000000

#define TIME_EXPERIMENTS


using namespace std;
using namespace argpar;

typedef PM::DataPoints DP;
typedef PM::Parameters Parameters;

typedef std::vector< std::pair<int, ResultInfo> > QueryResult;


#include <time.h>

#define PANORAMA_FOV 360

const QString HLOC_PLACE_SET = ".hloc_places.xml";

const QString photoparamDirName = QString("./");
void findPose(QString photoparam);

//Mutex for synchronization of ICP threads.
QMutex icpMutex;

extern void calculateICPOnCandidate(BOWDatabase *db, float *dir, QueryResult::iterator &it,
                             std::vector< std::pair<int, int> >queryPoints, QString queryName, float fov,
                                    QString placeSetName);

bool resultLessThan(std::pair<int, float> a, std::pair<int, float> b)
{
    return a.second < b.second;
}

QString dbNameToDocName(QString name)
{
    QStringList comp = name.split(QDir::separator());
    QFileInfo info(name);
    return comp[comp.size() - 2];
    //return info.baseName();
}

int train_list(const char *list, const char *dbname)
{

    //add entries to db
    QString qlist(list);
    QFile textFile(qlist);
    if(!textFile.open(QIODevice::ReadOnly))
    {
        std::cerr << "Could not open list file";
        return 1;
    }
    //Open file for reading
    QStringList sl;

    std::cout << "reading file list" << std::endl;
    while(!textFile.atEnd())
    {
        sl.append(textFile.readLine());
    }
    std::cout << "file list reading DONE." << std::endl;

    //create db
    BOWDatabase *d = new BOWDatabase(string(dbname), MAX_FEAT_CNT);

    long cnt = 1;
    for (QStringList::iterator it = sl.begin(); it != sl.end(); ++it)
    {
        QString filename = it->remove(QRegExp("[\n\t\r]"));
        std::cout << "training " << filename.toStdString() << std::endl;
        try
        {

            ContourlettExtractor ce = ContourlettExtractor::loadFromBin(filename);
            DatabaseEntry de(ce, filename, PANORAMA_FOV, DatabaseEntry::FeaType::BOTH);
            d->addSearchEntry(de);

            if (cnt % 100 == 0) //each 10 000 items save the database to file.
            {
                qDebug() << "Processed " << cnt << " horizons, out of " << sl.size();
            }

            ++cnt;
        }
        catch(std::runtime_error &re)
        {
            std::cerr << "Training error: " << std::string(re.what()) << std::endl;
        }
    }
    d->finalizeDatabase();

    return EXIT_SUCCESS;
}


void printContainsMetricResults(BOWDatabase &db, std::vector< std::pair<int, float> > &res)
{
    std::sort(res.begin(), res.end(), resultLessThan);
    for (std::vector< std::pair<int, float> >::iterator it = res.begin(); it != res.begin() + min(1000, db.size()); ++it)
    {
        printf("distance: %f, to doc: %s\n", it->second, db.getDocumentName(it->first).c_str());
    }
}

void prepareDirAndLocResults(BOWDatabase &db, float *dir, QueryResult &win)
{
    float max = 1000000;
    int in;
    int index = 0;


    for (int i = 0; i < db.size(); ++i)
    {
        for (int j = 0; j < 360; ++j)
        {
            in = i * 360 + j;
            float val = dir[in];
            if (val < max)
            {
                max = val;
                index = j;
            }
        }
        ResultInfo rInfo(index);
        win.push_back(std::pair<int, ResultInfo>(i, rInfo));
        max = 1000000;
    }

    std::sort(win.begin(), win.end(),
            [dir](std::pair<int, ResultInfo> a, std::pair<int, ResultInfo> b){return dir[(size_t)a.first * (size_t)360 + (size_t)a.second.angle] < dir[(size_t)b.first * (size_t)360 + (size_t)b.second.angle]; });

}

inline int angleFromRes(std::vector< std::pair<int, int> >::iterator it)
{
    int angle_bin = it->second - (240 * it->first);
    return (int)round(((float)angle_bin / 240) * 360);
}

Place findInPlaceSet(QList<Place> &places, QString docName)
{

    for (QList<Place>::iterator pit = places.begin(); pit != places.end(); ++pit)
    {
        if (pit->name == docName)
        {
            return *pit;
        }
    }
    return Place();
}

/**
 * @brief loadHorizonFromCsv
 *        Loads the horizon points from CSV file generated by LibPointMatcher.
 * @param fileName  Name of the csv file to load.
 * @return vector of points in 2D.
 */
std::vector< std::pair<double, double> >  loadHorizonFromCsv(const char *fileName)
{
    std::ifstream infile(fileName);
    std::string line;
    std::vector< std::pair<double, double> > points;

    int idx = 0;
    while (std::getline(infile, line))
    {
        if (idx == 0)
        {
            idx++;
            continue; //skip the first (header) line
        }
        double x, y;
        char delim;
        std::istringstream iss(line);
        if (! (iss >> x >> delim >> y))
        {
            break;
        }
        points.push_back(std::pair<double, double>(x, y));
    }
    return points;
}

void printDirAndLocResultsRaw(BOWDatabase &db, float *dir, QueryResult &win, int limit, QString photoName)
{
    printf("Results for query: %s\n", photoName.toStdString().c_str());
    printf("document; score; angle; icp ratio\n");

    /** with this comment is commented out the code dealing with json
     *  QVariantList result_json; **/

    for (QueryResult::iterator it = win.begin();
                it != std::min(win.begin() + limit, win.end()); ++it)
    {
        printf("%s; %f; %d; %f\n", db.getDocumentName(it->first).c_str(), //name
               dir[(size_t)it->first * (size_t)360 + (size_t)it->second.angle],                           //BOW score
                it->second.icpAngle == -1 ? it->second.angle : it->second.icpAngle, //angle
                it->second.matchRatio);                                           //ICP match score

    }
}


QByteArray printDirAndLocResults(BOWDatabase &db, float *dir, QueryResult &win, int limit, QString photoName, QList<Place> places, QHash<QString, Place*> &placeMap, const std::vector< std::pair<int, int> > &queryPoints)
{
    printf("Results for query: %s\n", photoName.toStdString().c_str());
    printf("document; score; angle; icp ratio\n");

    /** with this comment is commented out the code dealing with json**/
    QVariantList result_json;

    for (QueryResult::iterator it = win.begin();
                it != std::min(win.begin() + limit, win.end()); ++it)
    {
        //int angle = angleFromRes(it);


        QVariantMap map;
        map.insert("yaw", it->second.angle);

        QString panorama_name(db.getDocumentName(it->first).c_str());
        QString doc_name = dbNameToDocName(panorama_name);
        QHash<QString, Place*>::iterator qit = placeMap.find(doc_name);

        double lat = 0; double lon = 0;
        if (qit != placeMap.end())
        {
            Place *candidate = qit.value();
            lat = candidate->latitude;
            lon = candidate->longitude;
        }

        map.insert("latitude", lat);
        map.insert("longitude", lon);



        //QString icp_dir = photoparamDirName + "/" + doc_name + "/maps/"; //FIXME: wrong now
        QString icp_dir = panorama_name;

#ifndef TIME_EXPERIMENTS
        ContourlettExtractor ce = ContourlettExtractor::create(panorama_name);
        std::vector< std::pair<int, int> > panoramaPoints = ce.getHorizonLinePanoramaPoints();

        // insert panorama points
        QVariantList panoPointXList;
        QVariantList panoPointYList;
        //there are three panoramas next to each other, we take only the central one.
        int panoramaWidth = panoramaPoints.size() / 3;
        for (int i = 0; i < panoramaWidth; ++i)
        {
            panoPointXList.append(panoramaPoints[i].first);
            panoPointYList.append(panoramaPoints[i].second);
        }
        map.insert("pano_x", panoPointXList);
        map.insert("pano_y", panoPointYList);

        QVariantList queryPointXList;
        QVariantList queryPointYList;
        for (size_t i = 0; i < queryPoints.size(); ++i)
        {
            //translate the x coordinate to the origin, since they are mapped
            //to the central panorama
            queryPointXList.append(queryPoints[i].first - panoramaWidth);
            queryPointYList.append(queryPoints[i].second);
        }
        map.insert("query_x", queryPointXList);
        map.insert("query_y", queryPointYList);
#endif //TIME_EXPERIMENTS

        result_json.append(map);

        printf("%s; %f; %f; %f; %d; %f\n", db.getDocumentName(it->first).c_str(), //name
               lat, lon,                         //gps
               dir[(size_t)it->first * (size_t)360 + (size_t)it->second.angle],                           //BOW score
                it->second.icpAngle == -1 ? it->second.angle : it->second.icpAngle, //angle
                it->second.matchRatio);                                           //ICP match score

    }

    QJson::Serializer serializer;
    bool ok;
    QByteArray json = serializer.serialize(result_json, &ok);
    if (ok)
    {
        return json;
    }
    else
    {
        std::cerr << "Unable to create output json with results.";
    }

    printf("Done\n");
    QByteArray empty;
    return empty;
}

void pointCloudToCSV(std::string filename, std::vector< std::pair<int, int> > pointCloud)
{
    remove(filename.c_str());
    std::fstream fs;
    fs.open (filename, std::fstream::in | std::fstream::out | std::fstream::app);

    for (std::vector< std::pair<int, int> >::iterator it = pointCloud.begin();
            it != pointCloud.end(); ++it)
    {
        fs << it->first << " , " << it->second << std::endl;
    }
    fs.close();
}

int calculateXOffsetForAngle(int angle, float fov, float pano_width)
{
    angle = (angle + 450 - (int)floor(fov/2.0)) % 360 + 360;
    float chunkf = pano_width / 360.0;
    int offset_x = (int)floor(chunkf * angle);
    return offset_x;
}

void pointCloudToCSVTranslated(std::string filename, std::vector< std::pair<int, int> > pointCloud, int angle, float fov)
{
    //calculate translation according to the angle + translate to second panorama
    float pano_width = 8192.0; //FIXME
    int offset_x = calculateXOffsetForAngle(angle, fov, pano_width);

    remove(filename.c_str());
    std::fstream fs;
    fs.open (filename, std::fstream::in | std::fstream::out | std::fstream::app);
    if (!fs.good())
    {
        std::cerr << "Unable to open csv output stream." << std::endl;
    }
    for (std::vector< std::pair<int, int> >::iterator it = pointCloud.begin();
            it != pointCloud.end(); ++it)
    {
        fs << it->first + offset_x << " , " << it->second << std::endl;
    }
    fs.close();
}


DP pointCloudToDataPointsTranslated(
        std::vector< std::pair<int, int> > pointCloud,
        int angle,
        float fov,
        float pano_width = 8192.0)
{
    int offset_x = calculateXOffsetForAngle(angle, fov, pano_width);

    typedef PM::DataPoints::Labels PMLabels;
    typedef PM::DataPoints::Label Label;
    //create point cloud to be aligned
    PMLabels lbl;
    lbl.push_back(Label("x", 1));
    lbl.push_back(Label("y", 1));
    Eigen::MatrixXf feat(3, pointCloud.size());
    int idx = 0;
    for (std::pair<int, int> &p : pointCloud)
    {
        feat.col(idx) << p.first + offset_x, p.second, 1.0;
        ++idx;
    }
    DP points_cloud(feat, lbl);
    return points_cloud;
}

DP pointCloudToDataPoints(std::vector< std::pair<int, int> > pointCloud)
{
    typedef PM::DataPoints::Labels Labels;
    typedef PM::DataPoints::Label Label;
    //create point cloud to be aligned
    Labels lbl;
    lbl.push_back(Label("x", 1));
    lbl.push_back(Label("y", 1));
    Eigen::MatrixXf feat(3, pointCloud.size());
    int idx = 0;
    for (std::pair<int, int> &p : pointCloud)
    {
        feat.col(idx) << p.first, p.second, 1.0;
        ++idx;
    }
    DP points_cloud(feat, lbl);
    return points_cloud;
}

std::vector< std::pair<int, int> > translatePoints(const std::vector< std::pair<int, int> >queryPoints, int angle)
{
    //as our panorama has on x=0 270 degrees, we have to transform the
    //angle: angle = (angle + 90) % 360 to correct the x coordinates
    angle = (angle + 90) % 360;
    float pano_width = 8192.0;
    float chunkf = pano_width / 360.0;
    int offset_x = (int)floor(chunkf * angle);
    //angle * chunkf is the translation in the cloud point = rotation in panorama
    std::vector< std::pair<int, int> >res;
    for (std::vector< std::pair<int, int> >::const_iterator it = queryPoints.begin();
            it != queryPoints.end(); ++it)
    {
        res.push_back(std::pair<int, int>(it->first + offset_x, it->second));
    }
    return res;
}



extern void calculateICPOnCandidate(BOWDatabase *db, float *dir, QueryResult::iterator &it,
                             std::vector< std::pair<int, int> >queryPoints, QString queryName, float fov,
                                    QString placeSetName, bool saveVtk = false)
{
    QString panorama_name(db->getDocumentName(it->first).c_str());
    vector< pair< PM::TransformationParameters, float> > sampling_res;

    int st = 1;
    int cnt = 6;
    int *angle_arr = new int[cnt+1];
    int idx = 0;


    //ContourlettExtractor ce = ContourlettExtractor(panorama_name);
    ContourlettExtractor ce = ContourlettExtractor::create(panorama_name);
    std::vector< std::pair<int, int> > panoramaPoints = ce.getHorizonLinePanoramaPoints();

    for (int angle_step = -(cnt/2); angle_step <= (cnt/2); angle_step += st)
    {
        int c_angle = it->second.angle + angle_step;
        angle_arr[idx] = c_angle;
        string c_angle_s = std::to_string(c_angle);

        std::string panoramaDir = dbNameToDocName(panorama_name).toStdString();
        try
        {
            QFileInfo placeSetNameInfo(placeSetName);

            const DP ref = pointCloudToDataPoints(panoramaPoints);
            const DP data = pointCloudToDataPointsTranslated(queryPoints,
                                                             c_angle, fov,
                                                             panoramaPoints.size() / 3);


            // Create the default ICP algorithm
            PM::ICP icp;

            // load YAML config
            std::string icpdef = "locateIcp.yaml";
            ifstream ifs(icpdef);
            if (ifs.good())
            {
                //cerr << "Cannot open config file " << icpdef << " could not do ICP" << std::endl;
                icp.loadFromYaml(ifs);
            }
            else
            {
                cerr << "cannot find configfile " << icpdef << ", using default one." << std::endl;
                icp.setDefault();
            }

            // Compute the transformation to express data in ref

            DP data_out(data);

            PM::TransformationParameters T = icp(data, ref);

            // Transform data to express it in ref
            icp.transformations.apply(data_out, T);

            // extract closest points
            icp.matcher->init(ref);
            PM::Matches matches = icp.matcher->findClosests(data_out);
            float matchRatio = matches.getDistsQuantile(1.0);
            // weight paired points
            /*const PM::OutlierWeights outlierWeights = icp.outlierFilters.compute(data_out, ref, matches);

            // generate tuples of matched points and remove pairs with zero weight
            const PM::ErrorMinimizer::ErrorElements matchedPoints( data_out, ref, outlierWeights, matches);

            // extract relevant information for convenience
            const int dim = matchedPoints.reading.getEuclideanDim();
            const int nbMatchedPoints = matchedPoints.reading.getNbPoints();
            const PM::Matrix matchedRead = matchedPoints.reading.features.topRows(dim);
            const PM::Matrix matchedRef = matchedPoints.reference.features.topRows(dim);

            // compute mean distance
            const PM::Matrix dist = (matchedRead - matchedRef).colwise().norm(); // replace that by squaredNorm() to save computation time
            const float meanDist = dist.sum()/nbMatchedPoints;
            float matchRatio = meanDist;*/

            // Safe files to see the results in case they do not exist

            if (saveVtk)
            {
                data.save("icp_" + queryName.toStdString() + "_" + c_angle_s + "_in.vtk");
                ref.save("icp_" + queryName.toStdString() + "_ref.vtk");
                data_out.save("icp_" + queryName.toStdString() + "_" + c_angle_s + "_out.vtk");
            }

            //save to csv to be able to further use it
            //ref.save(outputBaseFile + "icp" + "_ref.csv");
            //data_out.save("icp_" + queryName.toStdString() + "_" + c_angle_s + "_out.csv");

            sampling_res.push_back(pair< PM::TransformationParameters, float>(T, matchRatio));
            //it->second.T = T;
            //it->second.matchRatio = matchRatio;

            std::cerr << "." << std::flush;
            ++idx;
        }
        catch(runtime_error &e)
        {
            std::cerr << "error: " << panoramaDir << ", " << e.what() << std::endl;
        }
    }
    //find the best
    float matchMin = std::numeric_limits<float>::max();
    int best_angle = -1;
    PM::TransformationParameters bestT;
    int index = 0;
    for (pair< PM::TransformationParameters, float> &p : sampling_res)
    {
        if (p.second < matchMin)
        {
            matchMin = p.second;
            bestT = p.first;
            best_angle = angle_arr[index];
            //best_angle = it->second.angle + index;
        }
        ++index;
    }
    //store the best as output
    QMutexLocker locker(&icpMutex);
    it->second.T = bestT;
    it->second.matchRatio = matchMin;
    it->second.icpAngle = best_angle;
    delete[] angle_arr;
    locker.unlock();
}

void runICP(BOWDatabase &db, float *dir, QueryResult &win,
            std::vector< std::pair<int, int> >queryPoints, int limit, QString queryName, float fov,
            QString placeSetName, bool saveVtk)
{
    std::cerr << "Calculating ICP:" << std::flush;
    QList< QFuture<void> > resultList;
    for (QueryResult::iterator it = win.begin();
                    it != std::min(win.begin() + limit, win.end()); ++it)
    {
        float score = dir[it->first * 360 + it->second.angle];
        //if (score > 0)
        //{
            //run threads on qthreadpool
            QFuture<void> future = QtConcurrent::run(boost::bind(calculateICPOnCandidate, &db, dir, it, queryPoints, queryName, fov, placeSetName, saveVtk));
            resultList.push_back(future);
        //}
    }
    //wait for threads to be finished
    for (QFuture<void> &f : resultList)
    {
        f.waitForFinished();
    }
    std::cerr << std::endl;
    std::sort(win.begin(), win.begin() + std::min((size_t)limit, win.size()),
        [dir](std::pair<int, ResultInfo> a, std::pair<int, ResultInfo> b){return a.second.matchRatio < b.second.matchRatio; });
}

QByteArray processQuery(const QImage &query, QString fileName, BOWDatabase &db, float fov,
                  QString placeSetName, QList<Place> places,
                  QHash<QString, Place*> &placeMap, bool batch = false,
                  int icpLimit = -1,
                  DatabaseEntry::FeaType featureType = DatabaseEntry::FeaType::BOTH,
                  bool saveVtk = false)
{
    QFileInfo queryInfo(fileName);
    QString photoName = queryInfo.fileName();
    if(fov <= 0)
    {
        std::cerr <<
        "Either photoparam photo has to be used, or FOV parameter has to be set."
        << std::endl;
        QByteArray empty;
        return empty;
    }

    FieldOfViewUtil fovu(query, fov, 8192);
    ContourlettExtractor qce(fovu.fitToFOV());
    DatabaseEntry q(qce, fileName, fov, featureType);

    //create dir voting array
    size_t dirsize = (size_t)360 * (size_t)db.size() + (size_t)360;
    float *dir = new float[dirsize];
    std::fill(dir, dir + dirsize, 1000000);

    clock_t t;
    t = clock();
    std::vector< std::pair<int, float> > res = db.query(q, dir);
    t = clock() - t;
    //double time = ((float)t)/CLOCKS_PER_SEC;

    //extract point cloud for ICP from query image

    //ContourlettExtractor ce(fileName);
    std::vector< std::pair<int, int> > queryPoints = qce.getHorizonLinePoints(fov);
    //printContainsMetricResults(db, res);

    QueryResult win;

    prepareDirAndLocResults(db, dir, win);

    //print first 100 elements of dir array
    /*for (int i = 0; i < 100; ++i)
    {
        for (int angle = 0; angle < 360; ++angle)
        {
            std::cout << dir[win[i].first * 360 + angle] << ", ";
        }
        std::cout << std::endl;
    }*/

    //save loc results
    //	includeLocationResultsToPlace(placeSetName, photoName, db, win, dir);
    //int limit = win.size();
    int limit;
    if (icpLimit < 0)
    {
        limit = win.size();
    }
    else
    {
        limit = icpLimit;
    }
    try{
        if (limit > 0)
        {
            runICP(db, dir, win, queryPoints, limit, photoName, fov, placeSetName, saveVtk);
        }
    }
    catch(std::exception &e)
    {
        std::cerr << e.what() << std::endl;
    }


    //save ICP results
    //includeICPResultsToPlace(placeSetName, photoName, db, win, limit);

    if (limit == 0)
    {
       limit = 100;
    }
    QByteArray result = printDirAndLocResults(db, dir, win, limit, photoName, places, placeMap, queryPoints);
    //visualizeDirAndLocResults(db, dir, win, limit, placeSetName, photo);


    delete []dir;
    dir = NULL;

    return result;
}

QByteArray processQuery(std::string file, BOWDatabase &db, float fov,
                  QString placeSetName, QList<Place> places,
                  QHash<QString, Place*> &placeMap, bool batch = false,
                  int icpLimit = -1,
                  DatabaseEntry::FeaType featureType = DatabaseEntry::FeaType::BOTH,
                  bool saveVtk = false)
{
    QString fileName(file.c_str());

    QFileInfo queryInfo(fileName);
    QString photoName = queryInfo.fileName();
    QImage qim(fileName);
    return processQuery(qim, fileName, db, fov, placeSetName, places,
                        placeMap, batch, icpLimit, featureType, saveVtk);

}

int rundb(vector<string> dbnames, const char *placeSetName,
          DatabaseEntry::FeaType ft = DatabaseEntry::FeaType::BOTH,
          bool saveVtk = false)
{
    QString qPlaceSetName(placeSetName);
    QFileInfo placeSetInfo(qPlaceSetName);
    PlaceSet ps(placeSetInfo.fileName());
    QList<Place> places = ps.getAll();

    QHash<QString, Place*> placeMap;
    for (Place &p : places)
    {
        placeMap.insert(p.name, &p);
    }

    //QDomDocument *databaseDoc = new QDomDocument("BOWDatabase");
    //load database
    BOWDatabase *d = new BOWDatabase(dbnames[0], MAX_FEAT_CNT);
    if (d == NULL)
    {
        return EXIT_FAILURE;
    }

    std::cout << "db size: " << d->size() << std::endl;
    //wait for commands
    bool quit = false;

    //Welcome
    std::cout << "Welcome to HLocDB! You are in interactive mode." << std::endl;
    std::cout << "Usage: \n	q			exits the program. \n\
    query <image_path> 	processes the query into the database with the \n\
                            image at <image_path>" << std::endl;

    while (!quit)
    {
        //prompt
        std::cout << "hlocdb> ";

        std::string input;
        std::getline(std::cin, input);
        std::string filename;
        float fov = -1;
        if (input.compare("q") == 0)
        {
            quit = true;
        }
        else{
            //tokenize the input and do what the input wanted or print error.
            std::istringstream ss( input );

            int state = 0;
            while (!ss.eof())
            {
                std::string x;
                std::getline(ss, x, ' ');
                if (state == 1)
                {
                    filename = x;
                    state = 2;
                    continue;
                }
                if (state == 2)
                {
                    fov = atof(x.c_str());
                    state = 3;
                    continue;
                }
                if (state == 3)
                {
                    state = 0;
                    int limit = QString(x.c_str()).toInt();
                    processQuery(filename, *d, fov, QString(placeSetName), places, placeMap, false, limit, ft, saveVtk);
                    break;
                }
                else{
                    if (x.compare("query") == 0)
                    {
                        state = 1;
                    }
                    else if (x.compare("") != 0)
                    {
                        std::cout << "Unsupported command" << std::endl;
                    }
                }

            }
            if (state == 2 || state == 3)
            {
                processQuery(filename, *d, fov, QString(placeSetName), places, placeMap, false, -1, ft, saveVtk);
            }
        }
    }

    delete d;

    return EXIT_SUCCESS;
}

BOWDatabase * loadDbFromStdin()
{
    QFile *stdi = new QFile();
    stdi->open(stdin, QIODevice::ReadOnly);
    QTextStream txtStream(stdi);
    BOWDatabase *db = new BOWDatabase;
    while(!txtStream.atEnd())
    {
        QXmlStreamReader *xml = new QXmlStreamReader(txtStream.device());
        xml->readNext();
        while (xml->hasError())
        {
            qDebug() << "xml error detected";
            delete xml;
            xml = new QXmlStreamReader(txtStream.device());
        }
        db->addEntriesFromXmlStreamReader(*xml);
        delete xml;
    }
    db->finalizeDatabase();

    std::cout << "db size: " << db->size() << std::endl;
    return db;
}

/**
 * @brief compute3DRotation
 * Computes 3D rotation from 2D transform and initial yaw assuming that
 * the 2D transform is made in cylindric projection.
 * @param [in] width of the panorama image
 * @param [in] height of the panorama image
 * @param [in] T 2D transformation (3x3 matrix)
 * obtained usually from ICP on query and db horizon
 * @param yaw [out] yaw rotation
 * @param pitch [out] pitch rotation
 * @param roll [out] roll rotation
 */
void compute3DRotation(int width, int height, int im_width, int im_height, float yaw_init, PM::TransformationParameters T, float fov,
                       float &yaw, float &pitch, float &roll)
{

    float dyPix = 2.0 * tanf(0.5 * (2.0 * M_PI / width));

    int angle = ((int)yaw_init + 450) % 360 + 360;
    float chunkf = width / 360.0;
    int x_off = (int)floor(chunkf * angle);

    Eigen::Vector3f origin(x_off, 0.0, 1.0);
    Eigen::Vector3f torigin = T * origin;
    torigin = torigin / torigin(2);
    Eigen::Vector3f diff = torigin - origin;
    float dx = diff(0);
    float dy = diff(1) - (height/2.0);

    float sin_roll = T(0, 1);
    roll = -asin(sin_roll);
    float angle_pix_ratio = 2.0 * M_PI / (float)width;
    float d_yaw = dx * angle_pix_ratio;
    pitch = atan(dy * dyPix);
    float yaw_init_deg = (yaw_init / 180.0) * M_PI;
    //270 - yaw so that the rotation corresponds with Locate
    yaw = ((270.0 / 180.0) * M_PI) - (yaw_init_deg + d_yaw);
}

void findPose(QString pano, QString query, float fov, bool saveVtk)
{
    //convert fov to degrees
    fov = (180.0 * fov) / M_PI;
    //train db on single panorama
    QString dbpath = "./tempPoseDB";
    QFileInfo dbinfo(dbpath);
    if (dbinfo.exists())
    {
        std::cerr << "Temporary pose DB already exists, deleting." << std::endl;
        QDir dbdir(dbpath);
        dbdir.removeRecursively();
    }
    BOWDatabase db(dbpath.toStdString(), MAX_FEAT_CNT);

    ContourlettExtractor dce(pano);
    DatabaseEntry de(dce, pano, PANORAMA_FOV, DatabaseEntry::FeaType::BOTH);
    db.addSearchEntry(de);

    //save database
    QFile db_xml("singleQuery.xml");
    if(!db_xml.open(QIODevice::WriteOnly | QIODevice::Text))
    {
        qDebug("Failed to open file for writing.");
    }
    QXmlStreamWriter xml(&db_xml);
    xml.writeStartDocument();
    db.saveToXmlStream(xml);
    xml.writeEndDocument();
    db_xml.close();
    //end saving database

    QFileInfo queryInfo(query);
    QString photoName = queryInfo.baseName();

    FieldOfViewUtil fovu(query, fov, dce.width());
    ContourlettExtractor qce(fovu.fitToFOV());
    DatabaseEntry q(qce, query, fov, DatabaseEntry::FeaType::BOTH);

    //create dir voting array
    size_t dirsize = (size_t)360 * (size_t)db.size() + (size_t)360;
    float *dir = new float[dirsize];
    std::fill(dir, dir + dirsize, 0.0);

    clock_t t;
    t = clock();
    std::vector< std::pair<int, float> > res = db.query(q, dir);
    t = clock() - t;
    //double time = ((float)t)/CLOCKS_PER_SEC;

    //extract point cloud for ICP from query image
    std::vector< std::pair<int, int> > queryPoints = qce.getHorizonLinePoints(fov);
    QueryResult win;

    prepareDirAndLocResults(db, dir, win);
    try{
        runICP(db, dir, win, queryPoints, 100, photoName, fov, "", saveVtk);
    }
    catch(std::exception &e)
    {
        std::cerr << e.what() << std::endl;
    }

    //printDirAndLocResultsRaw(db, dir, win, 100, photoName);

    QueryResult::iterator best = win.begin();
    if (best->second.icpAngle != -1)
    {
        float yaw, pitch, roll;
        compute3DRotation(dce.width(), dce.height(),
                          qce.width(), qce.height(),
                          best->second.icpAngle, best->second.T, fov,
                          yaw, pitch, roll);
        std::cout << "query: " << query.toStdString() << " matchRatio " << best->second.matchRatio << " yaw icp: "
                  << best->second.icpAngle << ", ypr: " << yaw << ", "
                  << pitch << ", " << roll << std::endl;
    }
    else
    {
        throw std::runtime_error("Unable to find the pose, ICP was not sucessful.");
    }

    delete []dir;
    dir = NULL;

    //delete the temporary database
    if (dbinfo.exists())
    {
        QDir dbdir(dbpath);
        dbdir.removeRecursively();
    }

}

int runBatch(const char *batchfile, const char *placeSetName, BOWDatabase *d,
             DatabaseEntry::FeaType ft = DatabaseEntry::FeaType::BOTH,
             bool saveVtk = false)
{
    QList<Place> places;
    QHash<QString, Place*> placeMap;

    QString qPlaceSetName(placeSetName);
    QFileInfo placeSetInfo(qPlaceSetName);
    if (placeSetInfo.suffix() == "xml")
    {
        //load as xml
        PlaceSet ps(placeSetInfo.fileName());
        places = ps.getAll();
    }
    else
    {
        //load as csv list
        ifstream pf(placeSetName, ios::in);
        std::string str;
        std::vector<std::string> comp;
        while (std::getline(pf, str))
        {
            comp.clear();
            char *dup = strdup(str.c_str());
            char *pch = strtok(dup, ",");
            for (int i = 0; i < 3; ++i)
            {
                if (pch != NULL)
                {
                    comp.push_back(std::string(pch));
                    strtok(NULL, ",");
                }
            }
            free(dup); dup = NULL;
            Place p;
            try
            {
                p.name = QString(comp.at(0).c_str());
                p.latitude = QString(comp.at(1).c_str()).toFloat();
                p.longitude = QString(comp.at(2).c_str()).toFloat();
            }
            catch(std::out_of_range &oor)
            {
                std::cerr << "Unable to parse place. Skipping.";
            }
            places.push_back(p);
        }
    }

    for (Place &p : places)
    {
        placeMap.insert(p.name, &p);
    }

    QString qBatchFile(batchfile);
    QFile inputFile(qBatchFile);
    if (inputFile.open(QIODevice::ReadOnly))
    {
       QTextStream in(&inputFile);
       while (!in.atEnd())
       {
          QString queryLine = in.readLine();
          QStringList queryList = queryLine.split(":");
          if (queryList.size() != 3)
          {
              throw std::runtime_error("Each line of query list must be in format: "
                                       "<query img path>:<hfov>:<ICP candidates count>");
          }
          QString query = queryList[0];
          float fov = queryList[1].toFloat();
          int limit = queryList[2].toInt();
          processQuery(query.toStdString(), *d, fov, qPlaceSetName, places, placeMap, true, limit, ft, saveVtk);
       }
       inputFile.close();
    }

    return EXIT_SUCCESS;
}

int runBatch(const char *batchfile, vector<string> dbnames,
             const char *placeSetName,
             DatabaseEntry::FeaType ft = DatabaseEntry::FeaType::BOTH,
             bool saveVtk = false)
{
    //load database
    BOWDatabase *d;
    int elapsed;
    QTime myTimer;
    myTimer.start();
    //d = BOWDatabase::loadWithLXML2(dbnames);
    //elapsed = myTimer.elapsed();
    //qDebug() << "Database loaded with libxml2: " << elapsed / 1000.0 << ", size " << d->size();
    /*delete d;
    myTimer.restart();*/
    //d = BOWDatabase::load(dbnames);
    d = new BOWDatabase(dbnames[0], MAX_FEAT_CNT);
    elapsed = myTimer.elapsed();
    qDebug() << "Database loaded with QTXmlStreamReader: " << elapsed / 1000.0;
    int res = runBatch(batchfile, placeSetName, d, ft, saveVtk);
    delete d;
    return res;
}

#ifdef USE_LIBHTTPSERVER

/*****************************************************/
/*** REST SERVICE THROUGH WEB SERVER FUNCTIONALITY ***/
using namespace httpserver;

class query_resource : public http_resource<query_resource> {
public:
    query_resource(vector<string> dbnames, QString placeSetName);
    ~query_resource();
    void render(const http_request&, http_response**);
    void set_some_data(const std::string &s) {data = s;}
    QImage imageFromJson(const QVariantMap &result);
    std::string data;

private:
    BOWDatabase *d;
    QString placeSetName;
    QList<Place> places;
    QHash<QString, Place*> placeMap;

    QAtomicInt index;
};

query_resource::query_resource(vector<string> dbnames, QString _placeSetName)
    :placeSetName(_placeSetName)
{
    index = 0;
    //load places
    PlaceSet ps(placeSetName);
    places = ps.getAll();

    //initialize place map to lookup place GPS more efficiently
    for (Place &p : places)
    {
        placeMap.insert(p.name, &p);
    }

    QFileInfo psInfo(placeSetName);
    //load database
    int elapsed;
    QTime myTimer;
    myTimer.start();
    std::string db_path = psInfo.baseName().toStdString();
    d = new BOWDatabase(dbnames[0], MAX_FEAT_CNT);
    elapsed = myTimer.elapsed();
}

query_resource::~query_resource()
{
    delete d;
    d = NULL;
}

QImage query_resource::imageFromJson(const QVariantMap &result)
{
    int width = result["width"].toInt();
    int height = result["height"].toInt();
    qDebug() << "Creating image: " << width << ", " << height;
    QImage res(width, height, QImage::Format_RGB32);
    res.fill(qRgb(0,0,0));

    QVariantMap x = result["coordinates"].toMap();
    for (int i = 1; i <= width; ++i)
    {
        QString idx = QString::number(i);
        QVariantList xc = x[idx].toList();
        int y = height - xc[0].toInt();
        res.setPixel(i-1, y, qRgb(255, 255, 255));
    }
    return res;
}

//using the render method you are able to catch each type of request you receive
void query_resource::render(const http_request& req, http_response** res)
{
    //it is possible to store data inside the resource object that can be altered
    //through the requests
    QTime myTimer;
    myTimer.start();

    std::string cnt = req.get_content();

    http_response_builder response_b("{\"msg\" : OK}", 200);

    QByteArray json(cnt.c_str());
    QJson::Parser parser;
    bool ok;

    if (json.size() > 0)
    {
        QVariantMap result = parser.parse (json, &ok).toMap();
        if (!ok) {
            response_b = http_response_builder("{\"msg\" : Error during parsing JSON input}", 500);
        }
        else
        {
            qDebug() << "Processing image...";

            QImage image = imageFromJson(result);

            int num = index.fetchAndAddAcquire(1);
            QString imgname = QString("/tmp/hloc-query-").append(QString::number(num)).append(".png");

            float fov = result["fov"].toFloat();
            qDebug() << "FOV: " << fov;

#ifdef TIME_EXPERIMENTS
            //set icp candidates to 0
            QByteArray result_json = ::processQuery(image, imgname, *d, fov, placeSetName, places, placeMap, false, 0, DatabaseEntry::FeaType::BOTH, false);
#else
            //set icp candidates to 100
            QByteArray result_json = ::processQuery(image, imgname, *d, fov, placeSetName, places, placeMap, false, 100, DatabaseEntry::FeaType::BOTH, false);
#endif
            //zip and base64 encode the json output
            ByteArray result_json_b(result_json.toStdString().c_str(), result_json.length());
            StringZip sz(result_json_b);
            ByteArray zipped = sz.zip();
            QByteArray zipped_b(zipped.getData(), zipped.size());
            QByteArray zipped_b_base64 = zipped_b.toBase64();
            std::string zipped_b_base64_str = std::string(zipped_b_base64.constData());
            response_b = http_response_builder(zipped_b_base64_str, 200);
        }
    }

    //it is possible to send a response initializing an http_string_response
    //that reads the content to send in response from a string.

    response_b = response_b.with_header("Access-Control-Allow-Origin", "*");
    response_b = response_b.with_header("Access-Control-Allow-Headers", "Content-Type");
    *res = new http_response(response_b.string_response());
    qDebug() << "Query elapsed: " << myTimer.elapsed() / 1000.0;
}

int runAsHttpService(vector<string> dbnames, const char *placeSetName)
{
    //it is possible to create a webserver passing a great number of parameters.
    //In this case we are just passing the port and the number of thread running.
    webserver ws = create_webserver(8080).max_threads(5);

    QString psn = placeSetName;
    query_resource hwr(dbnames, psn);
    //this way we are registering the hello_world_resource to answer for the endpoint
    //"/hello". The requested method is called (if the request is a GET we call the render_GET
    //method. In case that the specific render method is not implemented, the generic "render"
    //method is called.
    ws.register_resource("/hloc-query", &hwr, true);

    //This way we are putting the created webserver in listen. We pass true in order to have
    //a blocking call; if we want the call to be non-blocking we can just pass false to the
    //method.
    bool res = ws.start(true);
    return res == true ? 0 : 1;
}

/*************** END OF REST SERVICE *****************/
/*****************************************************/

#endif //USE_LIBHTTPSERVER

ArgumentParser buildArgumentParser(std::string app_name)
{
    ArgumentParser ap(app_name,
                      "HorizonLOCate tool implements bag-of-words retrieval "
                      "on horizon lines extracted from mountainous like "
                      "photos, introduced by Baatz et al.: Image Based "
                      "Geo-Localization in the Alps.");

    vector<Parameter> t_par_list;
    t_par_list.push_back(Parameter("list",
                              "list with each line containing horizon "
                              "(in binary format .bin) file name "
                              "which will be added into the database."));
    t_par_list.push_back(Parameter("database_name",
                              "Name of the output database."));

    ap.addArgument("tl", "train-list", t_par_list,
                   "Trains the database of contourletts from detected horizon "
                   "lines. When training big databases, the database will be "
                   "divided so that each part of the database contains 100K"
                   "horizons (places). Input horizons to be included into the "
                   "database will be taken from the list.", true);


    vector<Parameter> d_par;
    d_par.push_back(Parameter("place set xml file",
                              "XML places file to be used to resolve the "
                              "locations of the place names stored in the "
                              "database -- the <place_set_name> shall be the "
                              "same which was used to generate the panorama "
                              "data using silhouettes tool."));
    d_par.push_back(Parameter("dbname", "List of database files to be loaded. "
                                        "More databases can be loaded at "
                                        "once.", true));
    ap.addArgument("d", "database", d_par,
                   "Loads the database previously trained with --train option."
                   , true);

    ap.addArgument("s", "stdin", vector<Parameter>(),
                   "This option must be combined with --batch option. When"
                   "using this option, the tool loads the databases from "
                   "standard input - ie. database files has to be fed through"
                   "stdin. After last database there must be EOF. Afterwards"
                   "the tool starts to process the queries from the batch "
                   "file.", true);

    vector<Parameter> tc_par;
    tc_par.push_back(Parameter("number of threads",
                               "Number of thread greater or equal to one."));
    ap.addArgument("c", "thread-count", tc_par,
                   "Sets maximum number of threads to be used during ICP"
                   "verification step.", true);

    vector<Parameter> f_par;
    f_par.push_back(Parameter("panorama-image", "Name of the black-and-white "
                              "panorama image. Terrain must be white,"
                              "sky must be black."));
    f_par.push_back(Parameter("query-image", "Name of the black-and-white "
                              "query image. Terrain must be white,"
                              "sky must be black."));
    f_par.push_back(Parameter("field-of-view", "fiel-of-view of the image "
                              "in radians."));

    ap.addArgument("f", "find-pose", f_par,
                   "Finds pose estimation of single query photograph in"
                   "a single panorama.", true);

    vector<Parameter> cf_par;
    cf_par.push_back(Parameter("type", "Type of the features to be used. 0 for "
                                      "2.5 features only, 1 for 10 features "
                                      "only, any other number for both types "
                                      "together."));
    ap.addArgument("cf", "contourlett-features", cf_par,
                   "Restricts feature type. Defaults to using both "
                   "types of features together.", true);

    #ifdef USE_LIBHTTPSERVER
    ap.addArgument("r", "rest-service", vector<Parameter>(),
                   "Runs the program as a HTTP server listening on port"
                   "8080, accepting REST requests.", true);
    #endif //USE_LIBHTTPSERVER
    vector<Parameter> b_par;
    b_par.push_back(Parameter("query file",
                              "File containing list of queries to be batch "
                              "processed. The format of each line must be:"
                              "<path to image>:<hfov>:<ICP count>. "
                              "<path to image> is path to image containing white"
                              "color for foreground and black color for "
                              "background (defining the horizon line). <hfov>"
                              "is horizontal field-of-view of the image. "
                              "<ICP count> is number defining how many top "
                              "candidates will be verified using ICP."));
    ap.addArgument("b", "batch", b_par,
                   "Runs the tool in batch mode. Usable only together with "
                   "--database option. Loads the database, and processes "
                   "the query file line by line. Results are printed to "
                   "stdout.", true);

    vector<Parameter> i_par;
    ap.addArgument("i", "icp-save", i_par,
                   "When this flag is used, the ICP result is saved into VTK"
                   "files in the working directory.", true);

    return ap;
}


int main(int argc, char *argv[])
{
    ArgumentParser ap = buildArgumentParser(argv[0]);

    //QApplication app(argc, argv);

    if (ap.parse(argc, (const char **)argv))
    {
        Argument *tl_arg = ap.argumentByShortname("tl");
        Argument *d_arg = ap.argumentByShortname("d");
        Argument *c_arg = ap.argumentByShortname("c");
        Argument *f_arg = ap.argumentByShortname("f");
        Argument *b_arg = ap.argumentByShortname("b");
        Argument *s_arg = ap.argumentByShortname("s");
        Argument *r_arg = ap.argumentByShortname("r");
        Argument *cf_arg = ap.argumentByShortname("cf");
        Argument *i_arg = ap.argumentByShortname("i");

        bool run = false;
        bool batch = false;
        vector<string> dbnames;
        string places_xml_file;
        string batch_query_file;

        DatabaseEntry::FeaType featureType = DatabaseEntry::FeaType::BOTH;
        if (cf_arg->exists())
        {
            bool ok;
            int ft = QString(cf_arg->getResult()[0].c_str()).toInt(&ok);
            if (!ok)
                throw std::runtime_error("Unable to parse feature type. It must"
                                         "be int number.");
            if (ft == 0)
            {
                featureType = DatabaseEntry::FeaType::ONLY_2_5;
            }
            else if (ft == 1)
            {
                featureType = DatabaseEntry::FeaType::ONLY_10;
            }
            else
            {
                featureType = DatabaseEntry::FeaType::BOTH;
            }
        }
        if (tl_arg->exists())

        {
            //train
            vector<string> t_res = tl_arg->getResult();
            const char *list = t_res[0].c_str();
            const char *dbname = t_res[1].c_str();
            return train_list(list, dbname);
        }
        if (d_arg->exists())
        {
            //load database
            run = true;
            vector<string> d_res = d_arg->getResult();
            places_xml_file = d_res[0];
            dbnames = vector<string>(d_res.begin() + 1, d_res.end());
        }
        if (c_arg->exists())
        {
            //set maximum thread count
            int threadCount = QString(c_arg->getResult()[0].c_str()).toInt();
            QThreadPool::globalInstance()->setMaxThreadCount(threadCount - 1);
        }
        if (f_arg->exists())
        {
            //find pose estimation for single photoparam
            QString panorama_img = QString(f_arg->getResult()[0].c_str());
            QString query_img = QString(f_arg->getResult()[1].c_str());
            bool ok;
            float fov = QString(f_arg->getResult()[2].c_str()).toFloat(&ok);
            if (!ok)
            {
                throw std::runtime_error("Unable to parse field-of-view value."
                                         "Field-of-view must be float value in"
                                         "radians");
            }
            findPose(panorama_img, query_img, fov, i_arg->exists());
        }
#ifdef USE_LIBHTTPSERVER
        if (r_arg->exists())
        {
            return runAsHttpService(dbnames, places_xml_file.c_str());
        }
#endif //USE_LIBHTTPSERVER
        if (b_arg->exists())
        {
            batch = true;
            batch_query_file = b_arg->getResult()[0];
        }

        if (run && batch)
        {
            return runBatch(batch_query_file.c_str(),
                            dbnames,
                            places_xml_file.c_str(), featureType,
                            i_arg->exists());
        }
        else if (run)
        {
            return rundb(dbnames, places_xml_file.c_str(), featureType,
                         i_arg->exists());
        }
        else if (s_arg->exists() && batch)
        {
            BOWDatabase *db = loadDbFromStdin();
            int res = runBatch(batch_query_file.c_str(),
                               places_xml_file.c_str(), db, featureType,
                               i_arg->exists());
            delete db;
            return res;
        }


    }
    return EXIT_SUCCESS;
}
