/**
 *  c2gdist
 *  Tool for calculating distances between rotations.
 *
 *  Author: Jan Brejcha <ibrejcha@fit.vutbr.cz>, <brejchaja@gmail.com>
 *  Copyright (C) 2018  Jan Brejcha
 *
 *  OPEN SOURCE LICENCE VUT V BRNĚ
 *  Verze 1.
 *  Copyright (c) 2010, Vysoké učení technické v Brně, Antonínská 548/1, PSČ 601 90
 *  -------------------------------------------------------------------------------
 */

#include <cstdio>
#include <cstdlib>
#include <iostream>
#include <cmath>
#include <stdexcept>

#include <QString>
#include <QFileInfo>
#include <QDebug>

#include <EulerZYZ.h>
#include <vector>

#include "CamUtil.h"
#include "XMLUtil.h"

#include "parameter.h"
#include "argumentparser.h"

using namespace argpar;

void printVec(Vector3f v)
{
    std::cout << v.x() << " " << v.y() << " " << v.z() << std::endl;
}

ArgumentParser buildArgumentParser()
{
    const char *appDescription = "Tool for manipiulating with rotation "
                                 "matrices - calculation of euclidian distance "
                                 "between ground truth and calculated rotation "
                                 "by Match3D_visu. It also supports options "
                                 "for printing and converting the rotation "
                                 "matrix to euler angles.";

    ArgumentParser ap("match3D_C2GDist", std::string(appDescription));

    ////////////////////////////////////////////////////////////////////////////
    /// distance of rotationC2G.xml and calculated Match3D_visu par file
    ////////////////////////////////////////////////////////////////////////////

    /*std::vector<Parameter> distParams;
    distParams.push_back(Parameter("rotationC2G.xml file path",
                                   "The XML file containing the ground truth"
                                   "rotation matrix."));
    distParams.push_back(Parameter("calculated m3D_params.par file path",
                                   ".par file containing the match3D_visu "
                                   "result"));
    ap.addArgument("d", "distance", distParams,
                   "Measures the euclidian distance "
                   "between ground thruth rotation and calculated "
                   "rotation of Match3D_visu.", true);*/

    std::vector<Parameter> sd_params;
    sd_params.push_back(Parameter("Z1", "Z component of the first ZYZ angle"));
    sd_params.push_back(Parameter("Y1", "Y component of the first ZYZ angle"));
    sd_params.push_back(Parameter("Z1p", "Second Z component of the first ZYZ angle"));

    sd_params.push_back(Parameter("Z2", "Z component of the second ZYZ angle"));
    sd_params.push_back(Parameter("Y2", "Y component of the second ZYZ angle"));
    sd_params.push_back(Parameter("Z2p", "Second Z component of the second ZYZ angle"));

    ap.addArgument("sd", "spherical-distance", sd_params,
                    "Measures spherical distance between two ZYZ euler angles.",
                    true);

    ap.addArgument("ed", "euclidean-distance", sd_params,
                    "Measures euclidean distance between two ZYZ euler angles."
                    "The euler angles are transformed into the yaw, pitch, roll"
                    "and the distance on shortest angles (<180deg) is calculated.",
                    true);

    std::vector<Parameter> edm_params;
    edm_params.push_back(Parameter("rotationC2G.xml file path",
                                   "The XML file containing the ground truth"
                                   "rotation matrix."));

    edm_params.push_back(Parameter("Z2", "Z component of the second ZYZ angle"));
    edm_params.push_back(Parameter("Y2", "Y component of the second ZYZ angle"));
    edm_params.push_back(Parameter("Z2p", "Second Z component of the second ZYZ angle"));

    ap.addArgument("edm", "euclidean-distance-matrix", edm_params,
                   "Measures euclidean distance between rotationC2G matrix and"
                   "ZYZ euler angle.", true);


    ////////////////////////////////////////////////////////////////////////////
    /// print the rotation matrix to stdout
    ////////////////////////////////////////////////////////////////////////////

    std::vector<Parameter> printMatrixParams;
    printMatrixParams.push_back(Parameter("matrix XML file path",
                                          "XML file containing the definition"
                                          "of the rotation matrix."));
    ap.addArgument("prm", "print-rotation-matrix", printMatrixParams,
                   "Prints given rotation matrix to stdout.", true);

    ////////////////////////////////////////////////////////////////////////////
    /// convert the rotation matrix to yaw, pitch, roll angles
    ////////////////////////////////////////////////////////////////////////////

    std::vector<Parameter> eaZYZParams;
    eaZYZParams.push_back(Parameter("matrix XML file path",
                                    "XML file containing the definition"
                                    "of the rotation matrix."));
    ap.addArgument("a", "angles", eaZYZParams,
                   "Converts the rotation matrix to yaw, pitch, roll angles "
                   "in the panorama.", true);

    ap.addArgument("ea", "euler-angles", eaZYZParams,
                   "Converts the rotation matrix to Euler ZYZ angles.", true);

    std::vector<Parameter> ma_par;
    ma_par.push_back(Parameter("params.par file path", "match3D params.par file"
                                                  "containing results of m3D."));

    /* FIXME: add support for this - loading of par files without bablib
     * ap.addArgument("ma", "m3D-angles", ma_par,
                   "Loads result pose found by match3D from .par file and"
                   "converts it to three angles (yaw, pitch, roll).", true);
     */

    ////////////////////////////////////////////////////////////////////////////
    /// convert panorama alpha, beta, gamma angles to rotation matrix
    ////////////////////////////////////////////////////////////////////////////

    std::vector<Parameter> atm_par;
    atm_par.push_back(Parameter("yaw", "yaw angle in radians"));
    atm_par.push_back(Parameter("pitch", "pitch angle in radians"));
    atm_par.push_back(Parameter("roll", "roll angle in radians"));
    atm_par.push_back(Parameter("output-name", "name of output xml file with "
                                               "rotation matrix."));

    ap.addArgument("atm", "angles-to-matrix", atm_par,
                   "Converts panorama angles to rotation matrix.", true);

    std::vector<Parameter> etm_par;
    etm_par.push_back(Parameter("Z1", "Z component of the first ZYZ angle"));
    etm_par.push_back(Parameter("Y1", "Y component of the first ZYZ angle"));
    etm_par.push_back(Parameter("Z1p", "Second Z component of the first ZYZ angle"));
    etm_par.push_back(Parameter("output-name", "name of output xml file with "
                                               "rotation matrix."));

    ap.addArgument("etm", "euler-angles-to-matrix", etm_par,
                   "Converts euler angles to rotation matrix.", true);

    return ap;
}

Matrix3f loadMatrixFromFile(std::string matrixFile)
{
    QString c2gref_xml_path(matrixFile.c_str());
    QFileInfo c2gref_info(c2gref_xml_path);
    if (!c2gref_info.exists())
    {
        throw std::runtime_error("Matrix file does not exist: " + c2gref_xml_path.toStdString());
    }
    Matrix3f mat;

    QDomDocument doc = XMLUtil::loadQDomDocFromFile(c2gref_xml_path);
    QDomElement e = doc.documentElement();
    if (e.tagName() == XML_ELEMENT_NAME)
    {
        e = e.firstChild().toElement();
        for (int i = 0; i < 3; ++i)
        {
            float x = e.attribute("x0", "0.0").toFloat();
            float y = e.attribute("x1", "0.0").toFloat();
            float z = e.attribute("x2", "0.0").toFloat();
            mat.row(i) << x, y, z;
            //std::cout << x << ", " << y << ", " << z << std::endl;
            e = e.nextSibling().toElement();
        }
    }
    else
    {
        throw std::runtime_error("The XML file: "
                                 + c2gref_xml_path.toStdString()
                                 + " does not contain XML element: "
                                 + XML_ELEMENT_NAME);
    }
    return mat.transpose();
}

void saveMatrixToXMLFile(Matrix3f mat, QString fileName)
{
    QDomDocument doc("ObjectXML");
    QDomElement res = doc.createElement(XML_ELEMENT_NAME);
    for (int j = 0; j < 3; ++j)
    {
        QDomElement column = doc.createElement("C" + QString::number(j));
        column.setAttribute("x0", QString::number(mat(1,j)));
        column.setAttribute("x1", QString::number(mat(2,j)));
        column.setAttribute("x2", QString::number(mat(0,j)));
        res.appendChild(column);
    }
    doc.appendChild(res);

    QFile dataFile(fileName);
    if (dataFile.open(QIODevice::WriteOnly))
    {
        QTextStream dataStream(&dataFile);
        doc.save(dataStream, 2);
        dataFile.flush();
        dataFile.close();
    }
    else
    {
        throw std::runtime_error("Cannot save XML file: "
                                 + fileName.toStdString());
    }
}

/* @deprecated
void printDistance(Argument *distanceArg)
{
    std::vector<std::string> distanceArgs = distanceArg->getResult();

    QString calculated_params_path(distanceArgs[1].c_str());

    //load Ground Thruth
    Matrix3f C2GRef = loadMatrixFromFile(distanceArgs[0]);

    //load calculated result
    ParamSet paramSet;
    paramSet.load(calculated_params_path);
    Vec3d rotCalc = paramSet.get<Vec3d>("result.rotOpti");

    //calculate rotation from ground thruth matrix
    Vec3d rotOpti = EulerZYZ::angles(C2GRef) / Vec3d(2 * M_PI, M_PI, 2* M_PI);

    //calculate distance
    double res = dist(rotCalc, rotOpti);

    std::cout << res << std::endl;
}*/

void printRotationMatrix(Argument *matrixArg)
{
    std::vector<std::string> matrixArgs = matrixArg->getResult();
    Matrix3f C2GRef = loadMatrixFromFile(matrixArgs[0]);
    std::cout << C2GRef << std::endl;
}

Matrix4f projTransfoFromRot(Matrix3f C2GRef)
{
    Matrix4f projTransfo = Matrix4f::Identity();
    for (int i = 0; i < 3; ++i)
    {
        projTransfo.row(i) << C2GRef(i, 0), C2GRef(i, 1), C2GRef(i, 2), 0;
    }
    return projTransfo;
}

void printAngles(Argument *matrixArg)
{
    std::vector<std::string> matrixArgs = matrixArg->getResult();
    Matrix3f C2GRef = loadMatrixFromFile(matrixArgs[0]);
    Matrix4f projTransfo = projTransfoFromRot(C2GRef);

    double yaw, pitch, roll;
    CamUtil::rotationAngles(projTransfo.cast<double>(), yaw, pitch, roll);
    //Vec3d angles = EulerZYZ::angles(C2GRef) / Vec3d(2 * M_PI, M_PI, 2* M_PI);
    Vector3d angles(yaw, pitch, roll);
    printVec(angles.cast<float>());
}

void printEulerZYZAngles(Argument *eaZYZArg)
{
    std::vector<std::string> args = eaZYZArg->getResult();
    Matrix3f C2GRef = loadMatrixFromFile(args[0]);
    //std::cout << C2GRef << std::endl;
    Vector3f angles = EulerZYZ::angles(C2GRef);
    printVec(angles);
}

/* FIXME: add support for this - loading of par files without bablib
 * void printAnglesMatch3D(Argument *maArg)
{
    QString paramFile = QString(maArg->getResult()[0].c_str());

    ParamSet params;
    params.load(paramFile);
    Vec3 rot_opti = params.get("result.rotOpti", Vec3(-1.0, -1.0, -1.0));

    rot_opti = M_PI * Vec3(2.0, 1.0, 2.0) * rot_opti;
    Matrix3 G2C = EulerZYZ::matrix(rot_opti).inv();

    double yaw, pitch, roll;
    CamUtil::rotationAngles(Proj3d(ProjTransfo(Vec3(0,0,0), G2C)), yaw, pitch, roll);
    Vec3d angles(yaw, pitch, roll);
    printVec(angles);
}*/

double calculateSphericalDistance(Vector3f p1, Vector3f p2)
{
    return acos(p1.dot(p2));
}

void printSphericalDistance(Argument *spArg)
{
    float a1, b1, c1, a2, b2, c2;
    std::vector<std::string> res = spArg->getResult();
    a1 = QString(res[0].c_str()).toFloat();
    b1 = QString(res[1].c_str()).toFloat();
    c1 = QString(res[2].c_str()).toFloat();

    a2 = QString(res[3].c_str()).toFloat();
    b2 = QString(res[4].c_str()).toFloat();
    c2 = QString(res[5].c_str()).toFloat();

    Matrix3f m1 = EulerZYZ::matrix(a1, b1, c1);
    Matrix3f m2 = EulerZYZ::matrix(a2, b2, c2);

    Vector3f p(1.0f, 0.0f, 0.0f);
    Vector3f p1 = m1 * p;
    Vector3f p2 = m2 * p;

    std::cout << calculateSphericalDistance(p1, p2) << std::endl;
}

/**
 * @brief angleToBaseInterval
 * converts the angle in radians from any interval to the interval <0,2*pi)
 * @param angle input angle from real numbers
 * @return  angle in the interval <0,2*pi).
 */
double angleToBaseInterval(double angle)
{
    //convert the angle to positive interval
    int two_pi = 360;
    //to degrees
    int d_angle = (int)round((angle * 180.0) / M_PI);
    while (d_angle < 0)
    {
        d_angle += two_pi;
    }
    //positive angle modulo two pi to get the <0, 360) degrees interval.
    int n_angle = d_angle % two_pi;
    //back to radians
    return (n_angle * M_PI) / 180.0;
}


/**
 * @brief angleToBaseHalfInterval
 * converts the angle in radians from any interval to the interval <0,pi)
 * @param angle input angle from real numbers
 * @return  angle in the interval <0,pi).
 */
double angleToBaseRadInterval(double angle)
{
    //convert the angle to positive interval
    //to degrees
    /*int d_angle = (int)round((angle * 180.0) / M_PI);
    while (d_angle < 0)
    {
        d_angle += pi_rad;
    }
    //positive angle modulo two pi to get the <0, 180) degrees interval.
    int n_angle = d_angle % pi_rad;
    //back to radians
    return (n_angle * M_PI) / 180.0;*/
    while (angle < 0)
    {
        angle += 2*M_PI;
    }
    return angle;
}

double calculateShortestEuclideanDistance(float yaw1, float pitch1, float roll1,
                                        float yaw2, float pitch2, float roll2)
{
    yaw1 = angleToBaseRadInterval(yaw1);
    pitch1 = angleToBaseRadInterval(pitch1);
    roll1 = angleToBaseRadInterval(roll1);
    yaw2 = angleToBaseRadInterval(yaw2);
    pitch2 = angleToBaseRadInterval(pitch2);
    roll2 = angleToBaseRadInterval(roll2);

    float yaw_d = fabs(yaw1 - yaw2);
    yaw_d = yaw_d > M_PI ? 2 * M_PI - yaw_d : yaw_d;
    float pitch_d = fabs(pitch1 - pitch2);
    pitch_d = pitch_d > M_PI ? 2 * M_PI - pitch_d : pitch_d;
    float roll_d = fabs(roll1 - roll2);
    roll_d = roll_d > M_PI ? 2 * M_PI - roll_d : roll_d;

    Vector3f p(yaw_d, pitch_d, roll_d);
    return p.norm();
}
double calculateShortestEuclideanDistance(Matrix3f m1, Matrix3f m2)
{
    Matrix4f projTransfo1 = projTransfoFromRot(m1);
    Matrix4f projTransfo2 = projTransfoFromRot(m2);

    float yaw1, pitch1, roll1, yaw2, pitch2, roll2;
    //CamUtil::rotationAngles(Proj3d(projTransfo1), yaw1, pitch1, roll1);
    //CamUtil::rotationAngles(Proj3d(projTransfo2), yaw2, pitch2, roll2);
    EulerZYZ::angles(m1, yaw1, pitch1, roll1);
    EulerZYZ::angles(m2, yaw2, pitch2, roll2);
    return calculateShortestEuclideanDistance(yaw1, pitch1, roll1,
                                               yaw2, pitch2, roll2);
}
void printEuclideanDistanceWithGTMatrix(Argument *edmArg)
{
    std::vector<std::string> res = edmArg->getResult();
    Matrix3f C2G = loadMatrixFromFile(res[0]);
    float a2, b2, c2;
    a2 = QString(res[1].c_str()).toFloat();
    b2 = QString(res[2].c_str()).toFloat();
    c2 = QString(res[3].c_str()).toFloat();
    Matrix3f m2 = EulerZYZ::matrix(a2, b2, c2);

    //double dist = calculateShortestEuclideanDistance(C2G, m2);

    double venturiDist = acos(((C2G.transpose() * m2).trace() - 1) / 2.0);

    std::cout << venturiDist << std::endl;
    //std::cout << dist << std::endl;
}

void printEuclideanDistance(Argument *edArg)
{
    float a1, b1, c1, a2, b2, c2;
    std::vector<std::string> res = edArg->getResult();
    a1 = QString(res[0].c_str()).toFloat();
    b1 = QString(res[1].c_str()).toFloat();
    c1 = QString(res[2].c_str()).toFloat();

    a2 = QString(res[3].c_str()).toFloat();
    b2 = QString(res[4].c_str()).toFloat();
    c2 = QString(res[5].c_str()).toFloat();

    Matrix3f m1 = EulerZYZ::matrix(a1, b1, c1);
    Matrix3f m2 = EulerZYZ::matrix(a2, b2, c2);

    double dist = calculateShortestEuclideanDistance(m1, m2);
    std::cout << dist << std::endl;
}

void checkFloatOK(bool ok)
{
    if (!ok)
        throw std::runtime_error("Unable to parse float value.");
}

void anglesToMatrix(Argument *atmArg)
{
    std::vector<std::string> res = atmArg->getResult();
    bool ok;
    float alpha = QString(res[0].c_str()).toFloat(&ok); checkFloatOK(ok);
    float beta = QString(res[1].c_str()).toFloat(&ok); checkFloatOK(ok);
    float gamma = QString(res[2].c_str()).toFloat(&ok); checkFloatOK(ok);
    QString output = QString(res[3].c_str());

    Matrix4f rot = CamUtil::transfoW2C(alpha, beta, gamma);
    //ProjTransfo rot = CamUtil::transfoW2C((float)(2.0 * M_PI - (alpha + M_PI/2.0)), beta, gamma);
    //ProjTransfo rot = CamUtil::transfoW2C((float)(M_PI/2.0 * 1.0f), 0.0f, (float)(M_PI * 1.0f));
    //ProjTransfo rot = CamUtil::transfoW2C((float)(0.0f), 0.0f, (float)(0.0f));
    Matrix3f res_rot = rot.block(0, 0, 3, 3).inverse();
    saveMatrixToXMLFile(res_rot, output);
}

void eulerAnglesToMatrix(Argument *atmArg)
{
    std::vector<std::string> res = atmArg->getResult();
    bool ok;
    float Z = QString(res[0].c_str()).toFloat(&ok); checkFloatOK(ok);
    float Y = QString(res[1].c_str()).toFloat(&ok); checkFloatOK(ok);
    float Zp = QString(res[2].c_str()).toFloat(&ok); checkFloatOK(ok);
    QString output = QString(res[3].c_str());

    Matrix3f m = EulerZYZ::matrix(Z, Y, Zp);
    saveMatrixToXMLFile(m, output);
}


int main(int argc, const char *argv[])
{
    ArgumentParser ap = buildArgumentParser();
    if (ap.parse(argc, argv))
    {
        /* @deprecated
         * TODO: remove
         * Argument *distanceArg = ap.argumentByName("distance");
        if (distanceArg->exists())
        {
            printDistance(distanceArg);
        }*/

        Argument *printMatrixArg = ap.argumentByName("print-rotation-matrix");
        if (printMatrixArg->exists())
        {
            printRotationMatrix(printMatrixArg);
        }

        Argument *eaArg = ap.argumentByName("angles");
        if (eaArg->exists())
        {
            printAngles(eaArg);
        }

        Argument *eaZYZArg = ap.argumentByName("euler-angles");
        if (eaZYZArg->exists())
        {
            printEulerZYZAngles(eaZYZArg);
        }

        Argument *sdArg = ap.argumentByName("spherical-distance");
        if (sdArg->exists())
        {
            printSphericalDistance(sdArg);
        }

        Argument *edArg = ap.argumentByName("euclidean-distance");
        if (edArg->exists())
        {
            printEuclideanDistance(edArg);
        }

        Argument *edmArg = ap.argumentByName("euclidean-distance-matrix");
        if (edmArg->exists())
        {
            printEuclideanDistanceWithGTMatrix(edmArg);
        }

        /* FIXME: add support for this - loading of par files without bablib
         * Argument *maArg = ap.argumentByName("m3D-angles");
        if (maArg->exists())
        {
            printAnglesMatch3D(maArg);
        }*/

        Argument *atmArg = ap.argumentByName("angles-to-matrix");
        if (atmArg->exists())
        {
            anglesToMatrix(atmArg);
        }

        Argument *etmArg = ap.argumentByName("euler-angles-to-matrix");
        if (etmArg->exists())
        {
            eulerAnglesToMatrix(etmArg);
        }
    }

	return EXIT_SUCCESS;
}
