/**
 *  Author: Jan Brejcha <ibrejcha@fit.vutbr.cz>, <brejchaja@gmail.com>
 *  Copyright (C) 2014  Jan Brejcha
 *
 *  OPEN SOURCE LICENCE VUT V BRNĚ
 *  Verze 1.
 *  Copyright (c) 2010, Vysoké učení technické v Brně, Antonínská 548/1, PSČ 601 90
 *  -------------------------------------------------------------------------------
 */

#ifndef CONTOURLETT
#define CONTOURLETT

#include <cstdio>
#include <climits>
#include <cmath>

#include <QDomDocument>

#include "LocalFeature.h"

#include <libxml/xmlreader.h>

#define CONTOURLETT_SAMPLES 8

///Count of the bins to which will be the contourlett quantized.
///It ALWAYS has to be a power of two.
///so the log2(BIN_COUNT) is always natural number (n).
#define BIN_COUNT 8

/**
 * Representation of contourlett feature.
 * The horizon outline is divided into countourletts with sliding window.
 * Each contourlett is then created from the horizon segment - each dimension
 * of this horizon segment vector is quantized separately into 2^3 = 8 bins.
 * From such quantized vector is then calculated binary word which is
 * transformed into single integer featureId.
 */
class Contourlett : public LocalFeature {
public:

	Contourlett(){}
    
    /**
     * Initializer of Contourlett. 
     * @param horizonSegment        Horizon segment vector.
     * @param pause                 Pause in pixels between samples.
     * @param start					Offset from where to start sampling this
     * 								contourlett from horizon segment vector.
     */
    Contourlett(double *horizonSegment, int pause, int start, int alpha);

    /**
     * Unpacked feature id getter.
     * @return feature id
     */
    inline virtual int featureId() const
    {
    	return Contourlett::unpackFeatureId(feature_id);
    }

    /**
	 * Packed feature id getter.
	 * @return feature id
	 */
	inline virtual int pFeatureId() const
	{
		return feature_id;
	}

    /**
	 * Converts the object into XMLElement of QtXML library
	 */
	virtual QDomElement toXMLElement(QDomDocument &doc);

	/**
	 * Initializes the object from XMLElement of QtXML library
	 */
	virtual void initFromXMLElement(QDomElement &e);

    virtual void initWithXmlStreamReader(QXmlStreamReader &xml);

    virtual int initWithXmlStreamReader(xmlTextReaderPtr &reader);

    virtual void saveToXmlStream(QXmlStreamWriter &xml);

	/**
	 * Packs alpha into feature id without alpha.
	 * @param feature_id 	feature id
	 * @param alpha			the direction of the contourlett center in degrees
	 * @return 				feature id with alpha packed in.
	 */
	static int packDirection(int feature_id, int alpha);

	/**
	 * Unpacks alpha from feature id with alpha.
	 * @param feature_id 	feature id
	 * @return 				alpha - the direction of the contourlett center in
	 * 						degrees.
	 */
	static int unpackDirection(int feature_id);

	/**
	 * Deletes alpha from feature id with alpha.
	 * @param feature_id 	feature id
	 * @return 				feature id without alpha.
	 */
	static int unpackFeatureId(int feature_id);

private:
    int feature_id;
    int alpha;

    double data[CONTOURLETT_SAMPLES];

    /**
     * Normalizes the contourlett to be able to quantize it.
     */
    void normalize(int pause);

    /**
     * Quantizes the contourlett into N bins.
     */
    void quantize();

    ///DEBUG methods
    void print();
};



#endif

