#!/usr/bin/env python3

import sys
import PyQt5
import json
import os
import pickle
from PyQt5.QtWidgets import QMainWindow
from PyQt5.QtWidgets import QApplication
from PyQt5.QtWidgets import QWidget
from PyQt5.QtWidgets import QVBoxLayout
from PyQt5.QtWidgets import QHBoxLayout
from PyQt5.QtWidgets import QGridLayout
from PyQt5.QtWidgets import QTabWidget
from PyQt5.QtWidgets import QFileDialog
from PyQt5.QtWidgets import QPushButton
from PyQt5.QtWidgets import QLineEdit
from PyQt5.QtWidgets import QLabel
from PyQt5.QtWidgets import QPlainTextEdit
from PyQt5.QtWidgets import QSpinBox
from PyQt5.QtWidgets import QErrorMessage
from PyQt5.QtWidgets import QCompleter
from PyQt5.QtCore import *

import src.globs as globs
import src.pcapparser as pcapparser
import src.method1 as method1
import src.method2 as method2
import src.method3 as method3

class Arguments:
    correct = False
    incorrect = False
    check = False
    ifile = None
    errfile = None
    checkfile = None
    clientfield = []
    serverfield = []
    maxlen = 6
    idir = None
    trid = ""
    json = False

class Table_widget(QWidget):
    
    def __init__(self, args, parent, app):
        super().__init__(parent)
        self.error_dialog = QErrorMessage(parent)
        self.app = app
        self.args = args
        with open(globs.fdir + "/fields.json") as f:
            content = f.read()
        jsn = json.loads(content)
        fields = list(jsn.keys())
        self.completer = QCompleter()
        self.model = QStringListModel()
        self.completer.setModel(self.model)
        self.model.setStringList(fields)

        self.layout = QVBoxLayout(self)
    
        self.tabs = QTabWidget()
        self.tabs.blockSignals(True)
        self.tab1 = QWidget()
        self.tab2 = QWidget()
        self.tab3 = QWidget()
        self.tabs.currentChanged.connect(self.tab_changed)
        self.activtab = 1

        self.tabs.addTab(self.tab1, "Correct")
        self.tabs.addTab(self.tab2, "Incorrect")
        self.tabs.addTab(self.tab3, "Check")

        self.create_correct_tab()
        self.create_incorrect_tab()
        self.create_check_tab()

        self.tabs.blockSignals(False)

        self.layout.addWidget(self.tabs)
        self.setLayout(self.layout)

    def create_correct_tab(self):
        self.tab1.dirtext = QLineEdit(self)
        if self.args.idir != None:
            self.tab1.dirtext.setText(self.args.idir)
        else:
            placeholder = "Directory with training dataset"
            self.tab1.dirtext.setPlaceholderText(placeholder)

        self.tab1.dirbtn = QPushButton("Open directory", self)
        self.tab1.dirbtn.resize(100, 40)
        self.tab1.dirbtn.clicked.connect(self.opendir)

        self.tab1.clientlbl = QLabel(self)
        self.tab1.clientlbl.setText("Client field")

        self.tab1.clienttext = QLineEdit(self)
        if len(self.args.clientfield) != 0:
            f = self.fields_to_str(self.args.clientfield)
            self.tab1.clienttext.setText(f)
        self.tab1.clienttext.setCompleter(self.completer)

        self.tab1.serverlbl = QLabel(self)
        self.tab1.serverlbl.setText("Server field")

        self.tab1.servertext = QLineEdit(self)
        if len(self.args.serverfield) != 0:
            f = self.fields_to_str(self.args.serverfield)
            self.tab1.servertext.setText(f)
        self.tab1.servertext.setCompleter(self.completer)

        self.tab1.maxlenlbl = QLabel(self)
        self.tab1.maxlenlbl.setText("Maximum command length:")

        self.tab1.maxlen = QSpinBox(self)
        self.tab1.maxlen.setValue(6)

        self.tab1.traidenlbl = QLabel(self)
        self.tab1.traidenlbl.setText("Training identificator:")

        self.tab1.traidentext = QLineEdit(self)
        if self.args.trid != "":
            self.tab1.traidentext.setText(self.args.trid)

        self.tab1.trainbtn = QPushButton("Train", self)
        self.tab1.trainbtn.setFixedSize(100, self.tab1.trainbtn.height())
        self.tab1.trainbtn.clicked.connect(self.correct)

        self.tab1.resultlbl = QLabel(self)
        self.tab1.resultlbl.setAlignment(Qt.AlignHCenter)

        self.tab1.grid1 = QGridLayout()
        self.tab1.grid1.addWidget(self.tab1.dirtext, 0, 0)
        self.tab1.grid1.addWidget(self.tab1.dirbtn, 0, 1)
        self.tab1.grid2 = QGridLayout()
        self.tab1.grid2.setColumnMinimumWidth(0, 214)
        self.tab1.grid2.addWidget(self.tab1.clientlbl, 0, 0)
        self.tab1.grid2.addWidget(self.tab1.clienttext, 0, 1)
        self.tab1.grid2.addWidget(self.tab1.serverlbl, 1, 0)
        self.tab1.grid2.addWidget(self.tab1.servertext, 1, 1)
        self.tab1.grid2.addWidget(self.tab1.maxlenlbl, 2, 0)
        self.tab1.grid2.addWidget(self.tab1.maxlen, 2, 1)
        self.tab1.grid2.addWidget(self.tab1.traidenlbl, 3, 0)
        self.tab1.grid2.addWidget(self.tab1.traidentext, 3, 1)
        self.tab1.btnbox = QHBoxLayout()
        self.tab1.btnbox.addWidget(self.tab1.trainbtn)
        self.tab1.resbox = QHBoxLayout()
        self.tab1.resbox.addWidget(self.tab1.resultlbl)

        self.tab1.box = QVBoxLayout()
        self.tab1.box.addLayout(self.tab1.grid1)
        self.tab1.box.addLayout(self.tab1.grid2)
        self.tab1.box.addLayout(self.tab1.btnbox)
        self.tab1.box.addLayout(self.tab1.resbox)

        self.tab1.setLayout(self.tab1.box)

    def create_incorrect_tab(self):
        self.tab2.errfiletext = QLineEdit(self)
        plh = "File containing communication with error"
        self.tab2.errfiletext.setPlaceholderText(plh)

        self.tab2.filebtn = QPushButton("Open file", self)
        self.tab2.filebtn.resize(100, 40)
        self.tab2.filebtn.clicked.connect(self.openfile)

        self.tab2.clientlbl = QLabel(self)
        self.tab2.clientlbl.setText("Client field")

        self.tab2.clienttext = QLineEdit(self)
        self.tab2.clienttext.setCompleter(self.completer)

        self.tab2.serverlbl = QLabel(self)
        self.tab2.serverlbl.setText("Server field")

        self.tab2.servertext = QLineEdit(self)
        self.tab2.servertext.setCompleter(self.completer)

        self.tab2.errmsglbl = QLabel(self)
        self.tab2.errmsglbl.setText("Error description:")

        self.tab2.errmsgtext = QLineEdit(self)

        self.tab2.traidenlbl = QLabel(self)
        self.tab2.traidenlbl.setText("Training identificator:")

        self.tab2.traidentext = QLineEdit(self)

        self.tab2.trainbtn = QPushButton("Train", self)
        self.tab2.trainbtn.setFixedSize(100, self.tab2.trainbtn.height())
        self.tab2.trainbtn.clicked.connect(self.incorrect)

        self.tab2.resultlbl = QLabel(self)
        self.tab2.resultlbl.setAlignment(Qt.AlignHCenter)

        self.tab2.grid1 = QGridLayout()
        self.tab2.grid1.addWidget(self.tab2.errfiletext, 0, 0)
        self.tab2.grid1.addWidget(self.tab2.filebtn, 0, 1)
        self.tab2.grid2 = QGridLayout()
        self.tab2.grid2.setColumnMinimumWidth(0, 214)
        self.tab2.grid2.addWidget(self.tab2.clientlbl, 0, 0)
        self.tab2.grid2.addWidget(self.tab2.clienttext, 0, 1)
        self.tab2.grid2.addWidget(self.tab2.serverlbl, 1, 0)
        self.tab2.grid2.addWidget(self.tab2.servertext, 1, 1)
        self.tab2.grid2.addWidget(self.tab2.errmsglbl, 2, 0)
        self.tab2.grid2.addWidget(self.tab2.errmsgtext, 2, 1)
        self.tab2.grid2.addWidget(self.tab2.traidenlbl, 3, 0)
        self.tab2.grid2.addWidget(self.tab2.traidentext, 3, 1)
        self.tab2.btnbox = QHBoxLayout()
        self.tab2.btnbox.addWidget(self.tab2.trainbtn)
        self.tab2.resbox = QHBoxLayout()
        self.tab2.resbox.addWidget(self.tab2.resultlbl)

        self.tab2.box = QVBoxLayout()
        self.tab2.box.addLayout(self.tab2.grid1)
        self.tab2.box.addLayout(self.tab2.grid2)
        self.tab2.box.addLayout(self.tab2.btnbox)
        self.tab2.box.addLayout(self.tab2.resbox)

        self.tab2.setLayout(self.tab2.box)

    def create_check_tab(self):
        self.tab3.errfiletext = QLineEdit(self)
        plh = "File containing communication for error checking"
        self.tab3.errfiletext.setPlaceholderText(plh)

        self.tab3.filebtn = QPushButton("Open file", self)
        self.tab3.filebtn.resize(100, 40)
        self.tab3.filebtn.clicked.connect(self.openfile)

        self.tab3.clientlbl = QLabel(self)
        self.tab3.clientlbl.setText("Client field")

        self.tab3.clienttext = QLineEdit(self)
        self.tab3.clienttext.setCompleter(self.completer)

        self.tab3.serverlbl = QLabel(self)
        self.tab3.serverlbl.setText("Server field")

        self.tab3.servertext = QLineEdit(self)
        self.tab3.servertext.setCompleter(self.completer)

        self.tab3.traidenlbl = QLabel(self)
        self.tab3.traidenlbl.setText("Training identificator:")

        self.tab3.traidentext = QLineEdit(self)

        self.tab3.checkbtn = QPushButton("Check", self)
        self.tab3.checkbtn.setFixedSize(100, self.tab3.checkbtn.height())
        self.tab3.checkbtn.clicked.connect(self.check)

        self.tab3.resultlbltext = QLabel(self)
        self.tab3.resultlbltext.setText("Results:")

        self.tab3.resultlbl = QPlainTextEdit(self)

        self.tab3.grid1 = QGridLayout()
        self.tab3.grid1.addWidget(self.tab3.errfiletext, 0, 0)
        self.tab3.grid1.addWidget(self.tab3.filebtn, 0, 1)
        self.tab3.grid2 = QGridLayout()
        self.tab3.grid2.setColumnMinimumWidth(0, 214)
        self.tab3.grid2.addWidget(self.tab3.clientlbl, 0, 0)
        self.tab3.grid2.addWidget(self.tab3.clienttext, 0, 1)
        self.tab3.grid2.addWidget(self.tab3.serverlbl, 1, 0)
        self.tab3.grid2.addWidget(self.tab3.servertext, 1, 1)
        self.tab3.grid2.addWidget(self.tab3.traidenlbl, 2, 0)
        self.tab3.grid2.addWidget(self.tab3.traidentext, 2, 1)
        self.tab3.btnbox = QHBoxLayout()
        self.tab3.btnbox.addWidget(self.tab3.checkbtn)
        self.tab3.resbox = QHBoxLayout()
        self.tab3.resbox.addWidget(self.tab3.resultlbl)

        self.tab3.box = QVBoxLayout()
        self.tab3.box.addLayout(self.tab3.grid1)
        self.tab3.box.addLayout(self.tab3.grid2)
        self.tab3.box.addLayout(self.tab3.btnbox)
        self.tab3.box.addWidget(self.tab3.resultlbltext)
        self.tab3.box.addLayout(self.tab3.resbox)

        self.tab3.setLayout(self.tab3.box)

    def correct(self):
        self.args.correct = True
        self.args.incorrect = False
        self.args.check = False
        self.args.idir = self.tab1.dirtext.text()
        self.args.clientfield = self.str_to_fields(self.tab1.clienttext.text())
        self.args.serverfield = self.str_to_fields(self.tab1.servertext.text())
        self.args.maxlen = self.tab1.maxlen.value()
        self.args.trid = self.tab1.traidentext.text()


        if self.args.idir == "":
            self.error_dialog.showMessage("No directory with training dataset")
            return
        if len(self.args.clientfield) == 0:
            self.error_dialog.showMessage("No client field")
            return
        if len(self.args.serverfield) == 0:
            self.error_dialog.showMessage("No server field")
            return
        if self.args.trid == "":
            self.error_dialog.showMessage("No trainig identficator")
            return

        if self.args.idir[-1] != "/":
            self.args.idir += "/"

        self.tab1.resultlbl.setText("Parsing pcaps...")
        self.app.processEvents()
        try:
            commands = pcapparser.get_commands(self.args)
        except RuntimeError as err:
            self.error_dialog.showMessage("ERROR: %s" % str(err))
            return 1

        if len(commands) == 0:
            self.tab1.resultlbl.setText("ERROR: No commands in pcaps found!")
            return
    
        self.tab1.resultlbl.setText("Training method 1...")
        self.app.processEvents()
        method1.train_correct(commands, self.args)
        self.tab1.resultlbl.setText("Training method 2...")
        self.app.processEvents()
        method2.train_correct(commands, self.args)
        self.tab1.resultlbl.setText("Training method 3...")
        self.app.processEvents()
        method3.train_correct(commands, self.args)

        self.tab1.resultlbl.setText("Correct communication trained!")

        with open(globs.fdir + "/output/gui_last_session.conf", "wb") as fp:
            pickle.dump(self.args, fp)

    def incorrect(self):
        self.args.correct = False
        self.args.incorrect = True
        self.args.check = False
        self.args.ifile = self.tab2.errfiletext.text()
        self.args.errfile = self.args.ifile
        self.args.clientfield = self.str_to_fields(self.tab2.clienttext.text())
        self.args.serverfield = self.str_to_fields(self.tab2.servertext.text())
        self.args.errmsg = self.tab2.errmsgtext.text()
        self.args.trid = self.tab2.traidentext.text()

        if self.args.ifile == "":
            self.error_dialog.showMessage("No file with error communication")
            return
        if len(self.args.clientfield) == 0:
            self.error_dialog.showMessage("No client field")
            return
        if len(self.args.serverfield) == 0:
            self.error_dialog.showMessage("No server field")
            return
        if self.args.errmsg == "":
            self.error_dialog.showMessage("No error description")
            return
        if self.args.trid == "":
            self.error_dialog.showMessage("No trainig identficator")
            return

        self.tab2.resultlbl.setText("Parsing pcap...")
        self.app.processEvents()
        try:
            commands = pcapparser.get_commands(self.args)
        except RuntimeError as err:
            self.error_dialog.showMessage("ERROR: %s" % str(err))
            return 1

        if len(commands) == 0:
            self.tab2.resultlbl.setText("ERROR: No commands in pcap found!")
            return

        self.tab2.resultlbl.setText("Training method 1...")
        self.app.processEvents()
        method1.train_incorrect(commands, self.args)
        self.tab2.resultlbl.setText("Training method 2...")
        self.app.processEvents()
        method2.train_incorrect(commands, self.args)
        self.tab2.resultlbl.setText("Training method 3...")
        self.app.processEvents()
        method3.train_incorrect(commands, self.args)

        self.tab2.resultlbl.setText("Incorrect communication trained!")

        with open(globs.fdir + "/output/gui_last_session.conf", "wb") as fp:
            pickle.dump(self.args, fp)

    def check(self):
        self.args.correct = False
        self.args.incorrect = False
        self.args.check = True
        self.args.ifile = self.tab3.errfiletext.text()
        self.args.checkfile = self.args.ifile
        self.args.clientfield = self.str_to_fields(self.tab3.clienttext.text())
        self.args.serverfield = self.str_to_fields(self.tab3.servertext.text())
        self.args.trid = self.tab3.traidentext.text()

        if self.args.ifile == "":
            self.error_dialog.showMessage("No file for error checking")
            return
        if len(self.args.clientfield) == 0:
            self.error_dialog.showMessage("No client field")
            return
        if len(self.args.serverfield) == 0:
            self.error_dialog.showMessage("No server field")
            return
        if self.args.trid == "":
            self.error_dialog.showMessage("No trainig identficator")
            return

        self.tab3.resultlbl.clear()
        self.tab3.resultlbl.insertPlainText("Parsing pcap...\n")
        self.app.processEvents()
        try:
            commands = pcapparser.get_commands(self.args)
        except RuntimeError as err:
            self.error_dialog.showMessage("ERROR: %s" % str(err))
            return 1

        self.tab3.resultlbl.clear()
        if len(commands) == 0:
            self.tab3.resultlbl.insertPlainText("ERROR: No commands in pcap found!\n")
            return

        resmsg = ""
        self.tab3.resultlbl.clear()
        self.tab3.resultlbl.insertPlainText("Checking with method 1...\n")
        self.app.processEvents()
        try:
            method1.check(commands, self.args)
        except RuntimeError as err:
            resmsg += str(err) + "\n"
        else:
            resmsg += "Method 1: Correct!\n"
        self.tab3.resultlbl.clear()
        self.tab3.resultlbl.insertPlainText("Checking with method 2...\n")
        self.app.processEvents()
        try:
            method2.check(commands, self.args)
        except RuntimeError as err:
            resmsg += str(err) + "\n"
        else:
            resmsg += "Method 2: Correct!\n"
        self.tab3.resultlbl.clear()
        self.tab3.resultlbl.insertPlainText("Checking with method 3...\n")
        self.app.processEvents()
        try:
            method3.check(commands, self.args)
        except RuntimeError as err:
            resmsg += str(err)
        else:
            resmsg += "Method 3: Correct!"

        self.tab3.resultlbl.clear()
        self.tab3.resultlbl.insertPlainText(resmsg)

        with open(globs.fdir + "/output/gui_last_session.conf", "wb") as fp:
            pickle.dump(self.args, fp)

    def tab_changed(self, i):
        if i == 0:
            self.activtab = 1
        if i == 1:
            self.activtab = 2
        if i == 2:
            self.activtab = 3

        if self.activtab == 1:
            if len(self.args.clientfield) != 0:
                f = self.fields_to_str(self.args.clientfield)
                self.tab1.clienttext.setText(f)
            if len(self.args.serverfield) != 0:
                f = self.fields_to_str(self.args.serverfield)
                self.tab1.servertext.setText(f)
            if self.args.trid != "":
                self.tab1.traidentext.setText(self.args.trid)
            self.tab1.resultlbl.setText("")
        if self.activtab == 2:
            if self.args.errfile != None:
                self.tab2.errfiletext.setText(self.args.errfile)
            if len(self.args.clientfield) != 0:
                f = self.fields_to_str(self.args.clientfield)
                self.tab2.clienttext.setText(f)
            if len(self.args.serverfield) != 0:
                f = self.fields_to_str(self.args.serverfield)
                self.tab2.servertext.setText(f)
            if self.args.trid != "":
                self.tab2.traidentext.setText(self.args.trid)
            self.tab2.resultlbl.setText("")
        if self.activtab == 3:
            if self.args.checkfile != None:
                self.tab3.errfiletext.setText(self.args.checkfile)
            if len(self.args.clientfield) != 0:
                f = self.fields_to_str(self.args.clientfield)
                self.tab3.clienttext.setText(f)
            if len(self.args.serverfield) != 0:
                f = self.fields_to_str(self.args.serverfield)
                self.tab3.servertext.setText(f)
            if self.args.trid != "":
                self.tab3.traidentext.setText(self.args.trid)
  
    def opendir(self):
        d = QFileDialog.getExistingDirectory(self, caption = "Select directory")
        self.tab1.dirtext.setText(d)

    def openfile(self):
        f = QFileDialog.getOpenFileName(self, caption = "Select file")
        if self.activtab == 2:
            self.tab2.errfiletext.setText(f[0])
        if self.activtab == 3:
            self.tab3.errfiletext.setText(f[0])

    def str_to_fields(self, s):
        return [x.strip() for x in s.split(",")]

    def fields_to_str(self, fields):
        s = ""
        for f in fields[:-1]:
            s += f + ", "
        s += fields[-1]
        return s

class DiagprotGUI(QMainWindow):
    
    def __init__(self, app):
        super().__init__()
        self.title = "Protocol diagnostics"
        self.width = 640
        self.height = 330
        globs.fdir = os.path.abspath(os.path.dirname(__file__))
        os.makedirs(os.path.dirname(globs.fdir + "/output/"), exist_ok = True)
        sys.path.append(globs.fdir + "/src/")
        try:
            with open(globs.fdir + "/output/gui_last_session.conf", "rb") as fp:
                self.args = pickle.load(fp)
        except:
            self.args = Arguments()
        self.app = app
        self.initGUI()

    def initGUI(self):
        self.setWindowTitle(self.title)
        self.setFixedSize(self.width, self.height)

        self.table_widget = Table_widget(self.args, self, self.app)
        self.setCentralWidget(self.table_widget)

        self.show()

    def closeEvent(self, event):
        with open(globs.fdir + "/output/gui_last_session.conf", "wb") as fp:
            pickle.dump(self.args, fp)
        event.accept()

def main():

    sys.setrecursionlimit(20000)

    app = QApplication(sys.argv)

    ex = DiagprotGUI(app)
    
    sys.exit(app.exec_())

if __name__ == "__main__":
    main()
