# Projekt:  Bakalářská práce
# Název:    Extrakce tunelovaných dat do samostatných toků
# Autor:    Roman Nahálka, xnahal01@stud.fit.vutbr.cz
# Datum:    02.04.2018
# Soubor:   fileCreator.py
# Popis:    Modul pro vytváření nových souboru. Konkrétně o souboru typu PDML a PCAP.

import xml.etree.ElementTree as ET
import struct
import socket
import binascii
import error
import os
from enum import Enum


class Protocols(Enum):
    NOT = 0
    IPV6 = 1
    IPV4 = 2
    TCP = 3
    UDP = 4
    ARP = 5


class PcapCreator:
    def newPcap(self, name, packets, globalHeader, removed, out):
        name = out + name + '.pcap'
        file = open(name, 'wb')
        self.__globalHeader(file, globalHeader)

        for packet in packets:
            headers = 0
            self.__packetHeader(file, packet[0], removed, packet)

            for proto in packet:
                if headers in removed:
                    headers += 1
                    continue

                protoName = proto.get('name')
                new = Protocols.NOT

                if headers + 1 in removed:
                    change = self.__change(removed, headers + 1)
                    new = self.__newHeader(change, packet[change])

                if protoName == 'eth':
                    self.__etherHeader(file, proto, new)

                elif protoName == 'ip':
                    self.__ip4Header(file, proto)

                elif protoName == 'ipv6':
                    self.__ip6Header(file, proto)

                elif protoName == 'tcp':
                    self.__tcpHeader(file, proto)

                elif protoName == 'udp':
                    self.__udpHeader(file, proto)

                elif protoName == 'icmp':
                    self.__icmpHeader(file, proto)

                elif protoName == 'icmpv6':
                    self.__icmpv6Header(file, proto)

                elif protoName == 'gre':
                    self.__greHeader(file, proto)

                elif protoName == 'arp':
                    self.__arpHeader(file, proto)

                elif protoName == 'fake-field-wrapper':
                    self.__data(file, proto)

                elif protoName == 'frame':
                    pass

                elif protoName == 'geninfo':
                    pass

                elif protoName == 'l2tp':
                    self.__l2tpHeader(file, proto)

                elif protoName == 'chdlc':
                    self.__ciscoHeader(file, proto)

                else:
                    print(protoName)
                    error.printErr(error.Errors.UNKNOWN_PROTOCOL, 0)
                    self.__stopCreate(file, name)
                    return

                headers += 1

        file.close()

    def __globalHeader(self, file, globalHeader):
        # Zapsani globalni hlaviky na zacatek souboru
        fmt = 'IHHiIII'
        file.write(struct.pack(fmt, *globalHeader))

    def __packetHeader(self, file, proto, removed, packet):
        # Zapsani hlavikcy paketu, dle fromatu PCAP
        fmt = 'IIII'
        header = []
        remNum = 0

        # Z pdml vytahneme cas
        time = proto.find('.//field[@name="timestamp"]').get('value').split('.')
        header.append(int(time[0]))
        time[1] = int(time[1])

        # Mikrosekundy jsou jako desetinne cislo
        while time[1] % 10 == 0:
            time[1] = int(time[1]/10)

        if removed:
            remNum = self.__removedBytes(removed, packet)

        # Vytahneme dalsi potrebne inforomace
        header.append(time[1])
        header.append(int(proto.find('.//field[@name="len"]').get('show')) - remNum)
        header.append(int(proto.find('.//field[@name="caplen"]').get('show')) - remNum)

        file.write(struct.pack(fmt, *header))

    def __etherHeader(self, file, proto, new):
        # Zapsani Ethernotove hlaviky do PCAPu
        fmt = '!6B6BH'
        ethHeader = []

        # Z PDMLka vythaneme vsechny potrebne informace
        dst = proto.find('.//field[@name="eth.dst"]').get('show').split(':')
        dst = [int(i, 16) for i in dst]
        ethHeader.extend(dst)
        src = proto.find('.//field[@name="eth.src"]').get('show').split(':')
        src = [int(i, 16) for i in src]
        ethHeader.extend(src)

        if new == Protocols.NOT:
            ethHeader.append(int(proto.find('.//field[@name="eth.type"]').get('show'), 16))

        else:
            ethHeader.append(self.__newEthType(new))

        file.write(struct.pack(fmt, *ethHeader))

    def __newEthType(self, new):
        ethType = 0

        if new == Protocols.IPV4:
            ethType = int('0x0800', 16)

        elif new == Protocols.IPV6:
            ethType = int('0x86dd', 16)

        elif new == Protocols.ARP:
            ethType = int('0x0806', 16)

        return ethType

    def __ip4Header(self, file, proto):
        # Zapsani IPv4 hlavicky do PCAPu
        fmt = '!BBHHHBBH4s4s'
        ipHeader = []

        # Z PDMLka vytahneme vsechny potrebne informace
        version = int(proto.find('.//field[@name="ip.version"]').get('show'))
        hl = int(proto.find('.//field[@name="ip.hdr_len"]').get('show'))/4
        ipHeader.append(int((version << 4) + hl))
        ipHeader.append(int(proto.find('.//field[@name="ip.dsfield"]').get('value'), 16))
        ipHeader.append(int(proto.find('.//field[@name="ip.len"]').get('show')))
        ipHeader.append(int(proto.find('.//field[@name="ip.id"]').get('show'), 16))
        ipHeader.append(int(proto.find('.//field[@name="ip.frag_offset"]').get('value'), 16))
        ipHeader.append(int(proto.find('.//field[@name="ip.ttl"]').get('show')))
        ipHeader.append(int(proto.find('.//field[@name="ip.proto"]').get('show')))
        ipHeader.append(int(proto.find('.//field[@name="ip.checksum"]').get('show'), 16))
        ipHeader.append(socket.inet_aton(proto.find('.//field[@name="ip.src"]').get('show')))
        ipHeader.append(socket.inet_aton(proto.find('.//field[@name="ip.dst"]').get('show')))

        file.write(struct.pack(fmt, *ipHeader))

    def __ip6Header(self, file, proto):
        fmtMain = '!HBB16s16s'
        ipHeader = []

        # Z PDMLka vytahneme vsechny potrebne informace
        data = proto.find('.//field[@name="ipv6.flow"]').get('unmaskedvalue')
        data = binascii.a2b_hex(data)
        ipHeader.append(int(proto.find('.//field[@name="ipv6.plen"]').get('show')))
        ipHeader.append(int(proto.find('.//field[@name="ipv6.nxt"]').get('show')))
        ipHeader.append(int(proto.find('.//field[@name="ipv6.hlim"]').get('show')))
        ipHeader.append(socket.inet_pton(socket.AF_INET6, proto.find('.//field[@name="ipv6.src"]').get('show')))
        ipHeader.append(socket.inet_pton(socket.AF_INET6, proto.find('.//field[@name="ipv6.dst"]').get('show')))

        file.write(data)
        file.write(struct.pack(fmtMain, *ipHeader))

    def __tcpHeader(self, file, proto):
        # Zapsani TCP hlavicky do souboru
        fmt = '!HHLL'
        fmt2 = '!HHH'
        tcpHeader = []


        # Z PDMLka vytahneme vsechny potrebne informace
        tcpHeader.append(int(proto.find('.//field[@name="tcp.srcport"]').get('show')))
        tcpHeader.append(int(proto.find('.//field[@name="tcp.dstport"]').get('show')))
        tcpHeader.append(int(proto.find('.//field[@name="tcp.seq"]').get('value'), 16))
        tcpHeader.append(int(proto.find('.//field[@name="tcp.ack"]').get('value'), 16))
        file.write(struct.pack(fmt, *tcpHeader))
        tcpHeader = []


        data = proto.find('.//field[@name="tcp.flags"]').get('unmaskedvalue')
        data = binascii.a2b_hex(data)
        file.write(data)

        tcpHeader.append(int(proto.find('.//field[@name="tcp.window_size_value"]').get('show')))
        tcpHeader.append(int(proto.find('.//field[@name="tcp.checksum"]').get('value'), 16))
        tcpHeader.append(int(proto.find('.//field[@name="tcp.urgent_pointer"]').get('show')))

        file.write(struct.pack(fmt2, *tcpHeader))

        options = proto.find('.//field[@name="tcp.options"]')
        if options is not None:
            options = options.get('value')
            options = binascii.a2b_hex(options)
            file.write(options)

    def __udpHeader(self, file, proto):
        # Zapsani UDP hlavicky do souboru
        fmt = '!HHHH'
        udpHeader = []

        # Z PDMLka vythaneme vsechny potrebne informace
        udpHeader.append(int(proto.find('.//field[@name="udp.srcport"]').get('show')))
        udpHeader.append(int(proto.find('.//field[@name="udp.dstport"]').get('show')))
        udpHeader.append(int(proto.find('.//field[@name="udp.length"]').get('show')))
        udpHeader.append(int(proto.find('.//field[@name="udp.checksum"]').get('show'), 16))

        file.write(struct.pack(fmt, *udpHeader))

    def __icmpHeader(self, file, proto):
        # Zapsani ICMP hlavicky do PCAPu
        fmt = '!BBHHH'
        icmpHeader = []

        # Vytahnuti potrebnych informaci z IMCP zpravy
        icmpHeader.append(int(proto.find('.//field[@name="icmp.type"]').get('show')))
        icmpHeader.append(int(proto.find('.//field[@name="icmp.code"]').get('show')))
        icmpHeader.append(int(proto.find('.//field[@name="icmp.checksum"]').get('show'), 16))
        icmpHeader.append(int(proto.find('.//field[@name="icmp.ident"]').get('show')))
        icmpHeader.append(int(proto.find('.//field[@name="icmp.seq"]').get('show')))

        file.write(struct.pack(fmt, *icmpHeader))
        data = proto.find('.//field[@name="icmp.data_time"]').get('value')
        data = binascii.a2b_hex(data)
        file.write(data)
        data = proto.find('.//field[@name="data"]').get('value')
        data = binascii.a2b_hex(data)
        file.write(data)

    def __icmpv6Header(self, file, proto):
        # Zapsani ICMPv6 hlavicky do PCAPu
        fmt = '!BBHHH'
        icmpHeader = []

        # Vytahnuti potrebnych informaci z IMCP zpravy
        icmpHeader.append(int(proto.find('.//field[@name="icmpv6.type"]').get('show')))
        icmpHeader.append(int(proto.find('.//field[@name="icmpv6.code"]').get('show')))
        icmpHeader.append(int(proto.find('.//field[@name="icmpv6.checksum"]').get('show'), 16))
        icmpHeader.append(int(proto.find('.//field[@name="icmpv6.echo.identifier"]').get('show'), 16))
        icmpHeader.append(int(proto.find('.//field[@name="icmpv6.echo.sequence_number"]').get('show')))

        file.write(struct.pack(fmt, *icmpHeader))
        data = proto.find('.//field[@name="data"]').get('value')
        data = binascii.a2b_hex(data)
        file.write(data)

    def __greHeader(self, file, proto):
        # Zapsani GRE hlavicky do PCAPu
        fmt = "!HH"
        greHeader = []

        # Vytahnuti potrebnych informaci z GRE hlavicky
        greHeader.append(int(proto.find('.//field[@name="gre.flags_and_version"]').get('show'), 16))
        greHeader.append(int(proto.find('.//field[@name="gre.proto"]').get('show'), 16))

        file.write(struct.pack(fmt, *greHeader))

    def __l2tpHeader(self, file, proto):
        # Zapsani L2TPv3 hlavicky do PCAPu
        data = proto.find('.//field[@name="l2tp.session_id"]').get('value')
        data = binascii.a2b_hex(data)
        file.write(data)

    def __ciscoHeader(self, file, proto):
        data = proto.find('.//field[@name="chdlc.address"]').get('value')
        data += proto.find('.//field[@name="chdlc.control"]').get('value')
        data += proto.find('.//field[@name="chdlc.protocol"]').get('value')

        data = binascii.a2b_hex(data)
        file.write(data)

    def __arpHeader(self, file, proto):
        # Zapsani ARP hlavicky do PCAPu
        fmt = "!HHBBH6B4s6B4s"
        arpHeader = []

        # Vytahnuti potrebnych informaci z ARP hlavicky
        arpHeader.append(int(proto.find('.//field[@name="arp.hw.type"]').get('show')))
        arpHeader.append(int(proto.find('.//field[@name="arp.proto.type"]').get('show'), 16))
        arpHeader.append(int(proto.find('.//field[@name="arp.hw.size"]').get('show')))
        arpHeader.append(int(proto.find('.//field[@name="arp.proto.size"]').get('show')))
        arpHeader.append(int(proto.find('.//field[@name="arp.opcode"]').get('show')))

        src = proto.find('.//field[@name="arp.src.hw_mac"]').get('show').split(':')
        src = [int(i, 16) for i in src]
        arpHeader.extend(src)
        arpHeader.append(socket.inet_aton(proto.find('.//field[@name="arp.src.proto_ipv4"]').get('show')))

        dst = proto.find('.//field[@name="arp.dst.hw_mac"]').get('show').split(':')
        dst = [int(i, 16) for i in dst]
        arpHeader.extend(dst)
        arpHeader.append(socket.inet_aton(proto.find('.//field[@name="arp.dst.proto_ipv4"]').get('show')))

        file.write(struct.pack(fmt, *arpHeader))

    def __data(self, file, proto):
        # Payload
        data = proto.find('.//field[@name="data"]').get('value')
        data = binascii.a2b_hex(data)
        file.write(data)

    def __removedBytes(self, removed, packet):
        header = 0
        bytes = 0

        for proto in packet:
            if header in removed:
                bytes += int(proto.get('size'))
            header += 1

        return bytes

    def __change(self, removed, header):
        # Naleznuti nasledujici neodstranene hlavicky
        while header in removed:
            header += 1

        return header

    def __newHeader(self, change, proto):
        # Urceni typu nasledujici hlavicky
        name = proto.get('name')
        new = ''

        if name == 'ip':
            new = Protocols.IPV4

        elif name == 'ipv6':
            new = Protocols.IPV6

        return new

    def __stopCreate(self, file, name):
        # Zastaveni vytvareni PCAP souboru
        file.close()
        os.remove(name)


def newPdml(name, packets):
    with open(name, 'wb') as file:
        file.write(b"<?xml version=\"1.0\"?>\n")
        file.write(b"<?xml-stylesheet type=\"text/xsl\" href=\"pdml2html.xsl\"?>\n")
        file.write(b"<pdml>\n")

        # Zapiseme vsechny paketu daneho toku do souboru
        for packet in packets:
            file.write(ET.tostring(packet, 'utf-8'))

        file.write(b"</pdml>")

    file.close()
