# Projekt:  Bakalářská práce
# Název:    Extrakce tunelovaných dat do samostatných toků
# Autor:    Roman Nahálka, xnahal01@stud.fit.vutbr.cz
# Datum:    27.03.2018
# Soubor:   extraktor.py
# Popis:    Hlavní třída programu.

import sys
import subprocess
import error
import os
import struct
from parser import Parser
from arguments import Arguments
from tunnels import Tunnels
from fileCreator import PcapCreator
import xml.etree.ElementTree as ET


# Začátek programu
def main():
    extr = Extraktor()
    extr.main()


# Třída reprezentující extraktor
class Extraktor:
    def __init__(self):
        self.__globalHeader = ''
        self.__outDir = ''
        self.__all = ''

    def main(self):
        # Zpracujeme argumenty a zjistime tak vstupni soubor aplikace
        args = Arguments()
        args.parseArguments()
        mainPDML = self.__pcapToPdml(args.getArgs.file)
        self.__globalHeader = self.__readGlobalHeader(args.getArgs.file)
        self.__outDir = args.getArgs.dir
        self.__all = args.getArgs.all

        if not self.__outDir.endswith('/'):
            self.__outDir += '/'

        self.__outDirectory()

        # Vstupni soubor rozdelime do samostatnych sitovych toku
        parser = Parser()
        parser.parseMain(mainPDML)

        files = parser.getFiles

        self.__extraction(files)

        # Tady budou odstraneny pomocne soubory a ukoncen program.
        self.__delete(mainPDML, files)
        sys.exit()

    @staticmethod
    def __pcapToPdml(file):
        # Převod PCAP souboru do formátu PDML
        # Take si ulozime nezev souboru, ktery budeme vytvaret
        new = os.path.basename(os.path.splitext(file)[0])
        new += '.pdml'

        out = open(new, 'w')

        # Zkusíme převést soubor do formátu PDML
        try:
            subprocess.run(['tshark', '-r', file, '-T', 'pdml'], check=True, stdout=out)

        except:
            error.printErr(error.Errors.CANT_CONVERT, -2)

        out.close()
        return new

    def __extraction(self, files):
        for file in files:
            tree = ET.parse(file)
            pdml = tree.getroot()
            self.__analyze(pdml, file)

    def __analyze(self, pdml, file):
        packet = pdml[0]
        header = 0
        encaps = 0
        removed = []
        creator = PcapCreator()

        # Vytvoreni PCAPu se vsemi zapouzdrenimi
        if not self.__all:
            name = os.path.basename(os.path.splitext(file)[0])
            creator.newPcap(name, pdml, self.__globalHeader, removed, self.__outDir)

        for proto in packet:
            protoName = proto.get('name')

            if protoName == 'ip' or protoName == 'ipv6':
                tunn = Tunnels()
                encap = tunn.isIpTunnell(proto)

                if encap:
                    # Je tam zapouzdreni, novy PCAP bez zapouzdreni
                    encaps += 1
                    removed.extend(self.__processing(packet, tunn, header, removed))
                    name = os.path.basename(os.path.splitext(file)[0]) + '_encap' + str(encaps)

                    if not self.__all:
                        creator.newPcap(name, pdml, self.__globalHeader, removed, self.__outDir)

            header += 1

        if self.__all:
            creator.newPcap(name, pdml, self.__globalHeader, removed, self.__outDir)

    def __processing(self, packet, tunn, header, removed):
        remove = []

        remove.extend(tunn.tunnelHeaders(header, packet, removed))

        return remove

    def __readGlobalHeader(self, file):
        # H - unsigned short, I - unsigned integer, h - short, i - integer
        structFmt = '=IHHiIII'
        structLen = struct.calcsize(structFmt)
        structUnpack = struct.Struct(structFmt).unpack_from

        pcap = open(file, 'rb')
        data = pcap.read(structLen)
        pcap.close()

        return structUnpack(data)

    def __delete(self, mainPDML, files):
        # Uklid vytvorenych PDML souboru
        os.remove(mainPDML)

        for file in files:
            os.remove(file)

    def __outDirectory(self):
        try:
            os.mkdir(self.__outDir)
        except Exception:
            pass


# Zavolání začátku programu
if __name__ == '__main__':
    main()
