"""
    Nazev souboru: filter.py
    Autor: Jindrich Dudek (xdudek04)
    Email: xdudek04@stud.fit.vutbr.cz
    Datum posledni modifikace: 05/18/2018
    Verze Pythonu: 2.7 (2.7.12)
"""

import error_checker as EC


class Filter:
    def __init__(self):
        self.__checker = EC.ErrorChecker()

    def process(self, packets, properties):
        """
        Main function for rule processing from section properties.
        :param packets: List of packets in input network communication.
        :param properties: List with properties.
        :return: List of filtered packets.
        """

        filtered_packets = packets

        # Iterate over filter rules and apply them
        for rule in properties:
            if len(rule) == 2:  # Simple rule with two attributes
                filtered_packets = self.__simple_rule(rule, filtered_packets)
            elif len(rule) == 3:  # Complex rule with four attributes
                filtered_packets = self.__complex_rule(rule, filtered_packets)
            else:
                self.__checker.report_error('Illegal filtering rule in in the '
                                            'description.')

        return filtered_packets

    def __simple_rule(self, rule, packets):
        """
        Method for application of simple rule consisting only from two
        attributes field-name and value.
        :param rule: Dictionary with parsed rule.
        :param packets: List of packets to be filtered.
        :return: List of filtered packets after applied rule.
        """

        filtered_packets = []
        # Xpath expression to find field with name specified by 'field-name'
        xpath_expr = './/*[@name="%s"]' % rule['field-name']

        for packet in packets:
            # If packet has no attribute with name specified by field-name:
            if packet.find(xpath_expr) is None:
                if not rule['valid']:
                    filtered_packets.append(packet)
            else:  # If field has specified attribute
                if rule['valid']:
                    filtered_packets.append(packet)

        return filtered_packets

    def __complex_rule(self, rule, packets):
        """
        Method for application of complex rule consisting from three attributes
        field-name, valid and value.
        :param rule: Dictionary with parsed rule.
        :param packets: List of packets to be filtered.
        :return: List of filtered packets after applied rule.
        """

        filtered_packets = []
        attribute = 'value'
        # Xpath expressions for packet filtering
        xpath_name = './/*[@name="%s"]' % rule['field-name']
        for packet in packets:
            field = packet.find(xpath_name)
            if field is None:  # If packet has no field with specified name
                continue

            attr_value = field.get(attribute)  # Get attribute value
            # If such attribute exists and it has desired value
            if attr_value == rule['value']:
                if rule['valid']:
                    filtered_packets.append(packet)
            else:
                if not rule['valid']:
                    filtered_packets.append(packet)

        return filtered_packets

    def is_packet_required(self, packet, properties):
        """
        Checks if all filtering rules from section properties are satisfied
        for given packet.
        :param packet: Packet to be checked.
        :param properties: List of filtering rules.
        :return: True if all provided rules are satisfied for given packet,
                 False otherwise.
        """
        for rule in properties:
            if len(rule) == 2:  # Simple rule with two attributes
                if not self.__is_simple_rule_satisfied(packet, rule):
                    return False
            elif len(rule) == 3:  # Complex rule with four attributes
                if not self.__is_complex_rule_satisfied(packet, rule):
                    return False
            else:
                self.__checker.report_error('Illegal filtering rule in in the '
                                            'description.')
                return False

        return True

    def __is_simple_rule_satisfied(self, packet, rule):
        """
        Check if filtering rule consisting of two attributes field-name and
        valid is satisfied for given packet.
        :param packet: Packet to be checked.
        :param rule: Parsed YAML filtering rule.
        :return: True if rule is satisfied for given packet, False otherwise.
        """
        # Xpath expression to find field with name specified by 'field-name'
        xpath_expr = './/*[@name="%s"]' % rule['field-name']

        # If packet has no attribute with name specified by field-name:
        if packet.find(xpath_expr) is None:
            if not rule['valid']:
                return True
        else:  # If field has specified attribute
            if rule['valid']:
                return True

        return False  # Packet does not satisfy the rule

    def __is_complex_rule_satisfied(self, packet, rule):
        """
        Check if filtering rule consisting of three attributes field-name, valid
        and value is satisfied for given packet.
        :param packet: Packet to be checked.
        :param rule: Parsed YAML filtering rule.
        :return: True if rule is satisfied for given packet, False otherwise.
        """
        attribute = 'value'
        # Xpath expression for packet filtering
        xpath_name = './/*[@name="%s"]' % rule['field-name']

        field = packet.find(xpath_name)
        if field is None:  # If packet has no field with specified name
            return False

        attr_value = field.get(attribute)  # Get attribute value
        # If such attribute exists and it has desired value
        if attr_value == rule['value']:
            if rule['valid']:
                return True
        else:
            if not rule['valid']:
                return True

        return False  # Packet does not satisfy the rule
