"""
    Nazev souboru: error_checker.py
    Autor: Jindrich Dudek (xdudek04)
    Email: xdudek04@stud.fit.vutbr.cz
    Datum posledni modifikace: 05/18/2018
    Verze Pythonu: 2.7 (2.7.12)
"""

import os.path
import sys


class ErrorChecker:
    def __init__(self):
        self.error_codes = self.__enum(OPTION_ERR=1, CONVERSION_ERR=2)

    def check_arguments(self, directory, file):
        """
        Checks if directory provided by -d is really directory and input file
        provided by -f is really file.
        :param directory: String with path to directory.
        :param file: String with path to file.
        """
        is_directory = os.path.isdir(directory)
        is_file = os.path.isfile(file)

        if not is_directory:
            self.report_error('\"%s\" is not directory.' % directory)
            sys.exit(self.error_codes.OPTION_ERR)

        if not is_file:
            self.report_error('File \"%s\" does not exists.' % file)
            sys.exit(self.error_codes.OPTION_ERR)

    def check_atomic_syntax(self, desc):
        """
        Checks if attack description for atomic attack type contains all
        required attributes.
        :param desc: Dictionary with parsed YAML description.
        :return: True if description is OK or False if not.
        """

        # If there is attribute properties, check syntax of the filtering rules
        if 'properties' in desc:
            if not self.__check_filtering_rules(desc['properties'],
                                                'properties'):
                return False

        # If there is no attribute detection-conditions
        if not ("detection-conditions" in desc):
            self.report_error('Attribute "detection-conditions" in the '
                              'description was not found.')
            return False

        # Check syntax of section "detection-conditions"
        if not self.__check_detection_conditions(desc):
            return False

        # Check if thresholds are defined in description:
        if not self.__check_packet_thresholds(desc):
            return False

        return True

    def __check_filtering_rules(self, section, section_name):
        """
        Method for checking syntax of filtering rules.
        :param section: Parsed YAML description.
        :return: True if syntax is OK or False otherwise.
        """

        # Check if attribute properties is a list.
        if not isinstance(section, list):
            self.report_error('Attribute "%s" in the description is not a '
                              'list.' % section_name)
            return False

        for rule in section:
            # Check if rule is a structure (dictionary, after parsing)
            if not isinstance(rule, dict):
                self.report_error('Filtering rule in section "%s" should be '
                                  'structure (dictionary).' % section_name)
                return False

            if len(rule) == 2:  # Simple rule with two attributes
                if 'field-name' not in rule \
                        or not isinstance(rule['field-name'], str):
                    self.report_error('Attribute "field-name" was not found '
                                      'in filtering rule (section %s) or its '
                                      'value is not a string.' % section_name)
                    return False
                if 'valid' not in rule or not isinstance(rule['valid'], bool):
                    self.report_error('Attribute "valid" was not found in '
                                      'filtering rule (section %s) or its '
                                      'value is not boolean.' % section_name)
            elif len(rule) == 3:  # Complex rule with three attributes
                if 'field-name' not in rule or 'value' not in rule:
                    self.report_error('One of attributes "field-name" or '
                                      '"value" is not present in filtering '
                                      'rule (section %s).' % section_name)
                    return False
                # Check if values of attributes are strings.
                if not isinstance(rule['field-name'], str) \
                        or not isinstance(rule['value'], str):
                    self.report_error('One of attributes "field-name" or '
                                      '"value" in filtering rule (section %s) '
                                      'is not a string.' % section_name)
                    return False

                if 'valid' not in rule or not isinstance(rule['valid'], bool):
                    self.report_error('Attribute "valid" was not found in '
                                      'filtering rule (section %s) or its '
                                      'value is not boolean.' % section_name)

            else:  # If there are unexpected number of attributes in rule
                self.report_error('Unexpected number of attributes of filtering'
                                  ' rule in section "%s". Should be two or '
                                  'four.' % section_name)
                return False

        return True

    def __check_detection_conditions(self, desc):
        """
        Method for syntax checking of section "detection-conditions".
        :param desc: Parsed description of the attack in YAML.
        :return: True if syntax is OK or False otherwise.
        """
        types = ['and', 'or']  # Possible values of attribute "type"
        det_conditions = desc['detection-conditions']

        # If attribute "detection-conditions" is not a structure
        if not isinstance(det_conditions, dict):
            self.report_error('Section "detection-conditions" should be '
                              'structure (dictionary).')
            return False

        # Check if attribute "type" is present in section "detection-conditions"
        if 'type' not in det_conditions:
            self.report_error('Required attribute "type" is not present in the '
                              'description in section "detection-conditions".')
            return False

        # Check if attribute "type" has supported value
        if det_conditions['type'] not in types:
            self.report_error('Unexpected value of attribute "type" in section '
                              '"detection-conditions."')
            return False

        # Check if there is attribute "conditions" in the section
        if 'conditions' not in det_conditions:
            self.report_error('Required attribute "conditions" is not present '
                              'in the description in section '
                              '"detection-conditions".')
            return False

        # Check if attribute conditions is a list
        if not isinstance(det_conditions['conditions'], list):
            self.report_error('Attribute "conditions" is not a list.')
            return False

        # Check every condition in the conditions list:
        for cond in det_conditions['conditions']:
            # Check if condition is dictionary
            if not isinstance(cond, dict):
                self.report_error('Condition in section "detection-conditions" '
                                  'in list "conditions" should be structure '
                                  '(dictionary).')
                return False

            if 'condition-type' not in cond:
                self.report_error('Attribute "condition-type" is not present '
                                  'in the description of condition.')
                return False

            # Check syntax of all conditions:
            if cond['condition-type'] == 'expression':
                if not self.__check_expression_condition(cond):
                    return False
            elif cond['condition-type'] == 'field-value':
                if not self.__check_field_value_condition(cond):
                    return False
            elif cond['condition-type'] == 'field-count':
                if not self.__check_field_count_condition(cond):
                    return False
            elif cond['condition-type'] == 'packet-ratio':
                if not self.__check_packet_ratio_condition(cond):
                    return False
            else:
                self.report_error('Unsupported value of attribute '
                                  '"condition-type" in the description of '
                                  'condition in section "detection-condition".')
                return False

        return True

    def __check_expression_condition(self, cond):
        """
        Method for syntax checking of detection condition of type "expression".
        :param cond: Dictionary with parsed condition in YAML.
        :return: True if syntax is OK or False otherwise.
        """
        # Check if attribute expression is present and if its value is a string
        if 'expression' not in cond or not isinstance(cond['expression'], str):
            self.report_error('Attribute "expression" in condition '
                              'of type "expression" was not found or its value '
                              'is not a string.')
            return False
        # Check if attribute variables is present
        if 'variables' not in cond:
            self.report_error('Attribute "variables" in condition of '
                              'type "expression" was not found.')
            return False
        # Check if variables attribute is a list
        if not isinstance(cond['variables'], list):
            self.report_error('Attribute "variables" in condition of type '
                              '"expression" should be list.')
            return False

        # Check syntax of all variable descriptions:
        for var in cond['variables']:
            if not self.__check_variable_syntax(var):
                return False

        return True

    def __check_variable_syntax(self, var):
        """
        Method for syntax checking of variables defined in section "variables"
        in detection condition of type "expression".
        :param var: Dictionary with parsed YAML description of variable.
        :return: True if syntax is OK or False otherwise.
        """
        if not isinstance(var, dict):
            self.report_error('Description of variable in condition of type '
                              '"expression" should be structure (dictionary).')

            return False
        # Check if attribute name is present in variable description
        if 'name' not in var:
            self.report_error('Attribute "name" was not found in the '
                              'description of variable in condition '
                              'of type "expression".')
            return False

        # Check if name of variable is a string and starts with uppercase:
        if not isinstance(var['name'], str) or not var['name'][0].isupper():
            self.report_error('Name of variable in condition of type '
                              '"expression" should start with uppercase '
                              'letter.')
            return False
        # Check if attribute type is in the description
        if 'type' not in var:
            self.report_error('Attribute "type" in description of variable in '
                              'condition of type "expression" was not found.')
            return False

        if var['type'] == 'field-value':
            # Check all required attributes:
            if 'value-type' not in var or 'field-name' not in var:
                self.report_error('Cannot find one of attributes: "value-type" '
                                  'or "field-name" in the description of '
                                  'variable in condition of type "expression".')
                return False
            # Check if attributes field-name and attribute has string value.
            if not isinstance(var['field-name'], str):
                self.report_error('Attribute "field-name attribute" in '
                                  'variable description in the condition of'
                                  ' type "expression" is not a string.')
                return False

            # Check values of attribute "value-type":
            if var['value-type'] == 'specific':
                pass
            elif var['value-type'] == 'abstract':
                values = ['different', 'same']
                # In this case there is one extra attribute "value":
                if 'value' not in var \
                        or var['value'] not in values:
                    self.report_error('Attribute "value" in the description '
                                      'of variable is required in case '
                                      'attribute "value-type" has value '
                                      '"abstract". This attribute was not found'
                                      ' or has unsupported value.')
                    return False
            else:  # Unsupported value of attribute "value-type"
                self.report_error('Unsupported value of attribute "value-type" '
                                  'in the description of variable in condition '
                                  'of type "expression".')
                return False

        elif var['type'] == 'filtered-frame-count':
            pass
        else:  # Unsupported value of attribute "type":
            self.report_error('Unsupported value of attribute "type" in the '
                              'description of variable in condition of type '
                              '"expression".')
            return False

        return True

    def __check_packet_thresholds(self, desc):
        """
        Checks if there are thresholds in the description.
        :param desc: Parsed YAML with attack description
        :return: True if description is OK, False otherwise.
        """
        cond_type = desc['detection-conditions']['type']
        conditions = desc['detection-conditions']['conditions']

        if cond_type == 'and':
            is_ok = False

            thresh = desc.get('threshold-error')
            if thresh is not None and isinstance(thresh, int):
                is_ok = True

            thresh = desc.get('threshold-warning')
            if thresh is not None and isinstance(thresh, int):
                is_ok = True

            if not is_ok:  # Thresholds were not found
                # If all conditions have attribute condition-type: expression
                # and all variables in expressions are abstract there do not
                # have to be thresholds. Condition-type: field-value
                # with value-type: abstract does not need thresholds as well.
                for cond in conditions:
                    if cond['condition-type'] == 'expression' \
                            and self.__all_variables_abstract(cond['variables']):
                        continue
                    value_type = cond.get('value-type')
                    if cond['condition-type'] == 'field-value' \
                            and value_type == 'abstract':
                        continue
                    if cond['condition-type'] == 'packet-ratio':
                        continue
                    # If condition is not expression or
                    # all variables are not abstract
                    else:
                        self.report_error('Attribute "threshold-error" or '
                                          '"threshold-warning" was not found '
                                          'in description or their value is '
                                          ' not an int.')
                        return False
        else:  # Type 'or'
            conditions = desc['detection-conditions']['conditions']
            # Every condition must have at least one threshold:
            for cond in conditions:
                is_ok = False
                thresh = cond.get('threshold-error')
                if thresh is not None and isinstance(thresh, int):
                    is_ok = True

                thresh = cond.get('threshold-warning')
                if thresh is not None and isinstance(thresh, int):
                    is_ok = True

                if not is_ok:
                    # If condition is an expression with all variables abstract
                    # there is no need for thresholds. Condition field-value
                    # with abstract value-type does not need them as well.
                    if cond['condition-type'] == 'expression' \
                            and self.__all_variables_abstract(cond['variables']):
                        continue
                    value_type = cond.get('value-type')
                    if cond['condition-type'] == 'field-value' \
                            and value_type == 'abstract':
                        continue
                    if cond['condition-type'] == 'packet-ratio':
                        continue
                    else:
                        self.report_error('At least one condition has no '
                                          'specified attribute "threshold-'
                                          'error" or "threshold-warning" or '
                                          'their value is not an integer value.'
                                          ' Every condition must have at least '
                                          'one threshold in case '
                                          'he attribute "type" has value "or".')
                        return False
        return True

    def __all_variables_abstract(self, variables):
        """
        Method that indicates whether all variables in list are abstract.
        Values of abstract variables are gained from more than one packet.
        :param variables: List with variables description.
        :return: True if all variables in list are abstract, False otherwise.
        """
        for v in variables:
            if v['type'] == 'field-value' and v['value-type'] == 'specific':
                return False

        return True

    def __check_field_value_condition(self, cond):
        """
        Method that checks syntax of condition of type "field-value".
        :param cond: Parsed description of YAML.
        :return: True if syntax is OK, False otherwise.
        """

        attr_count = len(cond)  # Count of attributes
        # Attributes "condition-type","field-name" + optionally two thresholds
        if 2 <= attr_count <= 4:
            if not self.__check_simple_field_value_condition(cond):
                return False
        # Attributes "condition-type", "value-type", "field-name",
        # "attribute", "value" + optionally two thresholds
        elif 5 <= attr_count <= 7:
            if not self.__check_complex_field_value_condition(cond):
                return False
        else:
            self.report_error('Number of attributes in condition of type '
                              '"field-value" is not valid.')
            return False

        return True

    def __check_simple_field_value_condition(self, cond):
        """
        Method that checks syntax of simple attribute value condition only with
        attribute "field-name".
        :param cond: Parsed description of YAML condition.
        :return: True if description is OK, False otherwise.
        """
        if 'field-name' not in cond:
            self.report_error('Attribute "field-name" was not found in '
                              'description of condition of type '
                              '"field-value".')
            return False
        if not isinstance(cond['field-name'], str):
            self.report_error('Value of attribute "field-name" in '
                              'description of condition of type '
                              '"field-value" should be string')
            return False

        return True

    def __check_complex_field_value_condition(self, cond):
        """
        Method that checks syntax of complex attribute value condition with
        attributes "field-name", "value-type" and "value".
        :param cond: Parsed description of YAML condition.
        :return: True if description is OK, False otherwise.
        """
        # Possible values of attribute "value-type":
        value_types = ['abstract', 'specific']
        # Possible values of attribute "value":
        values = ['same', 'different']

        if 'value-type' not in cond:
            self.report_error('Attribute "value-type" was not found in '
                              'the description of condition of type '
                              '"field-value".')
            return False

        if cond['value-type'] not in value_types:
            self.report_error('Unsupported value of attribute "value-type" in '
                              'description of condition of type '
                              '"field-value". Should be "abstract" or '
                              '"specific".')
            return False

        if 'field-name' not in cond  \
                or not isinstance(cond['field-name'], str):
            self.report_error('Attribute "field-name" was not found in '
                              'the description of condition of type '
                              '"field-value" or its value is not a string.')
            return False

        if 'value' not in cond:
            self.report_error('Attribute "value" was not found in the '
                              'description of condition of type '
                              '"field-value".')
            return False

        if cond['value-type'] == 'abstract':
            if cond['value'] not in values:
                self.report_error('Unsupported value of attribute "value" in '
                                  'description of condition of type '
                                  '"field-value". Should be "same" or '
                                  '"different".')
                return False

            if 'count' not in cond or not isinstance(cond['count'], int):
                self.report_error('Attribute "count" was not found in the '
                                  'description of condition of type '
                                  '"field-value" or its value is not an '
                                  'integer.')
                return False
        else:  # cond['value-type'] == 'specific'
            if not isinstance(cond['value'], str):
                self.report_error('Value of attribute "value" in description '
                                  'of condition of type "field-value" '
                                  'should be a string.')
                return False

        return True

    def __check_field_count_condition(self, cond):
        """
        Method that checks syntax of condition of type "field-count".
        :param cond: Parsed YAML description of condition.
        :return: True if description is OK, False otherwise.
        """
        error_msg = 'Attribute "%s" was not found in the description of ' \
                    'condition "field-count" or its value is not a %s.'

        if 'field-name' not in cond \
                or not isinstance(cond['field-name'], str):
            self.report_error(error_msg % ('field-name', 'string'))
            return False

        if 'count' not in cond or not isinstance(cond['count'], int):
            self.report_error(error_msg % ('count', 'integer'))
            return False

        return True

    def __check_packet_ratio_condition(self, cond):
        """
        Method that checks syntax of condition of type "packet-ratio".
        :param cond: Parsed YAML description of condition.
        :return: True if description is OK, False otherwise.
        """
        error_msg = 'Attribute "%s" was not found in the description of ' \
                    'condition "packet-ratio" or its value is not a %s.'

        if 'ratio' not in cond or not isinstance(cond['ratio'], float):
            self.report_error(error_msg % ('ratio', 'float'))
            return False

        if 'packets' not in cond or not isinstance(cond['packets'], list):
            self.report_error(error_msg % ('packets', 'list'))
            return False

        if not self.__check_packet_ratio_packets_desc(cond['packets']):
            return False

        return True

    def __check_packet_ratio_packets_desc(self, packets):
        """
        Method that checks syntax of section "packets" in description of
        condition "packet-ratio".
        :param packets: Parsed YAML description of packets section.
        :return: True if description is OK, False otherwise.
        """
        if len(packets) != 2:
            self.report_error('Section "packets" in description of condition '
                              '"packet-ratio" expects description of exactly '
                              'two packets.')
            return False

        for packet in packets:
            if 'properties' not in packet:
                self.report_error('Attribute "properties" was not found in '
                                  'the description of at least one packet in '
                                  'description of condition "packet-ratio".')
                return False

            if not self.__check_filtering_rules(packet['properties'],
                                                'properties'):
                return False

        return True

    def check_stream_syntax(self, desc):
        """
        Checks syntax of stream attack description.
        :param desc: Dictionary with parsed YAML description.
        :return: True if description is OK, False otherwise.
        """
        # If section properties is available in description, check filt. rules:
        if 'properties' in desc:
            if not self.__check_filtering_rules(desc['properties'],
                                                'properties'):
                return False

        if 'packets-specification' not in desc:
            self.report_error('Attr "packets-specification" was not found.')
            return False

        # Check section packets-specification:
        if not self.__check_packets_specification(desc):
            return False

        # Check presence of thresholds:
        if 'threshold-warning' not in desc and 'threshold-error' not in desc:
            self.report_error('Attributes "threshold-warning" and '
                              '"threshold-error" were not found. At least one '
                              'of them is required.')
            return False

        # Check data types of thresholds
        thresh_warn = desc.get('threshold-warning')
        thresh_err = desc.get('threshold-error')
        if thresh_warn is not None and not isinstance(thresh_warn, int):
            self.report_error('Value of attribute "threshold-warning" '
                              'should be an integer.')
            return False

        if thresh_err is not None and not isinstance(thresh_err, int):
            self.report_error('Value of attribute "threshold-warning" '
                              'should be an integer.')
            return False

        return True

    def __check_packets_specification(self, desc):
        """
        Method that checks syntax of section "packets-specification" in
        description of attack type stream.
        :param desc: Parsed YAML description of the attack.
        :return: True if description is OK, False otherwise.
        """
        spec_section = desc['packets-specification']
        err_msg = 'Attribute "%s" in section "packets-specification" was not ' \
                  'found or its value is not a %s.'

        if not isinstance(spec_section, dict):
            self.report_error('Section "packets-specification" should be '
                              'structure (dictionary).')
            return False

        if 'follow' not in spec_section \
                or not isinstance(spec_section['follow'], bool):
            self.report_error(err_msg % ('follow', 'boolean'))
            return False

        if 'specified-only' not in spec_section \
                or not isinstance(spec_section['specified-only'], bool):
            self.report_error(err_msg % ('specified-only', 'boolean'))
            return False

        if 'specification' not in spec_section \
                or not isinstance(spec_section['specification'], list):
            self.report_error(err_msg % ('specification', 'list'))
            return False

        for spec in spec_section['specification']:
            if not self.__check_specification(spec):
                return False

        return True

    def __check_specification(self, spec):
        """
        Method that checks specification of single packet in the description of
        attack type "stream", section "packets-specification".
        :param spec: Parsed YAML specification of packet.
        :return: True if specification is OK, False otherwise.
        """
        # In description must be attribute count or pair min-count, max-count
        if 'count' not in spec:
            if 'min-count' not in spec or 'max-count' not in spec:
                self.report_error('Attribute "count" or pair of attributes '
                                  '"max-count" and "min-count" were not found. '
                                  'At least one of them is required.')
                return False
            # If attributes min-count a max-count are present, check data types:
            else:
                min_count = spec['min-count']
                max_count = spec['max-count']
                if not isinstance(min_count, int):
                    self.report_error('Value of attribute "min-count" in '
                                      'specification of packet has to be '
                                      'integer.')
                    return False

                if not isinstance(max_count, int) and max_count != '*':
                    self.report_error('Value of attribute "max-count" in '
                                      'specification of packet has to be '
                                      'integer or "*" as symbol of unlimited '
                                      'number of packet occurrences.')
                    return False
        # If attribute count is present in description, check data type:
        else:
            if not isinstance(spec['count'], int):
                self.report_error('Value of attribute "count" in specification '
                                  'of packet has to be integer.')
                return False
            if spec['count'] == 0:
                self.report_error('Value of attribute "count" in specification '
                                  'of packet must not be zero.')
        # Check presence of attribute 'packet-properties' and its data type:
        if 'packet-properties' not in spec:
            self.report_error('Attribute "packet-properties" in specification '
                              'of packet was not found.')
            return False

        if not self.__check_filtering_rules(spec['packet-properties'],
                                            'packet-properties'):
            return False

        return True

    def check_group_syntax(self, desc):
        """
        Checks syntax of group attack description.
        :param desc: Dictionary with parsed YAML description.
        :return: True if description is OK, False otherwise.
        """
        # If section properties is available in description, check filt. rules:
        if 'properties' in desc:
            if not self.__check_filtering_rules(desc['properties'],
                                                'properties'):
                return False

        if 'packets-specification' not in desc:
            self.report_error('Attribute "packets-specification" was not '
                              'found.')
            return False

        # Check section packets-specification:
        if not self.__check_packets_specification(desc):
            return False

        # Check presence of thresholds:
        if 'threshold-warning' not in desc and 'threshold-error' not in desc:
            self.report_error('Attributes "threshold-warning" and '
                              '"threshold-error" were not found. At least one '
                              'of them is required.')
            return False

        # Check data types of thresholds
        thresh_warn = desc.get('threshold-warning')
        thresh_err = desc.get('threshold-error')
        if thresh_warn is not None and not isinstance(thresh_warn, int):
            self.report_error('Value of attribute "threshold-warning" '
                              'should be an integer.')
            return False

        if thresh_err is not None and not isinstance(thresh_err, int):
            self.report_error('Value of attribute "threshold-warning" '
                              'should be an integer.')
            return False

        if 'group-by' not in desc or not isinstance(desc['group-by'], list):
            self.report_error('Required attribute "group-by" was not found or '
                              'its value is not a list.')
            return False

        if not self.__check_group_by_section(desc['group-by']):
            return False

        return True

    def __check_group_by_section(self, group_by):
        """
        Method that checks syntax of "group-by" section in attack description.
        :param group_by: Parsed YAML description of group-by section.
        :return: True if description is OK, False otherwise.
        """
        err_msg = 'Attribute "%s" was not found in section "group-by" or its ' \
                  'value is not a string.'

        for item in group_by:
            if 'field-name' not in item \
                    or not isinstance(item['field-name'], str):
                self.report_error(err_msg % 'field-name')
                return False

        return True

    def report_error(self, error_msg):
        """
        Method for error reporting.
        :param error_msg: Message to be printed.
        :return:
        """
        sys.stderr.write("ERROR: %s\n" % error_msg)

    def __enum(self, **enums):
        return type('Enum', (), enums)



