#!/usr/bin/env python3

"""This module represents the main module of the application that is solely used for individual cli commands
.. module:: xtractor-cli
    :platform: linux
    :synopsis: Represents main module of the application for cli commands
.. moduleauthor:: Martin Bazik
"""

# TODO: TEST ON WINDOWS

import os
import re
import sys
import getopt
from device import Device
from device import App
from adb import ADB
from cache import Cache
from pathlib import Path
import sqlite3


def usage():
    """How to use this script
    """
      
    print("Parameters:\n"
            "\t-o,--output [output_file]\n"
            "\t-a,--app [app] - specified application\n"
            "\t-d,--d [device] - specified device\n"
            "\t-c,--command [command] - command for extractor\n"
            "\t-r,--record [record_id] - used instead of -d option, reads data from cache\n"
            "\t-f,--filter [filter] - sets up filter for future filtering\n"
            "\t-h,--help - help")

def printHelp():
    """Prints out script commands
    """

    print("Commands:\n"
            "\tconnect [ip] - connects wireless device with ip address\n"
            "\tlist-devices - lists currently connected devices\n"
            "\tlist-device-records - lists devices stored in cache\n"
            "\tdevice-info - prints info about device\n"
            "\tfind-applications - looks for applications\n"
            "\tfind-files - looks for files connected to specific apps or app defined in parameter --app\n"
            "\tfind-logs - looks for logs connected to specific apps or app defined in parameter --app\n"
            "\tpull - pull files connected to specific apps or app defined in parameter --app")


if __name__ == "__main__":
    try:
        opts, args = getopt.getopt(sys.argv[1:], "hr:f:a:d:c:o:", ["record=","filter=","app=","device=","command=","help","output="])
    except getopt.GetoptError as err:
        # print help information and exit:
        print(err) # will print something like "option -a not recognized"
        usage()
        printHelp()
        sys.exit(2)

    # Default output directory
    outputFolder = "output"
    command = None
    deviceName = ""
    device = None
    app = None
    filtering = []
    record = None
    load = False

    # Parses arguments
    for o, a in opts:
        if o in ("-h", "--help"):
            usage()
            printHelp()
            sys.exit()
        elif o in ("-o", "--output"):
            outputFolder = a
        elif o in ("-c", "--command"):
            command = a.split()
        elif o in ("-d", "--device"):
            deviceName = a
        elif o in ("-a", "--app"):
            app = a 
        elif o in ("-f", "--filter"):
            filtering = a.split()
        elif o in ("-r","--record"): 
            try:
                record = int(a)
                load = True
            except ValueError as e:
                print(e)
                sys.exit(1)
        else:
            assert False, "unhandled option"
   
    # Only one of them can be used
    if(load and deviceName):
        print("Only one of the option -r/-d can be used")
        usage()
        printHelp()
        exit(2)

    # Preparing phase
    try:
        adb = ADB()
    except ValueError as err:
        print("Adb was not found!")
        sys.exit(1)

    # Commnads
    if(not command):
        print("No command was specified!")
        usage()
        printHelp()
        sys.exit(1)
    elif(command[0] == "connect"):
        # If ip address was defined, it tries to connect device
        # TODO: test
        try:
            adb.connect(command[1])
        except ValueError as err:
            print(err)
        except IndexError as ie:
            print("No ip specified!")
            exit()
    elif(command[0] == "list-devices"):
        listDevices = adb.listDevices()
        if(listDevices):
            for dev in listDevices:
                print(dev[0]+":"+dev[1])
        else:
            print("No device was found!")
    elif(command[0] == "list-device-records"):
        cachePath = Path(outputFolder)/"cache.db"
        try:
            cache = Cache(cachePath)
            if(filtering):
                cache.setFilter(filtering)
            cache.printDevices()
        except sqlite3.OperationalError as e:
            print(e)
            exit(1)
    else:
        if(deviceName):
            adb.setDevice(deviceName)
        device = Device(adb,outputFolder,load=load,deviceID=record,filtering=filtering)
        if(not record is None):
            if(command[0] == "device-info"):
                # Info avout device
                device.printInfo()
            elif(command[0] == "find-applications"):
                device.printApps()
            elif(command[0] == "find-files"):
                # Looks for files
                if(app):
                    device.chooseExactApp(app)
                    device.printFilesApp()
                else:
                    device.printFiles()
            elif(command[0] == "find-logs"):
                if(app):
                    device.chooseExactApp(app)
                    device.printLogsApp()
                else:
                    device.printLogs()
            elif(command[0] == "find-all"):
                if(app):
                    device.chooseExactApp(app)
                    print("Apps:")
                    print("----------")
                    print("----------")
                    device.printChosenApp()
                    print("Files:")
                    print("----------")
                    print("----------")
                    device.printFilesApp()
                    print("Logs:")
                    print("----------")
                    print("----------")
                    device.printLogsApp()
                else:
                    print("Apps:")
                    print("----------")
                    print("----------")
                    device.printApps()
                    print("Files:")
                    print("----------")
                    print("----------")
                    device.printFiles()
                    print("Logs:")
                    print("----------")
                    print("----------")
                    device.printLogs()
        elif(deviceName):
            if(command[0] == "device-info"):
                # Info avout device
                device.printInfo()
            elif(command[0] == "find-applications"):
                # Looks for applications
                device.findApps()
                device.printApps()
            elif(command[0] == "find-files"):
                # Looks for files
                device.findApps()
                if(app):
                    device.chooseExactApp(app)
                    device.findFilesChosenApp()
                    device.printFilesApp()
                else:
                    device.lstFiles()
                    device.printFiles()
            elif(command[0] == "find-logs"):
                # Looks for logs
                device.findApps()
                if(app):
                    device.chooseExactApp(app)
                    device.findLogs("app")
                    device.printLogs("app")
                else:
                    device.findLogs()
                    device.printLogs()
            elif(command[0] == "find-all"):
                device.findApps()
                if(app):
                    device.chooseExactApp(app)
                    print("Apps:")
                    print("----------")
                    print("----------")
                    device.printChosenApp()
                    device.findFilesChosenApp()
                    print("Files:")
                    print("----------")
                    print("----------")
                    device.printFilesApp()
                    device.findLogs("app")
                    print("Logs:")
                    print("----------")
                    print("----------")
                    device.printLogs("app")
                else:
                    print("Apps:")
                    print("----------")
                    print("----------")
                    device.printApps()
                    device.lstFiles()
                    print("Files:")
                    print("----------")
                    print("----------")
                    device.printFiles()
                    device.findLogs()
                    print("Logs:")
                    print("----------")
                    print("----------")
                    device.printLogs()
            elif(command[0] == "pull"):
                # Pulls files
                device.findApps()
                if(app):
                    device.chooseExactApp(app)
                    device.findFilesChosenApp()
                    device.pullFiles("all")
                else:
                    device.lstFiles()
                    device.pullFiles("full")
            else:
                printHelp()
                sys.exit(1)
        else:
            print("Unknown device!")

