#!/usr/bin/env python3

"""This module represents the device and apps within the devices
.. module:: device
    :platform: linux
    :synopsis: Represents main module of the application (testing)
.. moduleauthor:: Martin Bazik
"""

import re
from adb import ADB
from datetime import datetime
from datetime import MINYEAR
from datetime import timedelta
from pathlib import Path
from subprocess import PIPE
import subprocess
from operator import methodcaller
from cache import Cache
import time

class Device:
    """This class represents the device used for the communication.
    Attributes: adb  (ADB)   adb object for communication
                apps (list)  list of App objects that represent individual apps
                version (string)    Android version
                identifier  (string)    identifier of the device
                chosenApp   (app)   currently used app
                dirs    (list)  list of all directories that are to be connected to the app
                defaultDirs (list)  list of all directories to be searched for subdirectories
                sdDirs  (list)  list of all directories in "/sdcard/" directory
                root    (boolean)   True if root access is present else False
                dir (string)    path to the output directory
                EOL (string)    used end of line sequence
    """

    def __init__(self,adb,output="output",load=False,deviceID=None,filtering=None):
        """Constructor
        It finds Android version, device identifier and if the device is rooted
        All attributes are declared here
        """

        # adb object for adb queries
        self.adb = adb

        # Identifier of the device
        self.identifier = self.adb.getDevice()

        try:
            self.setOutputDir(output)
        except ValueError as e:
            try:
                self.setOutputDir("output")
            except ValueError as err:
                print(err)
                sys.exit(1)

        #print(self.dir)
        self.cache = Cache(self.dir/"cache.db")

        if(filtering):
            self.setFilter(filtering)

        # initializes the device
        if(load and not deviceID is None):
            self.load(deviceID)
        else:
            self.initilize()

    def initilize(self):
        """Initializer
        It initializes all the object attributes.
        """
 
        # Checks if the device is rooted
        self.root = self.adb.checkRoot()

        # Sets up architecture of the device CPU
        self.arch = self.adb.checkArch()

        self.aapt = ""

        # Check sequence for end of line
        self.EOL = self.adb.checkEOL()

        # App currently worked on
        self.chosenApp = None

        # List of apps on the device
        self.apps = []

        # List of directories on the device to be connected to the app
        self.dirs = []

        # List of directories on the sdcard partition of the device to be connected to the app
        self.sdDirs = []

        # Default directories. Directories where apps save their data
        # TBA more
        self.defaultDirs = ["/data/data/","/sdcard/Android/data/","/sdcard/Android/media/","/sdcard/Android/obb/","/sdcard/"]

        # Looks for dirs on the device and fills dirs and sdDirs attributes
        self.setAllDirs()
        
        # Version of android
        self.adb.query(["shell","getprop","ro.build.version.release"])
        self.version = self.adb.getOutput().strip()

        
        # Identifier of the device
        self.identifier = self.adb.getDevice()

        # Used for sorting porpose
        self.sortKey = None
        self.sortReverse = None
        
        self.acquired = datetime.now()

        self.cache.addDevice(self.identifier,self.version,self.arch,self.root,self.acquired.timestamp())
        self.cache.setDevice(device=self.identifier)


    def load(self,deviceID):
        """Loading entire info of the apps from cache.
        Returns:    True if apps where loaded, else returns False
        """

        self.cache.setDevice(deviceID=deviceID)
        deviceDictionary = self.cache.getDevice(deviceID)
        if(not deviceDictionary):
            print("No such device!")
            exit(1)

        # Checks if the device is rooted
        self.root = deviceDictionary["root"]

        # Sets up architecture of the device CPU
        self.arch = deviceDictionary["arch"]

        # App currently worked on
        self.chosenApp = None

        # List of apps on the device
        self.apps = []

        # Loads version
        self.version = deviceDictionary["version"]
        
        # Identifier of the device
        self.identifier = deviceDictionary["id"]

        self.acquired = deviceDictionary["time"]

        # Used for sorting porpose
        self.sortKey = None
        self.sortReverse = None

        packages = None
        """
        # Package names
        packages = self.cache.getPackages(deviceID)

        # Iterates packages
        for pack in packages:

            # All app info
            app = App(pack[0])
            app.setName(pack[1])
            app.setActivityName(pack[2])
            app.setTotalTime(pack[4])
            app.setLastUsed(pack[3])
            for d in pack[5]:
                app.setDirExistance(True,d)
            for f in self.cache.getFiles(pack[0],deviceID):
                app.setDataFiles(f)
            for name, data in self.cache.getLogs(pack[0],deviceID).items():
                app.setLog(name,data)
            self.apps.append(app)

        """
        if(packages):
            return True
        else:
            return False

    def setFilter(self,filtering):
        if(len(filtering) == 3 and filtering[0] == "time"):
            try:
                dates = []
                dates.append(datetime.strptime(filtering[1], "%d-%m-%Y-%H-%M"))
                dates.append(datetime.strptime(filtering[2], "%d-%m-%Y-%H-%M"))
            except ValueError as e:
                print(e)

        self.cache.setFilter(filtering)



    def setAllDirs(self):
        """Setter for all default directories
        It iterates all to find all the directories that are supposed to be present on the device
        """

        # Iterates all the default directories
        for d in self.defaultDirs:
            self.setDirs(d)

    def setDirs(self,d):
        """Setter for present directories
        It looks for all directories within given directory d
        Parameters: d   (string)    given directory
        """
        
        commands = [["shell","ls","-lA",d],["shell","ls","-la",d],["shell","ls","-l","-a",d]]
        dirs = []
        for c in commands:
            try:
                self.adb.query(c)
            except ValueError as err:
                raise ValueError(err)
            output = self.adb.getOutput()
            if("Unknown option" in output or "No such file or directory" in output):
                continue
            else:
                dirs = output.splitlines()

        # Looks for lines beginning with character "d" because those represent direcotries based 
        # on ls command.
        for di in dirs:
            if(di and di[0] == "d"):
                name = ""
                match = re.match("[drwx-]+\s+\S+\s+\S+\s+[\d-]+\s+[\d:]+\s+(.+)",di)
                if(match):
                    name = match.group(1)
                else:
                    match = re.match("[drwx-]+\s+\d+\s+\S+\s+\S+\s+\d+\s+[\d-]+\s+[\d:]+\s+(.+)",di)
                    if(match):
                        name = match.group(1)

                # Full path of directory is saved
                self.dirs.append(d+name)

                # Special list for directories in "/sdcard/" directory
                if(d == "/sdcard/"):
                    self.sdDirs.append(d+name.strip())


    def setOutputDir(self,d):
        """Setter for directory where all extracted files are saved
        Parameters: d   (string)    Path of output directory
        """

        # Uses module pathlib for correct work with paths
        self.dir = Path(d)
        
        # If the directory is invalid it raises exception
        if(not(self.dir.parent.is_dir())):
            raise ValueError("Invalid dir")
            return

        # It tries to directory if it does not exist
        # If it is not possible it raises exception
        try:
            self.dir.mkdir(exist_ok=True)
        except FileExistsError:
            raise ValueError("Invalid dir")
            return

        # Special directory for the device
        self.deviceDir = self.dir / self.identifier.replace(":","-")
        self.deviceDir.mkdir(parents=True,exist_ok=True)


    def printInfo(self):
        """Prints info about the device
        """

        print("Andoid version: " + self.version)
        print("Device identifier: " + self.identifier)
        print("Device architecture: " + self.arch)
        if(self.isRoot()):
            print("Device is rooted!")
        else:
            print("Device is not rooted!")
        print("Aquired: " + str(self.acquired))



    def findApps(self):
        """This method finds all the apps present in the device. List of these apps is 
        stored in apps attribute.
        """

        # ADB query that lists all the usage info about the package
        self.adb.query(["shell","dumpsys","usagestats"])
        timeStats = self.adb.getOutput()

        # ADB query that lists all the packages
        self.adb.query(["shell","pm","list","packages"])
        packages = re.findall('package:(?P<pack>.*?)[\r\n]', self.adb.getOutput())
        

        lastUsedList = []
        usageList = []
        
        # It wokrs differently for different versions of Android
        versionList = self.version.split(".")
        if(versionList[0] == "2" or versionList[0] == "4"):
            lastUsedList = timeStats.split("Date: ")
            lastUsedList = [x for x in lastUsedList if x]
            usageList = lastUsedList
            newLastUsed = []
            for i in lastUsedList:
                newLastUsed.append((re.search("\d+",i).group(0),i))
            lastUsedList = newLastUsed
        
        # Iterates the packages
        for pack in packages:

            # Creates app object
            app = App(pack)

            lastUsed = datetime(1970,1,1)
            totalTime = timedelta()
            
            # For versions 2 and 4  
            if(versionList[0] == "2" or versionList[0] == "4"):
                lst = [x[0] for x in lastUsedList if pack in x[1]]
                datetimeList = []
                for i in lst:
                    if(len(lst) > 0):
                        datetimeList.append(datetime(int(i[:4]),int(i[4:6]),int(i[6:])))
                if(datetimeList):
                    lastUsed = max(datetimeList)

                totalTimeValue = 0 
                for i in usageList:
                    match = re.search(pack+":.*?(\d+)\s+ms",i)
                    if(match):
                        totalTimeValue += int(match.group(1))
                totalTime = timedelta(milliseconds=totalTimeValue)
            else:       
                # Parses usage info about the package
                lastUsedList = re.findall('package=' + pack + ' .*lastTime="([^"]*)"', timeStats)
                totalTimeList = re.findall('package=' + pack + ' .*totalTime="([^"]*)"', timeStats)

                datetimeList = []
                for i in lastUsedList:

                    # Parsing of date of the last use
                    datetimeMatch = re.fullmatch("(\d+)/(\d+)/(\d+), (\d+):(\d+) (PM|AM)",i)
                    if(datetimeMatch):
                        month = int(datetimeMatch.group(1))
                        day = int(datetimeMatch.group(2))
                        year = int(datetimeMatch.group(3))
                        daytime = datetimeMatch.group(6)
                        hour = int(datetimeMatch.group(4))
                        if(hour == 12):
                            hour = 0
                        if(daytime == "PM"):
                            hour += 12
                        minute = int(datetimeMatch.group(5))
                    else:
                        datetimeMatch = re.fullmatch("(\d+).(\d+).(\d+) (\d+):(\d+)",i)
                        if(datetimeMatch):
                            month = int(datetimeMatch.group(2))
                            day = int(datetimeMatch.group(1))
                            year = int(datetimeMatch.group(3))
                            hour = int(datetimeMatch.group(4))
                            minute = int(datetimeMatch.group(5))
                    if(datetimeMatch):
                        datetimeList.append(datetime(year,month,day,hour,minute))
                
                if(datetimeList):
                    lastUsed = max(datetimeList)

                # Parsing of extracted total time
                timeList = []
                for i in totalTimeList:
                    timeMatch = re.fullmatch("(\d+):(\d+)",i)
                    if(timeMatch):
                        hour = 0
                        minute = int(timeMatch.group(1))
                        second = int(timeMatch.group(2))
                    else:
                        timeMatch = re.fullmatch("(\d+):(\d+):(\d+)",i)
                        if(timeMatch):
                            hour = int(timeMatch.group(1))
                            minute = int(timeMatch.group(2))
                            second = int(timeMatch.group(3))
                    if(timeMatch):
                        timeList.append(timedelta(hours=hour,minutes=minute,seconds=second))
                if(timeList):
                    totalTime = max(timeList)

            app.setTotalTime(totalTime)
            app.setLastUsed(lastUsed)

            self.apps.append(app)

        # Tries to find a name for the apps
        self.findAppNames()
        self.findDirs()
        for app in self.apps:
            self.cache.addPackage(app.getID(),app.getName(),app.getActivityName(),app.getLastUsed(),app.getTotalTime(), ";".join(app.getDataDirs()))
        self.cache.commit()

    def setKeySortApps(self):
        """Selects sorting filter depending on key
        Input is taken from STDIN
        Possibilities:  "ID", "name", "time"
        """

        c = input("1) ID\n2) name\n3) time\nChoose: ")
        if(c == "1"):
            self.sortKey = methodcaller("getID")
        elif(c == "2"):
            self.sortKey = methodcaller("getName")
        elif(c == "3"):
            self.sortKey = methodcaller("getLastUsed")
        else:
            print("Choose number between 1 and 3")
            self.setKeySortApps()
    
    def setGradientSortApps(self):
        """Selects sorting filter to be increasing or decreasing
        Input is taken from STDIN
        Possibilities:  "increasing", "decreasing"
        """
        
        c = input("1) incresing\n2) decresing\nChoose: ")
        if(c == "1"):
            self.sortReverse = False
        elif(c == "2"):
            self.sortReverse = True
        else:
            print("Choose number between 1 and 2")
            self.setGradientSortApps()


    def sortApps(self):
        """Sorts the files
        """
        self.apps = sorted(self.apps,key=self.sortKey,reverse=self.sortReverse)


    def findAppNames(self):
        """Searches for the name of the app
        Dependent on "aapt" shell command on the device
        """
       
        #print("Looking for application names!")

        exists = True
        try:
            self.setAapt()
        except ValueError as e:
            print(e)
            exists = self.adb.commandExists(["aapt"])
            if(exists):
                self.aapt = "aapt"

        
        # List of all directories that possess apk file for an app
        self.adb.query(["shell","ls","/data/app/"])
        dirs = self.adb.getOutput().splitlines()        

        # Iterates all the apps
        for app in self.apps:

            # Name of the app
            name = []

            # Name of the activity of the app
            activity = []

            # Path to the extracted apk files
            apkFile = None

            # Path to apk file on the device
            filePath = ""

            # Tries to find app name and activity name in cache
            appActivityName = self.cache.getAppActivityName(app.getID())
            if(appActivityName):
                app.setActivityName(appActivityName)

            appName = self.cache.getAppName(app.getID())
            if(appName):
                app.setName(appName)
                continue

            codePath = str()
            
            # Package
            self.adb.query(["shell","dumpsys","package",app.getID()])
            
            # Save package log
            #app.setLog("package",self.adb.getOutput())
            packageLog = self.adb.getOutput()

            # Finds path to the code for the package
            codePaths = re.findall("codePath=(.*)",packageLog)
            if(len(codePaths) > 0):
                codePath = str(codePaths[0]).strip()

                # If the path itself leads to the apk file, it is saved 
                # else it looks for apk file within the code path
                if(codePath[-4:] != ".apk"):
                    codePath += "/*.apk"
                    self.adb.query(["shell","ls",codePath])
                    if(len(self.adb.getOutput())>0):
                        codePath = self.adb.getOutput().strip()
            if(exists):
                # ADB query to find the names
                self.adb.query(["shell",self.aapt,"d","badging",codePath])

                # Parses the names of the app and the activity produced by the app
                name = re.findall("application-label:'(.*)'",self.adb.getOutput())
                activity = re.findall("launchable-activity: name='(.*)'",self.adb.getOutput())

            
            else:
                # Creates directory for the app
                apkDir = self.deviceDir / app.getID()
                if(not apkDir.exists()):
                    apkDir.mkdir()

                # Path to the apk file
                apkFile = apkDir / "base.apk"

                # If the file is not yet downloaded it is pulled from the device
                if(not apkFile.exists()):
                    self.adb.query(["pull",codePath,str(apkFile)])

                # Looks for aapt command
                aapt = self.findAapt()

                # aapt is used locally on the computer
                output = self.command([aapt,"d","badging",str(apkFile)])

                # name is parsed
                name = re.findall("application-label:'(.*)'",output)

                # activity name is parsed
                activity = re.findall("launchable-activity: name='(.*)'",output)

            # If found it is saved
            if(len(name) >= 1):
                app.setName(name[0])
                
            # If found it is saved
            if(len(activity) >= 1):
                app.setActivityName(activity[0])

    def setAapt(self):
        """ Pushes aapt binary to the device and makes executable if possible
        TODO: test it
        """

        # Devices newer than version 5 need the binary to be build with -fPIE flag to work
        pie = ""
        if(self.version[0] >= "5"):
            pie = "-pie"

        # Pushes and enables binary depending on the architecture
        if(self.arch): 
            path = Path("aapt/aapt-"+self.arch+pie)
            self.adb.query(["push",path,"/data/local/aapt"])
            self.aapt = "/data/local/aapt"
            self.adb.query(["shell","chmod","0755",self.aapt])
        else:
            self.aapt = ""
            raise ValueError("Unknown architecture")

    def findAapt(self):
        """Searches for aapt command on the computer
        Returns:    Command or executable path to aapt command
        """

        # Tries aapt as command if it is not found it searches for the binary
        try:
            p = subprocess.Popen(["aapt"], stdin=PIPE, stdout=PIPE, stderr=PIPE)
            out_r, err_r = p.communicate(b"stdin")
        except FileNotFoundError:
            aapt = self.command(["find","/","-executable","-name","aapt"]).splitlines()
            if(len(aapt) >= 1):
                return aapt[0]
            else:
                print("Command aapt not found!")
                exit()
        return "aapt"

    def command(self,query):
        """Shell command on the device
        Parameters: query   (list)  shell command to be executed by the computer
        Returns:    string  output of the command
        """

        p = subprocess.Popen(query, stdin=PIPE, stdout=PIPE, stderr=PIPE)
        out_r, err_r = p.communicate(b"stdin")
        return out_r.decode("utf-8")

    def findDirs(self):
        """This method searches for the directories associated with the individual applications.
        If the directory exists it is stored within App object
        """

        #print("Looking for directories!")
        # List of all candidates for directory name
        names = []

        # List of all 
        dirs = []

        # Iterates all the basic directories using full identifier
        for d in self.defaultDirs:
            for app in self.apps:
                app.setDirExistance(self.exist(d+app.getID()),d+app.getID())

        # Iterates all apps and searches for special directories that may be connected with the app
        # It uses name and individual parts of identidier to search for potential directory
        for app in self.apps:
            names = app.getAppKeywords()

            # Search for full names
            dirs = self.findSDCardDirs(names)

             # Save directories
            for d in dirs:
                app.setDirExistance(True,d)
 
        for app in self.apps:

            names = app.getAppKeywords()       
            
            # Search for part of the name
            dirs = self.findPartialSDCardDirs(names)

            # Save directories
            for d in dirs:
                app.setDirExistance(True,d)



    def findSDCardDirs(self,names):
        """Searches for the name in /sdcard/ directory
        Parameters: names   (list)    names of potential directory names
        """

        finalDirs = []
        for name in names:

            # Potential paths
            dirs = ["/sdcard/"+name,"/sdcard/."+name]
            for d in dirs:

                # If the directory exists it is saved
                if(self.exist(d)):
                    finalDirs.append(d)
        return finalDirs


    def findPartialSDCardDirs(self,names):
        """Searches if there is partialy identical directory name
        Parameters: names   (list)    names of potential directory names
        """

        finalDirs = []

        # Iterates potential names
        for name in names:

            # In case it is found it is saved
            for d in self.sdDirs:
                if(name in d):
                    if(d in self.dirs):
                        self.dirs.remove(d)
                        finalDirs.append(d)
                    self.sdDirs.remove(d)
        return finalDirs

    def findFilesChosenApp(self):
        """Finds apps for the chosen app
        """
        self.findFilesApp(self.chosenApp)

    def findFilesApp(self,app):
        """Finds all the files for the application defined in parameter app
        Paramters:  app (string)    app indentifier
        """
        # Iterates all directories within application
        for d in app.iterateDir():

            # ADB query that recursively lists all the files in the directory
            self.adb.query(["shell","ls","-R","-l","-a","\""+d+"\""])
            recursiveRecord = self.adb.getOutput()
            subdirectoryList = recursiveRecord.split(self.EOL+self.EOL)

            # Iterates individual subdirectories
            for item in subdirectoryList:

                # Splits individual file records
                fileRecordList = [i.strip() for i in item.split(self.EOL) if i.strip()]
                if(len(fileRecordList) >= 1):
                    fileDir = fileRecordList[0][:-1]
                    for i in fileRecordList[1:]:
                        
                        # There are as far two possible formats
                        match = re.match("-[drwx-]+ +\S+ +\S+ +(\d+) +([\d-]+)\ +([\d:]+) +(.+)",i)
                        if(not match):
                            match = re.match("-[drwx-]+ +\d+ +\S+ +\S+ +(\d+) +([\d-]+) +([\d:]+) +(.+)",i)
                            
                        if(match):
                            #print(fileDir + "/" + name)
                            size = match.group(1)
                            date = match.group(2)
                            time = match.group(3)
                            name = match.group(4)
                            date = date.split("-")
                            time = time.split(":")
                            size = int(size)
                            dt = datetime(int(date[0]),int(date[1]),int(date[2]),int(time[0]),int(time[1]))
                            app.setDataFiles({"fullPath":fileDir + "/" + name,"name":name,"size":size,"time":dt})
                            self.cache.addFile(name, size, dt, fileDir + "/" + name, app.getID())
        self.cache.commit()


    def lstFiles(self):
        """This method iterates the directions of each application and looks for the files recursively
        alongside with the stats of these files.
        """
        
        # Iterates all the apps
        for app in self.apps:
            self.findFilesApp(app)


    def findLogs(self,apps=None):
        """This method takes care of colecting logs for each app
        """

        if(not apps):
            apps = self.apps
        else:
            apps = [self.chosenApp]
        print("Looking for logs!")
        # Notifications
        self.adb.query(["shell","dumpsys","notification"])
        notif = self.adb.getOutput()
        notifications = re.findall("(NotificationRecord[\s\S]*?pkg=(\S+)[\s\S]*?mRankingTimeMs=\d*)",notif)
  
        # Appops
        self.adb.query(["shell","dumpsys","appops"])
        appops = self.adb.getOutput()

        # Logcat
        self.adb.query(["logcat","-d"])
        logcat = self.adb.getOutput().splitlines()

        # To find PID for apps
        self.adb.query(["shell","ps"])
        pidOut = self.adb.getOutput()
        pidList = [x.split() for x in pidOut.splitlines()[1:] if x]

        # Batterystats
        self.adb.query(["shell","dumpsys","batterystats"])
        batterystats = self.adb.getOutput()

        # Procstats
        self.adb.query(["shell","dumpsys","procstats","--full-details","--hours","24"])
        procstats = self.adb.getOutput()

        # For each app
        for app in apps:
            log = ""
            pid = []

            # parses PID for an app
            # TODO: Test whether len(x) works
            if(len(pidList) > 0):
                pid = [x[1] for x in pidList if len(x) > 8 and x[8]==app.getID()]

            # parses logcat for an app
            for p in pid:
                app.setPID(p)
                for cat in logcat:
                    if((p in cat) or (app.getID() in cat)):
                        log += cat + "\n"
            app.setLog("logcat",log)
            log = ""

            # Package
            self.adb.query(["shell","dumpsys","package",app.getID()])
            app.setLog("package",self.adb.getOutput())
            packageLog = self.adb.getOutput()

            # Sets up UID for an app. It is unique for an app
            UIDFind = re.findall("userId=(\d+)",packageLog)
            if(len(UIDFind) > 0):
                app.setUID(UIDFind[0])
           
            # parses batterystats
            output = re.findall("\n  " + app.getShortUID() + ":"+self.EOL+"([\S\s]*?)(?:"+self.EOL+"  \S|"+self.EOL+self.EOL+")",batterystats)
            if(len(output) > 0):
                log = output[0]
            app.setLog("batterystats",log)
            log = ""

            # Activity
            self.adb.query(["shell","dumpsys","activity","package",app.getID()])
            app.setLog("activity",self.adb.getOutput())

            # parses procstats
            output = re.findall("  \* "+app.getID()+" [\S\s]*?:"+self.EOL+"([\S\s]*?)(?:"+self.EOL+"  \*|"+self.EOL+self.EOL+")",procstats)
            if(len(output) > 0):
                log = output[0]
            app.setLog("procstats",log)
            log = ""

            # parses appops
            output = re.findall("Uid "+ app.getShortUID() +":"+self.EOL+"([\S\s]*?)(?:Uid|\Z)",appops)
            if(len(output) > 0):
                log = output[0]
            app.setLog("appops",log)
            log = ""

            
            for notification in notifications:
                if(notification[1] == app.getID()):
                    log = notification[0]
            app.setLog("notification",log)

            # Caching
            self.cache.addLog(app.getLog("package"),app.getLog("logcat"),app.getLog("batterystats"),app.getLog("activity"),app.getLog("procstats"),app.getLog("appops"),app.getLog("notification"),app.getID())
        self.cache.commit()

    def getResources(self):
        """Used to get all the resources needed to run the program
        """

        print("Extracting data...")
        self.findApps()
        self.findLogs()
        #self.findDirs()
        self.lstFiles()
        print("Done!")


    def exist(self,d):
        """This method looks for existance of the directory.
        It also remove the directory from the list.
        It checks path in the prepared list.
        Return: True if exists, else False
        """
        
        if(d in self.dirs):
            self.dirs.remove(d)
            return True
        else:
            return False
        
    def findApp(self):
        """Looks for the app defined on STDIN
        """

        # Input
        name = input("Name the app: ")
        arr = []

        # App counter
        count = 0

        # Iterates all the apps and looks for match
        for app in self.apps:
            if name in app.getID():
                arr.append(app)
                print(str(count) + ") " + app.getID())
                count += 1

        # Choose the app from the list if exists
        if(count == 0):
            return self.findApp()
        else:
            return self.chooseApp(count,arr)

    def chooseApp(self,count,arr):
        """Choose the app you want to work with
        """

        # Input
        c = input("Choose app: ")

        # Input must be an integer within given range
        try:
            c = int(c)
        except ValueError:
            print("Insert value between 0 and " + str(count-1) + ": ")
            return self.chooseApp(count,arr)
        if(c >= count or c < 0):
            print("Insert value between 0 and " + str(count-1) + ": ")
            return self.chooseApp(count,arr)
        self.chosenApp = arr[c]
        return arr[c]

    def chooseExactApp(self,appName):
        """Chooses app defines in appName
        Parameters: appName (string)    identifier of the app
        """

        self.chosenAppName = appName
        for app in self.apps:
            if(app.getID() == appName):
                self.chosenApp = app

    def printApps(self):
        """This method is used solely for testing
        """

        packages = self.cache.getPackages()

        # Iterates packages
        for pack in packages:

            # All app info
            app = App(pack[0])
            app.setName(pack[1])
            app.setActivityName(pack[2])
            app.setTotalTime(pack[4])
            app.setLastUsed(pack[3])
            for d in pack[5]:
                app.setDirExistance(True,d)
            app.printInfo()
            print()

    def printChosenApp(self):
        """This method is used solely for testing
        """

        self.chosenApp.printInfo()
        print()


    def pullFiles(self,scope):
        """Method for downloading files from the device
        Characters ":" are replaced by character "-" because windows does not support character ":" in the name
        Parameters: scope   (string)    scope of the download
                                        possibilities:
                                        "all"   -   all files of the chosen app
                                        "filter"    -   files corresponding with the given filter
                                        "full"  -   all files on the device
        """

        files = []
        if(scope == "all"):
            app = App(self.chosenAppName)
            for f in self.cache.getFiles(self.chosenAppName):
                app.setDataFiles(f)
                p = Path(str(self.deviceDir)+"/"+self.chosenApp.getID()+str(Path(f["fullPath"]).parent).replace(":","-"))
                p.mkdir(parents=True,exist_ok=True)
                self.adb.query(["pull",f["fullPath"],str(self.deviceDir)+"/"+self.chosenApp.getID()+f["fullPath"].replace(":","-")])
        elif(scope == "filter"):
            files = self.chosenApp.getFilterFiles()
            for f in files:
                p = Path(str(self.deviceDir)+"/"+self.chosenApp.getID()+str(Path(f["fullPath"]).parent).replace(":","-"))
                p.mkdir(parents=True,exist_ok=True)
                self.adb.query(["pull",f["fullPath"],str(self.deviceDir)+"/"+self.chosenApp.getID()+f["fullPath"].replace(":","-")])

        elif(scope == "full"):
            packages = self.cache.getPackages(doFilter=False)
        
            for pack in packages:

                # All app info
                app = App(pack[0])
                for f in self.cache.getFiles(pack[0]):
                    p = Path(str(self.deviceDir)+"/"+app.getID()+str(Path(f["fullPath"]).parent).replace(":","-"))
                    p.mkdir(parents=True,exist_ok=True)
                    self.adb.query(["pull",f["fullPath"],str(self.deviceDir)+"/"+app.getID()+f["fullPath"].replace(":","-")])

    def saveLogs(self):
        """Saves log of the chosen app into its directory
        """

        # Path to app directory
        p = Path(str(self.deviceDir)+"/"+self.chosenApp.getID())
        p.mkdir(parents=True,exist_ok=True)

        # Path to log file
        p = p / "log.txt"

        # Saves logs
        p.write_bytes(self.chosenApp.getLogs().encode('utf-8'))

    def getID(self):
        """Returns device ID
        """
        return self.identifier

    def getVersion(self):
        """Returns device version
        """
        return self.version

    def isRoot(self):
        """Returns presence of root access on the device
        """
        return self.root

    def getApps(self):
        """Returns list of applications on the device
        """
        return self.apps

    def printFilesApp(self):
        """Prints out info of the chosen app
        """

        app = App(self.chosenAppName)
        for f in self.cache.getFiles(self.chosenAppName):
            app.setDataFiles(f)
            print(f)
        print(app.getID()+":")
        print()
        app.printFiles()

    def printFiles(self):
        """Prints out info of all the apps
        """

        packages = self.cache.getPackages(doFilter=False)
        print(packages)
        for pack in packages:

            # All app info
            app = App(pack[0])
            for f in self.cache.getFiles(pack[0]):
                app.setDataFiles(f)
            print(app.getID()+":")
            print()
            app.printFiles()


    def printLogs(self):
        """Prints out logs.
        Parameters: apps    if  None it uses all the apps else it uses chosen app
        """
        # Package names
        packages = self.cache.getPackages()

        # Iterates packages
        for pack in packages:

            # All app info
            app = App(pack[0])
            for name, data in self.cache.getLogs(pack[0]).items():
                app.setLog(name,data)
            print(app.getID()+":")
            print()
            app.printLogs()

    def printLogsApp(self):
        """Prints out logs.
        Parameters: apps    if  None it uses all the apps else it uses chosen app
        """

        app = App(self.chosenAppName)
        for name, data in self.cache.getLogs(self.chosenAppName).items():
            app.setLog(name,data)
        print(app.getID()+":")
        print()
        app.printLogs()


class App:
    """This class represents the application itself.
    Attributes: ID  (string)    indentifier
                lastUsed    (string)    last use of the application
                name    (string)    name of the application
                dirArr  (list)  list of all directories connected to the application
                dataFiles   (list)  list of all the application files 
                                    structure - {"fullPath":path,"name":name,"size":size,"time":time}
                                    path and name are strings
                                    size is int
                                    time is dateTime object
                filterFiles (list)  list of files corresponding wth the filter
                                    structure same as dataFiles
                sortKey (string)    sorting is based on this attribute
                sortReverse (boolean)   if False it sorts from the smallest to the biggest
                activityName    (string)    name of an executable activity
                log (dictionary)    dictionary of logs, with dumped service as a key
                UID (string)    UID of an app
                PID (list)  list of strings representing PIDs
                keywords    (list)  list of keywords to be ignored 
    """

    def __init__(self,ID):
        """Constructor
        It defines all object attributes
        """

        self.ID = ID
        self.lastUsed = ""
        self.totalTime = ""
        self.name = ""
        self.dirExistance = []
        self.dirArr = []
        self.dataFiles = []
        self.filterFiles = []
        self.sortKey = "name"
        self.sortReverse = False
        self.activityName = ""
        self.log = {}
        self.UID = ""
        self.PID = []
        
        # List keywords are not candidates for the directory name
        self.keywords = ["android","google","providers","com","org","ui","MainActivity"]


    def resetFilterFiles(self):
        """Resets filtered files
        """
        self.filterFiles = self.dataFiles


    def findFiles(self):
        """Looks for the file from STDIN
        It uses regular expressions to filter file names
        """

        reg = input("Name the file: ")

        # Uses * in meaning of .*
        regex = reg.replace("*",".*")
        arr = []

        # Looks for the files
        for f in self.getFilterFiles():
            if(re.fullmatch(regex,f["name"])):
                arr.append(f)
        self.filterFiles = arr


    def printFiles(self,t="default"):
        """Prints out the info about the files
        Parameters: t   (string)    files are separated into filtered "filter" and all "default"
        """

        dic = {}
        dic["filter"] = self.getFilterFiles()
        dic["default"] = self.dataFiles
        for f in dic[t]:
            print("Name: " + f["name"])
            print("Path: " + f["fullPath"])
            print("Size: " + str(f["size"]))
            print("Date: " + f["time"].ctime())
            print("-----------")

    def getLogs(self):
        """Getter for logs
        """
        out = ""
        for key, l in self.log.items():
            out += key + ":\n" + l + "\n-----------\n"
        return out


    def printLogs(self):
        """Prints logs
        """
        print(self.getLogs())


    def setKeySortFiles(self):
        """Selects sorting filter depending on key
        Input is taken from STDIN
        Possibilities:  "name", "size", "time"
        """

        c = input("1) name\n2) size\n3) time\nChoose: ")
        if(c == "1"):
            self.sortKey = "name"
        elif(c == "2"):
            self.sortKey = "size"
        elif(c == "3"):
            self.sortKey = "time"
        else:
            print("Choose number between 1 and 3")
            self.setKeySortFiles()
    
    def setGradientSortFiles(self):
        """Selects sorting filter to be increasing or decreasing
        Input is taken from STDIN
        Possibilities:  "increasing", "decreasing"
        """
        
        c = input("1) incresing\n2) decresing\nChoose: ")
        if(c == "1"):
            self.sortReverse = False
        elif(c == "2"):
            self.sortReverse = True
        else:
            print("Choose number between 1 and 2")
            self.setGradientSortFiles()


    def sortFiles(self):
        """Sorts the files
        """
        self.filterFiles = sorted(self.getFilterFiles(),key=self.getKey,reverse=self.sortReverse)

    def getFilterFiles(self):
        """Returns filtered files
        """

        if(len(self.filterFiles) > 0):
            return self.filterFiles
        else:
            return self.dataFiles

    def getKey(self,item):
        """Returns key for sorted()
        Parameters: item    (list)  item to be sorted
        """
        return item[self.sortKey]
          
    def findTimeInterval(self):
        """Looks for the files that were modified withon given time interval
        """

        # Beginning of the given interval
        filterFrom = self.inputDate("From")

        # End of the given interval
        filterTo = self.inputDate("To")

        # Temporary list for filtered files
        arr = []

        # Looks for the files
        for f in self.getFilterFiles():

            # Filters the files
            if(f["time"] >= filterFrom and f["time"] <= filterTo):
                arr.append(f)

        # Output
        self.filterFiles = arr


    def inputDate(self,label):
        """Takes input from STDIN and parses it into list of individual items
        Parameters: label   (string)    text displayed on STDOUT as an information for user
        """

        # Input
        raw = input(label + ": ")

        # Items are separated by "-" character
        date = raw.split("-")

        # It must contain 5 items
        if(len(date) == 5):
            try:
                dateTime = self.makeDateTime(date)
            except ValueError as err:
                print(err)
                return self.inputDate(label)
        else:
            print("Date must be in form:\n\tyear-month-day-hour-minute")
            return self.inputDate(label)
        return dateTime

    def makeDateTime(self,lst):
        """Creates datetime object for the date defined in lst parameter
        Parameters: lst (list)  list of 5 strings that represent date and time
        """
        
        # Temporary list that contains integer values of the items
        date = []

        # Tries to convert date and time into integers
        for item in lst:
            try:
                number = int(item)
                date.append(number)
            except ValueError:
                raise ValueError("One of the items is not a number")
                return None

        # datetime constructor
        return datetime(date[0],date[1],date[2],date[3],date[4])


    def setLastUsed(self,lst):
        """Setter for last used datetime
        """
        self.lastUsed = lst

    def getLastUsed(self):
        """Returns last used datetime
        """
        return self.lastUsed

    def setTotalTime(self,lst):
        """Setter for total time timedelta
        """
        self.totalTime = lst

    def getTotalTime(self):
        """Returns total time timedelta
        """
        return self.totalTime


    def setDirExistance(self,b,d):
        """Saves existance of a directory
        """
        self.dirExistance.append(b)
        if(b):
            self.dirArr.append(d)

    def iterateDir(self):
        """Returns list of directories to be iterated
        """
        return self.dirArr
            
    def setDataFiles(self,files):
        """Setter for files
        """
        self.dataFiles.append(files)

    def getDataFiles(self):
        """Returns files
        """
        return self.dataFiles

    def getDataDirs(self):
        """Returns directories
        """
        return self.dirArr


    def getID(self):
        """Returns app identifier
        """
        return self.ID

    def setName(self,name):
        """Setter for app name
        """
        self.name = name

    def getName(self):
        """Returns app name
        """
        return self.name

    def setActivityName(self,name):
        """Setter for activity name
        """
        self.activityName = name

    def getActivityName(self):
        """Returns activity name
        """
        return self.activityName

    def setLog(self,name,log):
        """Setter for log
        """
        self.log[name] = log

    def getLog(self,name):
        """Returns log
        """
        return self.log[name]

    def getAllLogs(self):
        return self.log

    def setUID(self,uid):
        """Setter for app uid
        """
        self.UID = uid

    def getUID(self):
        """Returns app uid
        """
        return self.UID

    def setPID(self,pid):
        """Setter for app pid
        """
        self.PID.append(pid)

    def getPID(self):
        """Returns app pid
        """
        return self.PID


    def getShortUID(self):
        """Returns app uid
        """
        if(len(self.UID) == 4):
            return self.UID
        elif(len(self.UID) > 0):
            return "u0a" + re.findall("10*(\d*)",self.UID)[0]
        else:
            return ""
            
    def getAppKeywords(self):
        """Returns all the keywords connected to the app
        """
        
        names = []

        # Application name
        if(self.name):
            names.append(self.name)
            names.append(self.name.replace(" ",""))
            names.append(self.name.lower())
            names.append(self.name.replace(" ","").lower())
            
        # Interesting parts of full identifier
        for name in self.ID.split(".")[1:]:
            if(not name in self.keywords):
                names.append(name)

        # Interesting parts of activity identifier
        for name in self.activityName.split(".")[1:]:
            if(not name in self.keywords):
                names.append(name)

        return names


    
    def printInfo(self):
        """Prints out app info
        """

        print("APP: " + self.getID())
        print("Name: " + self.getName())
        print("Last used: " + str(self.getLastUsed()))
        print("Total time: " + str(self.getTotalTime()))
        print("Used directories:")
        for d in self.getDataDirs():
            print(d)


if __name__ == "__main__":
    adb = ADB()
    adb.choose()
    device = Device(adb)
    device.getResources()
    #print(device.arch)
